package ru.yandex.direct.core.entity.client.model;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.base.Joiner;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.rbac.ClientPerm;
import ru.yandex.direct.rbac.RbacRole;

import static java.util.Objects.requireNonNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Клиент с опциями для добавления в базу.
 */
public final class ClientWithOptions {
    private String login;
    private Long uid;
    private ClientId clientId;
    private String email;
    private String name;
    private CurrencyCode currency;
    private Long countryRegionId;
    private String notificationEmail;
    private Language notificationLang;
    private RbacRole role;
    private boolean sendNews;
    private boolean sendAccNews;
    private boolean sendWarn;
    private boolean hideMarketRating;
    private boolean noTextAutocorrection;
    private boolean allowEditCampaigns;
    private boolean allowImportXls;
    private boolean allowTransferMoney;
    private boolean sharedAccountDisabled;
    private boolean usesQuasiCurrency;
    private boolean paymentBeforeModeration;
    private boolean disableApi;

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    public String getLogin() {
        return login;
    }

    public ClientWithOptions withLogin(String login) {
        this.login = login;
        return this;
    }

    public Long getUid() {
        return uid;
    }

    public ClientWithOptions withUid(Long uid) {
        this.uid = uid;
        return this;
    }

    public ClientId getClientId() {
        return clientId;
    }

    public ClientWithOptions withClientId(ClientId clientId) {
        this.clientId = clientId;
        return this;
    }

    public String getEmail() {
        return email;
    }

    public ClientWithOptions withEmail(String email) {
        this.email = email;
        return this;
    }

    public String getName() {
        return name;
    }

    public ClientWithOptions withName(String name) {
        this.name = name;
        return this;
    }

    public CurrencyCode getCurrency() {
        return currency;
    }

    public ClientWithOptions withCurrency(CurrencyCode currency) {
        this.currency = currency;
        return this;
    }

    @Nonnull
    public Long getCountryRegionId() {
        return nvl(countryRegionId, 0L);
    }

    public ClientWithOptions withCountryRegionId(@Nullable Long countryRegionId) {
        this.countryRegionId = countryRegionId;
        return this;
    }

    public String getNotificationEmail() {
        return notificationEmail;
    }

    public ClientWithOptions withNotificationEmail(String notificationEmail) {
        this.notificationEmail = notificationEmail;
        return this;
    }

    public Language getNotificationLang() {
        return notificationLang;
    }

    public ClientWithOptions withNotificationLang(Language notificationLang) {
        this.notificationLang = notificationLang;
        return this;
    }

    public boolean isSendNews() {
        return sendNews;
    }

    public ClientWithOptions withSendNews(boolean sendNews) {
        this.sendNews = sendNews;
        return this;
    }

    public boolean isSendAccNews() {
        return sendAccNews;
    }

    public ClientWithOptions withSendAccNews(boolean sendAccNews) {
        this.sendAccNews = sendAccNews;
        return this;
    }

    public boolean isSendWarn() {
        return sendWarn;
    }

    public ClientWithOptions withSendWarn(boolean sendWarn) {
        this.sendWarn = sendWarn;
        return this;
    }

    public boolean isHideMarketRating() {
        return hideMarketRating;
    }

    public ClientWithOptions withHideMarketRating(boolean hideMarketRating) {
        this.hideMarketRating = hideMarketRating;
        return this;
    }

    public boolean isNoTextAutocorrection() {
        return noTextAutocorrection;
    }

    public ClientWithOptions withNoTextAutocorrection(boolean noTextAutocorrection) {
        this.noTextAutocorrection = noTextAutocorrection;
        return this;
    }

    public boolean isAllowEditCampaigns() {
        return allowEditCampaigns;
    }

    public ClientWithOptions withAllowEditCampaigns(boolean allowMakeSimpleSubclient) {
        this.allowEditCampaigns = allowMakeSimpleSubclient;
        return this;
    }

    public boolean isAllowImportXls() {
        return allowImportXls;
    }

    public ClientWithOptions withAllowImportXls(boolean allowImportXls) {
        this.allowImportXls = allowImportXls;
        return this;
    }

    public boolean isAllowTransferMoney() {
        return allowTransferMoney;
    }

    public ClientWithOptions withAllowTransferMoney(boolean allowTransferMoney) {
        this.allowTransferMoney = allowTransferMoney;
        return this;
    }

    public ClientWithOptions withSharedAccountDisabled(boolean sharedAccountDisabled) {
        this.sharedAccountDisabled = sharedAccountDisabled;
        return this;
    }

    public boolean isSharedAccountDisabled() {
        return sharedAccountDisabled;
    }

    public ClientWithOptions withUsesQuasiCurrency(final boolean usesQuasiCurrency) {
        this.usesQuasiCurrency = usesQuasiCurrency;
        return this;
    }

    public boolean isUsesQuasiCurrency() {
        return usesQuasiCurrency;
    }

    public ClientWithOptions withPaymentBeforeModeration(boolean paymentBeforeModeration) {
        this.paymentBeforeModeration = paymentBeforeModeration;
        return this;
    }

    public boolean isPaymentBeforeModeration() {
        return paymentBeforeModeration;
    }

    public Set<ClientPerm> getPerms() {
        Set<ClientPerm> perms = new HashSet<>();
        if (allowEditCampaigns) {
            perms.add(ClientPerm.SUPER_SUBCLIENT);
        }
        if (allowImportXls) {
            perms.add(ClientPerm.XLS_IMPORT);
        }
        if (allowTransferMoney) {
            perms.add(ClientPerm.MONEY_TRANSFER);
        }
        return perms;
    }

    /**
     * @return string with enabled flags (clients_options.client_flags), separated by comma
     */
    public String getFlags() {
        List<String> flags = new ArrayList<>();
        if (isNoTextAutocorrection()) {
            flags.add(ClientFlags.NO_TEXT_AUTOCORRECTION.getTypedValue());
        }
        // TODO: 'no_display_hrefs','not_agreed_on_creatives_autogeneration',
        //         'can_copy_ctr','not_convert_to_currency','auto_video','suspend_video','feature_access_auto_video'
        if (isSharedAccountDisabled()) {
            flags.add(ClientFlags.ADD_WITHOUT_SHARED_ACCOUNT.getTypedValue());
        }
        if (isPaymentBeforeModeration()) {
            flags.add(ClientFlags.CAN_PAY_BEFORE_MODERATION.getTypedValue());
        }
        return Joiner.on(",").join(flags);
    }

    /**
     * Возвращает роль клиента. Не должно быть null.
     */
    @Nonnull
    public RbacRole getRole() {
        return requireNonNull(role);
    }

    @Nonnull
    public ClientWithOptions withRole(@Nonnull RbacRole role) {
        this.role = role;
        return this;
    }

    /**
     * Возврщает флаг того, что api включать не надо
     */
    public boolean doDisableApi() {
        return disableApi;
    }

    public ClientWithOptions withDisableApi(boolean disableApi) {
        this.disableApi = disableApi;
        return this;
    }
}
