package ru.yandex.direct.core.entity.client.repository;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperWithSupplier;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.client.model.AgencyClientProve;
import ru.yandex.direct.dbschema.ppc.tables.records.AgencyClientProveRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.common.jooqmapperex.FieldMapperFactoryEx.timestampField;
import static ru.yandex.direct.dbschema.ppc.tables.AgencyClientProve.AGENCY_CLIENT_PROVE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Работа с запросами на подверждение принадлежности клиента (логина) - агентству
 */

@ParametersAreNonnullByDefault
@Repository
public class AgencyClientProveRepository {

    private final DslContextProvider dslContextProvider;
    private final OldJooqMapperWithSupplier<AgencyClientProve> agencyClientProveMapper;

    @Autowired
    public AgencyClientProveRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        this.agencyClientProveMapper = new OldJooqMapperBuilder<>(AgencyClientProve::new)
                .map(field(AGENCY_CLIENT_PROVE.AGENCY_CLIENT_ID, AgencyClientProve.AGENCY_CLIENT_ID))
                .map(field(AGENCY_CLIENT_PROVE.AGENCY_UID, AgencyClientProve.AGENCY_UID))
                .map(field(AGENCY_CLIENT_PROVE.CLIENT_UID, AgencyClientProve.CLIENT_UID))
                .map(convertibleField(AGENCY_CLIENT_PROVE.IS_CONFIRMED, AgencyClientProve.IS_CONFIRMED)
                        .convertFromDbBy(RepositoryUtils::booleanFromLong)
                        .convertToDbBy(RepositoryUtils::booleanToLong)
                        .withDatabaseDefault())
                .map(timestampField(AGENCY_CLIENT_PROVE.REQUEST_TIME, AgencyClientProve.REQUEST_TIME)
                        .withDatabaseDefault())
                .build();
    }

    /**
     * Получить список запросов, добавленных до {@code borderDateTime}
     *
     * @param shard          шард
     * @param borderDateTime {@link LocalDateTime} граничное время
     * @return {@link List<AgencyClientProve>} список запросов на подтверждение
     */
    @QueryWithoutIndex("Выборка по дате, только для jobs")
    public List<AgencyClientProve> getOutdatedRequests(int shard, LocalDateTime borderDateTime) {
        List<Record> result = dslContextProvider.ppc(shard)
                .select(agencyClientProveMapper.getFieldsToRead())
                .from(AGENCY_CLIENT_PROVE)
                .where(AGENCY_CLIENT_PROVE.REQUEST_TIME.lt(borderDateTime))
                .fetch();
        return mapList(result, agencyClientProveMapper::fromDb);
    }

    /**
     * Получить список всех запросов
     *
     * @param shard шард
     * @return {@link List<AgencyClientProve>} список запросов на подтверждение
     */
    @QueryWithoutIndex("Возвращает все значения")
    public List<AgencyClientProve> getAllRequests(int shard) {
        List<Record> result = dslContextProvider.ppc(shard)
                .select(agencyClientProveMapper.getFieldsToRead())
                .from(AGENCY_CLIENT_PROVE)
                .fetch();
        return mapList(result, agencyClientProveMapper::fromDb);
    }

    /**
     * Вставить в таблицу новый запрос
     *
     * @param shard   шард
     * @param request {@link AgencyClientProve} запрос
     * @return количество вставленных записей
     */
    public int addNewRequest(int shard, AgencyClientProve request) {
        if (request == null) {
            return 0;
        }
        InsertHelper<AgencyClientProveRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), AGENCY_CLIENT_PROVE);
        insertHelper.add(agencyClientProveMapper, request).newRecord();
        int rowsInserted = insertHelper.execute();
        return rowsInserted;
    }

    /**
     * Удаляет конкретный запрос на подтверждение
     * Запрос идертифицируется по первичному ключу - ClientID агентства и uid клиента
     * На подтвержденность запроса не смотрим, поскольку:
     * - если is_confirmed=0, то нужно удалить, чтоб агенство могло повторно отправить заявку (и письмо) клиенту
     * - если is_confirmed=1, то нужно удалить потому что "Если агенство не создало кампании клиенту, значит что-то у них пошло не так и лучше потом еще раз переспросить"
     *
     * @param shard        шард
     * @param proveRequest {@link AgencyClientProve} запрос для удаления
     * @return число удаленных записей
     */
    public int deleteRequest(int shard, AgencyClientProve proveRequest) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(AGENCY_CLIENT_PROVE)
                .where(AGENCY_CLIENT_PROVE.AGENCY_CLIENT_ID.eq(proveRequest.getAgencyClientId()))
                .and(AGENCY_CLIENT_PROVE.CLIENT_UID.eq(proveRequest.getClientUid()))
                .execute();
    }

}
