package ru.yandex.direct.core.entity.client.repository;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.jooq.InsertValuesStep3;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.client.model.AgencyClientRelation;
import ru.yandex.direct.dbschema.ppc.enums.AgencyClientRelationsBind;
import ru.yandex.direct.dbschema.ppc.enums.AgencyClientRelationsClientArchived;
import ru.yandex.direct.dbschema.ppc.tables.records.AgencyClientRelationsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.Tables.AGENCY_CLIENT_RELATIONS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Доступ к дополнительной информации о связях клиентов и агентств, хранящихся в agency_client_relations
 */
@Repository
public class AgencyClientRelationRepository {
    private final DslContextProvider dslContextProvider;

    @Autowired
    public AgencyClientRelationRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить список {@link AgencyClientRelation} по списку клиентов
     *
     * @param shard
     * @param clientClientIds
     */
    public List<AgencyClientRelation> getByClients(int shard, List<ClientId> clientClientIds) {
        return dslContextProvider.ppc(shard)
                .select(
                        AGENCY_CLIENT_RELATIONS.AGENCY_CLIENT_ID,
                        AGENCY_CLIENT_RELATIONS.BIND,
                        AGENCY_CLIENT_RELATIONS.CLIENT_ARCHIVED,
                        AGENCY_CLIENT_RELATIONS.CLIENT_CLIENT_ID,
                        AGENCY_CLIENT_RELATIONS.CLIENT_DESCRIPTION)
                .from(AGENCY_CLIENT_RELATIONS)
                .where(AGENCY_CLIENT_RELATIONS.CLIENT_CLIENT_ID.in(mapList(clientClientIds, ClientId::asLong)))
                .fetch()
                .into(AGENCY_CLIENT_RELATIONS)
                .stream()
                .map(this::toModel)
                .collect(Collectors.toList());
    }

    /**
     * Получить список {@link AgencyClientRelation} id агентства
     *
     * @param shard    номер шарда
     * @param agencyId clientId агентства
     */
    public List<AgencyClientRelation> getByAgencyId(int shard, ClientId agencyId) {
        return dslContextProvider.ppc(shard)
                .select(
                        AGENCY_CLIENT_RELATIONS.AGENCY_CLIENT_ID,
                        AGENCY_CLIENT_RELATIONS.BIND,
                        AGENCY_CLIENT_RELATIONS.CLIENT_ARCHIVED,
                        AGENCY_CLIENT_RELATIONS.CLIENT_CLIENT_ID,
                        AGENCY_CLIENT_RELATIONS.CLIENT_DESCRIPTION)
                .from(AGENCY_CLIENT_RELATIONS)
                .where(AGENCY_CLIENT_RELATIONS.AGENCY_CLIENT_ID.eq(agencyId.asLong()))
                .fetch()
                .into(AGENCY_CLIENT_RELATIONS)
                .stream()
                .map(this::toModel)
                .collect(Collectors.toList());
    }

    /**
     * Получить id неархивных субклиентов агенства из переданого списка id субклиентов
     *
     * @param agencyChiefClientId главный представитель агенства
     * @param clientClientIds     коллекция id проверяемых субклиентов
     * @return множество id неархивных субклиентов
     */
    public Set<Long> getUnarchivedAgencyClientsIds(int shard, Long agencyChiefClientId,
                                                   Collection<Long> clientClientIds) {
        return dslContextProvider.ppc(shard)
                .select(AGENCY_CLIENT_RELATIONS.CLIENT_CLIENT_ID)
                .from(AGENCY_CLIENT_RELATIONS)
                .where(AGENCY_CLIENT_RELATIONS.CLIENT_CLIENT_ID.in(clientClientIds))
                .and(AGENCY_CLIENT_RELATIONS.AGENCY_CLIENT_ID.eq(agencyChiefClientId))
                .and(AGENCY_CLIENT_RELATIONS.CLIENT_ARCHIVED.eq(AgencyClientRelationsClientArchived.No))
                .fetchSet(AGENCY_CLIENT_RELATIONS.CLIENT_CLIENT_ID);
    }

    private AgencyClientRelation toModel(AgencyClientRelationsRecord rec) {
        return new AgencyClientRelation()
                .withAgencyClientId(ClientId.fromLong(rec.getAgencyClientId()))
                .withClientClientId(ClientId.fromLong(rec.getClientClientId()))
                .withArchived(rec.getClientArchived() == AgencyClientRelationsClientArchived.Yes)
                .withBinded(rec.getBind() == AgencyClientRelationsBind.Yes)
                .withDescription(rec.getClientDescription());
    }

    /**
     * Привязать к агенству заданных клиентов
     *
     * @param shard     Номер shard-а
     * @param agencyId  Id агенства
     * @param clientIds Список Id-ков клиентов
     */
    public void bindClients(
            Integer shard, ClientId agencyId, Collection<ClientId> clientIds) {
        if (clientIds.isEmpty()) {
            return;
        }
        InsertValuesStep3<AgencyClientRelationsRecord, Long, Long, AgencyClientRelationsBind> batch =
                dslContextProvider.ppc(shard)
                        .insertInto(AGENCY_CLIENT_RELATIONS,
                                AGENCY_CLIENT_RELATIONS.AGENCY_CLIENT_ID,
                                AGENCY_CLIENT_RELATIONS.CLIENT_CLIENT_ID,
                                AGENCY_CLIENT_RELATIONS.BIND);

        for (ClientId clientId : clientIds) {
            batch.values(agencyId.asLong(), clientId.asLong(), AgencyClientRelationsBind.Yes);
        }

        batch.onDuplicateKeyUpdate()
                .set(AGENCY_CLIENT_RELATIONS.BIND, AgencyClientRelationsBind.Yes)
                .execute();
    }

    /**
     * Отвязать клиента от агенства
     *
     * @param shard    Номер shard-а
     * @param clientId Id клиента
     * @param agencyId Id агенства
     */
    public void unbindClient(Integer shard, ClientId clientId, ClientId agencyId) {
        dslContextProvider.ppc(shard)
                .update(AGENCY_CLIENT_RELATIONS)
                .set(AGENCY_CLIENT_RELATIONS.BIND, AgencyClientRelationsBind.No)
                .where(AGENCY_CLIENT_RELATIONS.AGENCY_CLIENT_ID.eq(agencyId.asLong())
                        .and(AGENCY_CLIENT_RELATIONS.CLIENT_CLIENT_ID.eq(clientId.asLong())))
                .execute();
    }

    /**
     * Архивировать заданных клиентов агенства
     *
     * @param shard     Номер shard-а
     * @param agencyId  Id агенства
     * @param clientIds Список Id-ков клиентов
     */
    public void archiveClients(int shard, ClientId agencyId, Collection<ClientId> clientIds) {
        setClientsArchived(shard, agencyId, clientIds, AgencyClientRelationsClientArchived.Yes);
    }

    /**
     * Разархивировать заданных клиентов агенства
     *
     * @param shard     Номер shard-а
     * @param agencyId  Id агенства
     * @param clientIds Список Id-ков клиентов
     */
    public void unarchiveClients(int shard, ClientId agencyId, Collection<ClientId> clientIds) {
        setClientsArchived(shard, agencyId, clientIds, AgencyClientRelationsClientArchived.No);
    }

    /**
     * Проставить статус архивности для заданных клиентов агенства
     *
     * @param shard          Номер shard-а
     * @param agencyId       Id агенства
     * @param clientIds      Список Id-ков клиентов
     * @param clientArchived Заархивирован ли клиент
     */
    private void setClientsArchived(int shard, ClientId agencyId, Collection<ClientId> clientIds,
                                    AgencyClientRelationsClientArchived clientArchived) {
        if (clientIds.isEmpty()) {
            return;
        }
        InsertValuesStep3<AgencyClientRelationsRecord, Long, Long, AgencyClientRelationsClientArchived> batch =
                dslContextProvider.ppc(shard)
                        .insertInto(AGENCY_CLIENT_RELATIONS,
                                AGENCY_CLIENT_RELATIONS.AGENCY_CLIENT_ID,
                                AGENCY_CLIENT_RELATIONS.CLIENT_CLIENT_ID,
                                AGENCY_CLIENT_RELATIONS.CLIENT_ARCHIVED);

        for (ClientId clientId : clientIds) {
            batch.values(agencyId.asLong(), clientId.asLong(), clientArchived);
        }

        batch.onDuplicateKeyUpdate()
                .set(AGENCY_CLIENT_RELATIONS.CLIENT_ARCHIVED, clientArchived)
                .execute();
    }
}
