package ru.yandex.direct.core.entity.client.repository;

import java.time.LocalDate;
import java.util.Collection;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.client.model.AgencyNds;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static ru.yandex.direct.dbschema.ppc.Tables.AGENCY_NDS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class AgencyNdsRepository {
    public static final JooqMapperWithSupplier<AgencyNds> AGENCY_NDS_MAPPER =
            JooqMapperWithSupplierBuilder.builder(AgencyNds::new)
                    .map(property(AgencyNds.CLIENT_ID, AGENCY_NDS.CLIENT_ID))
                    .map(property(AgencyNds.DATE_FROM, AGENCY_NDS.DATE_FROM))
                    .map(property(AgencyNds.DATE_TO, AGENCY_NDS.DATE_TO))
                    .map(convertibleProperty(AgencyNds.NDS, AGENCY_NDS.NDS,
                            RepositoryUtils::percentFromBigInteger,
                            RepositoryUtils::percentToBigInteger))
                    .build();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public AgencyNdsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public Collection<AgencyNds> fetchByClientIds(int shard, Collection<ClientId> clientIds, LocalDate onDate) {
        return dslContextProvider.ppc(shard)
                .select(AGENCY_NDS_MAPPER.getFieldsToRead())
                .from(AGENCY_NDS)
                .where(AGENCY_NDS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .and(AGENCY_NDS.DATE_FROM.le(onDate))
                .and(AGENCY_NDS.DATE_TO.ge(onDate))
                .fetch(AGENCY_NDS_MAPPER::fromDb);
    }

    public Collection<AgencyNds> fetchHistoryByClientId(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .select(AGENCY_NDS_MAPPER.getFieldsToRead())
                .from(AGENCY_NDS)
                .where(AGENCY_NDS.CLIENT_ID.eq(clientId.asLong()))
                .fetch(AGENCY_NDS_MAPPER::fromDb);
    }
}
