package ru.yandex.direct.core.entity.client.repository;

import java.util.Collection;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppc.tables.records.ApiSpecialUserOptionsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.dbschema.ppc.Tables.API_SPECIAL_USER_OPTIONS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Repository
public class ApiSpecialUserOptionsRepository {
    private final DslContextProvider dslContextProvider;

    public ApiSpecialUserOptionsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * @return соответствие clientId -> optionName -> value; во внешнем и внутренних контейнерах
     * какие-то ключи могут отсутствовать, если в базе нет таких данных
     */
    public Map<ClientId, Map<String, Long>> getSpecialUserOptions(int shard,
                                                                  Collection<ClientId> clientIds, Collection<String> optionNames) {
        return dslContextProvider.ppc(shard)
                .select(API_SPECIAL_USER_OPTIONS.CLIENT_ID,
                        API_SPECIAL_USER_OPTIONS.KEYNAME,
                        API_SPECIAL_USER_OPTIONS.VALUE)
                .from(API_SPECIAL_USER_OPTIONS)
                .where(API_SPECIAL_USER_OPTIONS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .and(API_SPECIAL_USER_OPTIONS.KEYNAME.in(optionNames))
                .fetchInto(API_SPECIAL_USER_OPTIONS)
                .stream()
                .collect(groupingBy(r -> ClientId.fromLong(r.getClientid()),
                        toMap(ApiSpecialUserOptionsRecord::getKeyname, ApiSpecialUserOptionsRecord::getValue)));
    }

    /**
     * @return соответствие clientId -> value; если для каких-то клиентов в базе ничего нет,
     * в результирующем контейнере соответствующих ключей не будет
     */
    public Map<ClientId, Long> getSingleSpecialUserOptionForMultipleClients(int shard,
                                                                            Collection<ClientId> clientIds, String optionName) {
        return getSpecialUserOptions(shard, clientIds, singletonList(optionName)).entrySet().stream()
                .collect(toMap(Map.Entry::getKey, entry -> entry.getValue().get(optionName)));
    }
}
