package ru.yandex.direct.core.entity.client.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.InsertValuesStep3;
import org.jooq.Record3;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.client.model.ClientBrand;
import ru.yandex.direct.dbschema.ppc.tables.records.ClientBrandsRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENT_BRANDS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class ClientBrandsRepository {
    private static final int DELETE_CHUNK_SIZE = 1_000;

    private final DslContextProvider dslContextProvider;

    @Autowired
    public ClientBrandsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Вставить новые записи в ppc.client_brands. Пошардовая версия.
     *
     * @return количество вставленных в таблицу строк
     */
    public int replaceClientBrands(int shard, Collection<ClientBrand> clientBrandList) {
        InsertValuesStep3<ClientBrandsRecord, Long, Long, LocalDateTime> step = dslContextProvider.ppc(shard)
                .insertInto(CLIENT_BRANDS, CLIENT_BRANDS.CLIENT_ID, CLIENT_BRANDS.BRAND_CLIENT_ID,
                        CLIENT_BRANDS.LAST_SYNC_TIME);
        for (ClientBrand clientBrand : clientBrandList) {
            step = step.values(clientBrand.getClientId(), clientBrand.getBrandClientId(),
                    clientBrand.getLastSync());
        }
        return step.onDuplicateKeyUpdate()
                .set(CLIENT_BRANDS.LAST_SYNC_TIME, MySQLDSL.values(CLIENT_BRANDS.LAST_SYNC_TIME))
                .set(CLIENT_BRANDS.BRAND_CLIENT_ID, MySQLDSL.values(CLIENT_BRANDS.BRAND_CLIENT_ID))
                .execute();
    }

    /**
     * Получить общее количество записей в таблице ppc.client_brands в заданном шарде
     */
    @QueryWithoutIndex("Подсчёт числа строк, используется только в jobs")
    public int getClientBrandsCount(int shard) {
        return dslContextProvider.ppc(shard).selectCount().from(CLIENT_BRANDS).fetchOne(0, int.class);
    }

    /**
     * Удалить из таблицы ppc.client_brands все записи в шарде, принадлежащие клиентам с заданным ClientId
     *
     * @return количество удаленных из таблицы строк
     */
    public int deleteEntriesByClientId(int shard, Collection<ClientId> clientIds) {
        return dslContextProvider.ppc(shard).deleteFrom(CLIENT_BRANDS)
                .where(CLIENT_BRANDS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .execute();
    }


    /**
     * Удалить из таблицы ppc.client_brands все записи в шарде, синхронизированные раньше заданного времени
     * <p>
     * Удаление делается чанками по {@value DELETE_CHUNK_SIZE}
     *
     * @return количество удаленных из таблицы строк
     */
    public int deleteEntriesOlderThanDateTime(int shard, LocalDateTime borderDateTime) {
        List<ClientBrand> clientBrandsList = getEntriesOlderThanDateTime(shard, borderDateTime);

        return StreamEx.ofSubLists(clientBrandsList, DELETE_CHUNK_SIZE)
                .mapToInt(chunk -> deleteEntriesByFullDescription(shard, chunk))
                .sum();
    }

    /**
     * Удалить из таблицы ppc.client_brands все записи в шарде, принадлежащие клиентам с заданным ClientId
     *
     * @return количество удаленных из таблицы строк
     */
    public int deleteEntriesByFullDescription(int shard, Collection<ClientBrand> clientBrands) {
        return dslContextProvider.ppc(shard).deleteFrom(CLIENT_BRANDS)
                .where(row(CLIENT_BRANDS.CLIENT_ID, CLIENT_BRANDS.BRAND_CLIENT_ID, CLIENT_BRANDS.LAST_SYNC_TIME)
                        .in(mapList(clientBrands,
                                cb -> row(cb.getClientId(), cb.getBrandClientId(),
                                        cb.getLastSync()))))
                .execute();
    }

    /**
     * Получить из таблицы ppc.client_brands в шарде все записи, синхронизированные раньше заданного времени
     */
    @QueryWithoutIndex("Выборка по дате, только для jobs")
    public List<ClientBrand> getEntriesOlderThanDateTime(int shard, LocalDateTime borderDateTime) {
        return dslContextProvider.ppc(shard)
                .select(CLIENT_BRANDS.CLIENT_ID, CLIENT_BRANDS.BRAND_CLIENT_ID, CLIENT_BRANDS.LAST_SYNC_TIME)
                .from(CLIENT_BRANDS)
                .where(CLIENT_BRANDS.LAST_SYNC_TIME.lessThan(borderDateTime))
                .fetch(this::extractClientBrand);
    }

    private ClientBrand extractClientBrand(Record3<Long, Long, LocalDateTime> e) {
        return new ClientBrand()
                .withClientId(e.getValue(CLIENT_BRANDS.CLIENT_ID))
                .withBrandClientId(e.getValue(CLIENT_BRANDS.BRAND_CLIENT_ID))
                .withLastSync(e.getValue(CLIENT_BRANDS.LAST_SYNC_TIME));
    }

}
