package ru.yandex.direct.core.entity.client.repository;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppc.enums.ClientsWorkCurrency;
import ru.yandex.direct.dbschema.ppc.enums.UsersHidden;
import ru.yandex.direct.dbschema.ppc.enums.UsersOptionsStatuseasy;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static org.jooq.impl.DSL.field;
import static org.jooq.impl.DSL.not;
import static org.jooq.impl.DSL.or;
import static org.jooq.impl.DSL.sum;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_TO_FORCE_MULTICURRENCY_TEASER;
import static ru.yandex.direct.dbschema.ppc.Tables.CURRENCY_CONVERT_QUEUE;
import static ru.yandex.direct.dbschema.ppc.Tables.FORCE_CURRENCY_CONVERT;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS_OPTIONS;
import static ru.yandex.direct.dbutil.SqlUtils.setField;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Репозиторий для проверки, затронуты ли клиенты переводом в валюту.
 * <p>
 * Это бэкэнд к {@link ru.yandex.direct.core.entity.client.service.ClientCurrencyConversionTeaserService}
 * с массовыми методами. Какое состояние клиентов проверяется в getClientsThatHaveToConvert, см.
 * документацию там.
 */
@ParametersAreNonnullByDefault
@Repository
public class ClientCurrencyConversionTeaserRepository {
    private final DslContextProvider ppcDslContextProvider;

    @Autowired
    public ClientCurrencyConversionTeaserRepository(DslContextProvider ppcDslContextProvider) {
        this.ppcDslContextProvider = ppcDslContextProvider;
    }

    public Set<ClientId> getClientsThatHaveToConvert(int shard, Collection<ClientId> clientIds) {
        List<Long> clientsThatHaveToConvert = ppcDslContextProvider.ppc(shard)
                .select(CLIENTS_TO_FORCE_MULTICURRENCY_TEASER.CLIENT_ID, field("1"))
                .from(CLIENTS_TO_FORCE_MULTICURRENCY_TEASER)
                .leftJoin(FORCE_CURRENCY_CONVERT)
                .on(FORCE_CURRENCY_CONVERT.CLIENT_ID.eq(CLIENTS_TO_FORCE_MULTICURRENCY_TEASER.CLIENT_ID))
                .leftJoin(CLIENTS)
                .on(CLIENTS.CLIENT_ID.eq(CLIENTS_TO_FORCE_MULTICURRENCY_TEASER.CLIENT_ID))
                .leftJoin(CLIENTS_OPTIONS)
                .on(CLIENTS_OPTIONS.CLIENT_ID.eq(CLIENTS_TO_FORCE_MULTICURRENCY_TEASER.CLIENT_ID))
                .leftJoin(CURRENCY_CONVERT_QUEUE)
                .on(CURRENCY_CONVERT_QUEUE.CLIENT_ID.eq(CLIENTS_TO_FORCE_MULTICURRENCY_TEASER.CLIENT_ID))
                .innerJoin(USERS)
                .on(USERS.CLIENT_ID.eq(CLIENTS_TO_FORCE_MULTICURRENCY_TEASER.CLIENT_ID))
                .leftJoin(USERS_OPTIONS)
                .on(USERS_OPTIONS.UID.eq(USERS.UID))
                .where(or(CLIENTS.WORK_CURRENCY.isNull(), CLIENTS.WORK_CURRENCY.eq(ClientsWorkCurrency.YND_FIXED)))
                // не принял оферту:
                .and(FORCE_CURRENCY_CONVERT.CLIENT_ID.isNull())
                // не установлен флаг не конвертироваться:
                .and(or(CLIENTS_OPTIONS.CLIENT_FLAGS.isNull(),
                        not(setField(CLIENTS_OPTIONS.CLIENT_FLAGS).contains("not_convert_to_currency"))))
                // конвертация ещё не заказана:
                .and(CURRENCY_CONVERT_QUEUE.CLIENT_ID.isNull())
                .and(or(USERS_OPTIONS.STATUS_EASY.isNull(), USERS_OPTIONS.STATUS_EASY.eq(UsersOptionsStatuseasy.No)))
                .and(CLIENTS_TO_FORCE_MULTICURRENCY_TEASER.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .groupBy(CLIENTS_TO_FORCE_MULTICURRENCY_TEASER.CLIENT_ID)
                .having(sum(field("IF({0} = {1}, 1, 0)", Integer.class, USERS.HIDDEN, UsersHidden.Yes))
                        .eq(BigDecimal.ZERO))
                .orderBy(field("NULL"))
                .fetch(CLIENTS_TO_FORCE_MULTICURRENCY_TEASER.CLIENT_ID);

        return clientsThatHaveToConvert.stream()
                .map(ClientId::fromLong)
                .collect(Collectors.toSet());
    }
}
