package ru.yandex.direct.core.entity.client.repository;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppc.tables.records.ClientManagersRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENT_MANAGERS;

@Repository
public class ClientManagersRepository {
    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    @Autowired
    public ClientManagersRepository(DslContextProvider dslContextProvider,
                                    ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    public List<Long> getClientsOfManagerWithoutCampaigns(Long managerUid) {
        return shardHelper.dbShards().stream().map(shard ->
                dslContextProvider.ppc(shard)
                        .select(CLIENT_MANAGERS.CLIENT_ID)
                        .from(CLIENT_MANAGERS)
                        .leftJoin(CAMPAIGNS)
                        .on(CAMPAIGNS.CLIENT_ID.eq(CLIENT_MANAGERS.CLIENT_ID)
                                .and(CAMPAIGNS.MANAGER_UID.eq(CLIENT_MANAGERS.MANAGER_UID)))
                        .where(CLIENT_MANAGERS.MANAGER_UID.eq(managerUid))
                        .and(CAMPAIGNS.CID.isNull())
                        .fetch(CLIENT_MANAGERS.CLIENT_ID))
                .flatMap(Collection::stream).collect(Collectors.toList());
    }

    public void bindClientsToManager(int shard, Long managerUid, List<ClientId> clientIds) {
        InsertHelper<ClientManagersRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), CLIENT_MANAGERS);

        for (ClientId clientId : clientIds) {
            insertHelper
                    .set(CLIENT_MANAGERS.MANAGER_UID, managerUid)
                    .set(CLIENT_MANAGERS.CLIENT_ID, clientId.asLong())
                    .newRecord();
        }

        insertHelper
                .onDuplicateKeyIgnore()
                .execute();
    }

    public void deleteFromClientManagers(int shard, Collection<Long> clientIds, Long managerUid) {
        dslContextProvider.ppc(shard)
                .deleteFrom(CLIENT_MANAGERS)
                .where(CLIENT_MANAGERS.CLIENT_ID.in(clientIds))
                .and(CLIENT_MANAGERS.MANAGER_UID.eq(managerUid))
                .execute();
    }

    public void deleteFromClientManagers(int shard, Long clientId, Collection<Long> managerUids) {
        dslContextProvider.ppc(shard)
                .deleteFrom(CLIENT_MANAGERS)
                .where(CLIENT_MANAGERS.CLIENT_ID.eq(clientId))
                .and(CLIENT_MANAGERS.MANAGER_UID.in(managerUids))
                .execute();
    }
}
