package ru.yandex.direct.core.entity.client.repository;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.client.model.ClientMeasurerSettings;
import ru.yandex.direct.core.entity.client.model.ClientMeasurerSystem;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.core.entity.client.model.ClientMeasurerSystem.toSource;
import static ru.yandex.direct.core.entity.client.repository.mapper.ClientMeasurerSettingsMapperProvider.getClientMeasurerSettingsMapper;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENT_MEASURERS_SETTINGS;

@Repository
@ParametersAreNonnullByDefault
public class ClientMeasurerSettingsRepository {

    private final JooqMapperWithSupplier<ClientMeasurerSettings> clientMeasurerSettingsMapper;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public ClientMeasurerSettingsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.clientMeasurerSettingsMapper = getClientMeasurerSettingsMapper();
    }

    public void insertOrUpdate(int shard, List<ClientMeasurerSettings> clientMeasurerSettingsList) {
        if (clientMeasurerSettingsList.isEmpty()) {
            return;
        }

        var dslContext = dslContextProvider.ppc(shard);
        var insertHelper = new InsertHelper<>(dslContext, CLIENT_MEASURERS_SETTINGS)
                .addAll(clientMeasurerSettingsMapper, clientMeasurerSettingsList);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(CLIENT_MEASURERS_SETTINGS.SETTINGS, MySQLDSL.values(CLIENT_MEASURERS_SETTINGS.SETTINGS));
        }

        insertHelper.executeIfRecordsAdded();
    }

    public void update(int shard, ClientMeasurerSettings clientMeasurerSettings) {
        var dslContext = dslContextProvider.ppc(shard);
        dslContext.update(CLIENT_MEASURERS_SETTINGS)
                .set(CLIENT_MEASURERS_SETTINGS.SETTINGS, clientMeasurerSettings.getSettings())
                .where(CLIENT_MEASURERS_SETTINGS.CLIENT_ID.eq(clientMeasurerSettings.getClientId())
                        .and(CLIENT_MEASURERS_SETTINGS.MEASURER_SYSTEM.eq(toSource(clientMeasurerSettings.getClientMeasurerSystem())))
                        .and(CLIENT_MEASURERS_SETTINGS.LAST_CHANGE.eq(clientMeasurerSettings.getLastChange())))
                .execute();
    }

    public List<ClientMeasurerSettings> getByClientId(int shard, Long clientId) {
        return dslContextProvider.ppc(shard)
                .select(CLIENT_MEASURERS_SETTINGS.CLIENT_ID,
                        CLIENT_MEASURERS_SETTINGS.MEASURER_SYSTEM,
                        CLIENT_MEASURERS_SETTINGS.SETTINGS,
                        CLIENT_MEASURERS_SETTINGS.LAST_CHANGE)
                .from(CLIENT_MEASURERS_SETTINGS)
                .where(CLIENT_MEASURERS_SETTINGS.CLIENT_ID.eq(clientId))
                .fetch()
                .map(clientMeasurerSettingsMapper::fromDb);
    }

    public Map<Long, ClientMeasurerSettings> getByClientIdsAndSystem(
            int shard, List<Long> clientIds, ClientMeasurerSystem measurerSystem
    ) {
        return dslContextProvider.ppc(shard)
                .select(CLIENT_MEASURERS_SETTINGS.CLIENT_ID,
                        CLIENT_MEASURERS_SETTINGS.MEASURER_SYSTEM,
                        CLIENT_MEASURERS_SETTINGS.SETTINGS,
                        CLIENT_MEASURERS_SETTINGS.LAST_CHANGE)
                .from(CLIENT_MEASURERS_SETTINGS)
                .where(CLIENT_MEASURERS_SETTINGS.CLIENT_ID.in(clientIds)
                        .and(CLIENT_MEASURERS_SETTINGS.MEASURER_SYSTEM.eq(toSource(measurerSystem))))
                .fetchMap(CLIENT_MEASURERS_SETTINGS.CLIENT_ID, clientMeasurerSettingsMapper::fromDb);
    }

    @QueryWithoutIndex("Индекс будет нерепрезентативен, типов измерителей всего 9, интегрируемся пока с одним")
    public List<ClientMeasurerSettings> getByMeasurerSystem(int shard, ClientMeasurerSystem clientMeasurerSystem) {
        return dslContextProvider.ppc(shard)
                .select(CLIENT_MEASURERS_SETTINGS.CLIENT_ID,
                        CLIENT_MEASURERS_SETTINGS.MEASURER_SYSTEM,
                        CLIENT_MEASURERS_SETTINGS.SETTINGS,
                        CLIENT_MEASURERS_SETTINGS.LAST_CHANGE)
                .from(CLIENT_MEASURERS_SETTINGS)
                .where(CLIENT_MEASURERS_SETTINGS.MEASURER_SYSTEM.eq(toSource(clientMeasurerSystem)))
                .fetch()
                .map(clientMeasurerSettingsMapper::fromDb);
    }

    public void deleteByClientIdAndSystem(
            int shard,
            Long clientId,
            ClientMeasurerSystem clientMeasurerSystem) {
        var dslContext = dslContextProvider.ppc(shard);

        dslContext.deleteFrom(CLIENT_MEASURERS_SETTINGS)
                .where(CLIENT_MEASURERS_SETTINGS.CLIENT_ID.eq(clientId)
                        .and(CLIENT_MEASURERS_SETTINGS.MEASURER_SYSTEM.eq(toSource(clientMeasurerSystem))))
                .execute();
    }
}
