package ru.yandex.direct.core.entity.client.repository;

import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperWithSupplier;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.stream.Collectors.groupingBy;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.common.jooqmapperex.FieldMapperFactoryEx.dateField;
import static ru.yandex.direct.common.jooqmapperex.FieldMapperFactoryEx.percentField;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENT_NDS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class ClientNdsRepository {

    public static final OldJooqMapperWithSupplier<ClientNds> CLIENT_NDS_MAPPER =
            new OldJooqMapperBuilder<>(ClientNds::new)
                    .map(field(CLIENT_NDS.CLIENT_ID, ClientNds.CLIENT_ID))
                    .map(dateField(CLIENT_NDS.DATE_FROM, ClientNds.DATE_FROM))
                    .map(dateField(CLIENT_NDS.DATE_TO, ClientNds.DATE_TO))
                    .map(percentField(CLIENT_NDS.NDS, ClientNds.NDS))
                    .build();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public ClientNdsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public Collection<ClientNds> fetchByClientIds(int shard, Collection<ClientId> clientIds, LocalDate onDate) {
        return dslContextProvider.ppc(shard)
                .select(CLIENT_NDS_MAPPER.getFieldsToRead())
                .from(CLIENT_NDS)
                .where(CLIENT_NDS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .and(CLIENT_NDS.DATE_FROM.le(onDate))
                .and(CLIENT_NDS.DATE_TO.ge(onDate))
                .fetch(CLIENT_NDS_MAPPER::fromDb);
    }

    public Collection<ClientNds> fetchHistoryByClientId(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .select(CLIENT_NDS_MAPPER.getFieldsToRead())
                .from(CLIENT_NDS)
                .where(CLIENT_NDS.CLIENT_ID.eq(clientId.asLong()))
                .fetch(CLIENT_NDS_MAPPER::fromDb);
    }

    public Map<Long, List<ClientNds>> fetchHistoryByClientIds(int shard, List<Long> clientIds) {
        return dslContextProvider.ppc(shard)
                .select(CLIENT_NDS_MAPPER.getFieldsToRead())
                .from(CLIENT_NDS)
                .where(CLIENT_NDS.CLIENT_ID.in(clientIds))
                .fetch(CLIENT_NDS_MAPPER::fromDb)
                .stream()
                .collect(groupingBy(ClientNds::getClientId));
    }
}
