package ru.yandex.direct.core.entity.client.repository;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientFlags;
import ru.yandex.direct.model.ModelProperty;

import static java.util.Collections.emptySet;

@ParametersAreNonnullByDefault
public class ClientOptionsMapping {
    public static final ImmutableMap<ClientFlags, ModelProperty<? super Client, Boolean>> TYPED_CLIENTS_OPTIONS_FLAGS_ENTRIES =
            ImmutableMap.<ClientFlags, ModelProperty<? super Client, Boolean>>builder()
                    .put(ClientFlags.NO_TEXT_AUTOCORRECTION, Client.NO_TEXT_AUTOCORRECTION)
                    .put(ClientFlags.NO_DISPLAY_HREF, Client.NO_DISPLAY_HREF)
                    .put(ClientFlags.NOT_AGREED_ON_CREATIVES_AUTOGENERATION,
                            Client.NOT_AGREED_ON_CREATIVES_AUTOGENERATION)
                    .put(ClientFlags.CAN_COPY_CTR, Client.CAN_COPY_CTR)
                    .put(ClientFlags.NOT_CONVERT_TO_CURRENCY, Client.NOT_CONVERT_TO_CURRENCY)
                    .put(ClientFlags.AUTO_VIDEO, Client.AUTO_VIDEO)
                    .put(ClientFlags.SUSPEND_VIDEO, Client.SUSPEND_VIDEO)
                    .put(ClientFlags.FEATURE_ACCESS_AUTO_VIDEO, Client.FEATURE_ACCESS_AUTO_VIDEO)
                    .put(ClientFlags.ADD_WITHOUT_SHARED_ACCOUNT, Client.SHARED_ACCOUNT_DISABLED)
                    .put(ClientFlags.FEATURE_CONTEXT_RELEVANCE_MATCH_ALLOWED, Client.FEATURE_CONTEXT_RELEVANCE_MATCH_ALLOWED)
                    .put(ClientFlags.FEATURE_CONTEXT_RELEVANCE_MATCH_INTERFACE_ONLY, Client.FEATURE_CONTEXT_RELEVANCE_MATCH_INTERFACE_ONLY)
                    .put(ClientFlags.CANT_UNBLOCK, Client.CANT_UNBLOCK)
                    .put(ClientFlags.CAN_PAY_BEFORE_MODERATION, Client.CAN_PAY_BEFORE_MODERATION)
                    .put(ClientFlags.AS_SOON_AS_POSSIBLE, Client.AS_SOON_AS_POSSIBLE)
                    .put(ClientFlags.AUTO_OVERDRAFT_NOTIFIED, Client.AUTO_OVERDRAFT_NOTIFIED)
                    .put(ClientFlags.IS_PRO_STRATEGY_VIEW_ENABLED, Client.IS_PRO_STRATEGY_VIEW_ENABLED)
                    .put(ClientFlags.VIDEOHINTS_DISABLED, Client.VIDEOHINTS_DISABLED)
                    .put(ClientFlags.IS_TOUCH, Client.IS_TOUCH)
                    .put(ClientFlags.IS_CONVERSION_MULTIPLIERS_POPUP_DISABLED, Client.IS_CONVERSION_MULTIPLIERS_POPUP_DISABLED)
                    .build();

    public static final ImmutableMap<String, ModelProperty<? super Client, Boolean>> CLIENTS_OPTIONS_FLAGS_ENTRIES =
            EntryStream.of(TYPED_CLIENTS_OPTIONS_FLAGS_ENTRIES)
                    .mapKeys(ClientFlags::getTypedValue)
                    .collect(ImmutableMap.toImmutableMap(Map.Entry::getKey, Map.Entry::getValue));

    public static final ImmutableSet<ModelProperty<? super Client, Boolean>> CLIENTS_OPTION_FLAGS =
            ImmutableSet.copyOf(TYPED_CLIENTS_OPTIONS_FLAGS_ENTRIES.values());

    private ClientOptionsMapping() {
    }

    @Nonnull
    static String extractOptionFlags(Client client) {
        return EntryStream.of(TYPED_CLIENTS_OPTIONS_FLAGS_ENTRIES)
                .mapValues(prop -> prop.get(client))
                .filterValues(Boolean.TRUE::equals)
                .keys()
                .map(ClientFlags::getTypedValue)
                .joining(",");
    }

    @Nullable
    public static String commonMetrikaCountersToDb(@Nullable List<Long> counters) {
        return counters != null ? StringUtils.join(counters, ",") : null;
    }

    @Nullable
    public static List<Long> commonMetrikaCountersFromDb(@Nullable String counters) {
        Function<String, List<Long>> converter =
                counter -> StreamEx.split(counter, ",").map(Long::valueOf).toList();
        return counters != null ? converter.apply(counters) : null;
    }

    @Nullable
    public static Set<String> clientFlagsFromDb(@Nullable String clientFlags) {
        if (clientFlags == null) {
            return null;
        }
        if (clientFlags.isEmpty()) {
            return emptySet();
        }

        return StreamEx.split(clientFlags, ",").toSet();
    }

    @Nullable
    public static String clientFlagsToDb(@Nullable Set<String> clientFlags) {
        if (clientFlags == null) {
            return null;
        }
        if (clientFlags.isEmpty()) {
            return "";
        }

        return StreamEx.of(clientFlags).joining(",");
    }
}
