package ru.yandex.direct.core.entity.client.repository;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertOnDuplicateSetMoreStep;
import org.jooq.InsertOnDuplicateSetStep;
import org.jooq.InsertOnDuplicateStep;
import org.jooq.Record;
import org.jooq.SelectFinalStep;
import org.jooq.SelectForUpdateStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.client.model.ClientFlags;
import ru.yandex.direct.core.entity.client.model.ClientsOptions;
import ru.yandex.direct.core.entity.client.model.PhoneVerificationStatus;
import ru.yandex.direct.dbschema.ppc.enums.ClientsOptionsStatusbalancebanned;
import ru.yandex.direct.dbschema.ppc.tables.records.ClientsOptionsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.jooqmapperhelper.UpdateHelper;
import ru.yandex.direct.model.AppliedChanges;

import static org.jooq.impl.DSL.coalesce;
import static org.jooq.impl.DSL.not;
import static org.jooq.impl.DSL.or;
import static org.jooq.util.mysql.MySQLDSL.values;
import static ru.yandex.direct.common.util.RepositoryUtils.booleanToLong;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Clients.CLIENTS;
import static ru.yandex.direct.dbutil.SqlUtils.mysqlZeroLocalDate;
import static ru.yandex.direct.dbutil.SqlUtils.setField;
import static ru.yandex.direct.jooqmapper.JooqMapperUtils.mysqlIf;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Repository
public class ClientOptionsRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<ClientsOptions> clientsOptionsMapper;

    @Autowired
    public ClientOptionsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.clientsOptionsMapper = JooqMapperWithSupplierBuilder.builder(ClientsOptions::new)
                .map(property(ClientsOptions.ID, CLIENTS_OPTIONS.CLIENT_ID))
                .map(convertibleProperty(ClientsOptions.IS_YA_AGENCY_CLIENT, CLIENTS_OPTIONS.IS_YA_AGENCY_CLIENT,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::booleanToLong))
                .map(property(ClientsOptions.NEXT_PAY_DATE, CLIENTS_OPTIONS.NEXT_PAY_DATE))
                .map(property(ClientsOptions.DEBT, CLIENTS_OPTIONS.DEBT))
                .map(convertibleProperty(ClientsOptions.COMMON_METRIKA_COUNTERS,
                        CLIENTS_OPTIONS.COMMON_METRIKA_COUNTERS,
                        ClientOptionsMapping::commonMetrikaCountersFromDb,
                        ClientOptionsMapping::commonMetrikaCountersToDb))
                .map(convertibleProperty(ClientsOptions.CLIENT_FLAGS, CLIENTS_OPTIONS.CLIENT_FLAGS,
                        ClientOptionsMapping::clientFlagsFromDb, ClientOptionsMapping::clientFlagsToDb))
                .map(property(ClientsOptions.CONSUMED_CASHBACK_BONUS, CLIENTS_OPTIONS.CASHBACK_BONUS))
                .map(property(ClientsOptions.AWAITING_CASHBACK_BONUS, CLIENTS_OPTIONS.CASHBACK_AWAITING_BONUS))
                .build();
    }

    /**
     * Условие на то, что у клиента влючен автоовердрафт
     */
    private static final Condition IS_AUTO_OVERDRAFT_SWITCHED_ON =
            CLIENTS_OPTIONS.AUTO_OVERDRAFT_LIM.gt(BigDecimal.ZERO)
                    .and(CLIENTS_OPTIONS.OVERDRAFT_LIM.gt(BigDecimal.ZERO))
                    .and(coalesce(CLIENTS_OPTIONS.STATUS_BALANCE_BANNED, ClientsOptionsStatusbalancebanned.No)
                            .ne(ClientsOptionsStatusbalancebanned.Yes));

    public static final Field<Boolean> IS_AUTO_OVERDRAFT_SWITCHED_ON_FIELD =
            DSL.when(IS_AUTO_OVERDRAFT_SWITCHED_ON, true).otherwise(false).as("isAutoOverdraftSwitchedOn");

    /**
     * Добавляет в секцию {@literal ON DUPLICATE KEY UPDATE} конструкцию вида
     * <pre>
     *     field = if(values(balance_tid) > balance_tid, values(field), field)
     * </pre>
     *
     * @param step  {@code InsertOnDuplicateSetStep}, в который будет добавлена конструкция
     * @param field поле для добавления в запрос
     * @return {@code InsertOnDuplicateSetMoreStep} от переданного {@code step}
     */
    private static <T> InsertOnDuplicateSetMoreStep<ClientsOptionsRecord> setFieldValueIfBalanceTidIncreases(
            InsertOnDuplicateSetStep<ClientsOptionsRecord> step, Field<T> field) {
        return step.set(field, mysqlIf(
                values(CLIENTS_OPTIONS.BALANCE_TID).greaterThan(CLIENTS_OPTIONS.BALANCE_TID),
                values(field),
                field));
    }

    public List<ClientsOptions> getClientsOptions(int shard, Collection<ClientId> clientIds) {
        return getClientsOptions(dslContextProvider.ppc(shard), clientIds, false);
    }

    public List<ClientsOptions> getClientsOptions(DSLContext dslContext, Collection<ClientId> clientIds,
                                                  boolean updateLock) {
        SelectFinalStep<Record> selectStep = dslContext.select(clientsOptionsMapper.getFieldsToRead())
                .from(CLIENTS_OPTIONS)
                .where(CLIENTS_OPTIONS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)));
        if (updateLock) {
            //noinspection unchecked
            selectStep = ((SelectForUpdateStep) selectStep).forUpdate();
        }
        return selectStep.fetch(clientsOptionsMapper::fromDb);
    }

    /**
     * при отсутствии соответвующих строк в clients_options, map так-же не будет их содержать
     */
    public Map<ClientId, Boolean> getIsYaAgencyClientByClientId(int shard, Collection<ClientId> clientIds) {
        return StreamEx.of(getClientsOptions(shard, clientIds))
                .mapToEntry(ClientsOptions::getId, ClientsOptions::getIsYaAgencyClient)
                .mapKeys(ClientId::fromLong)
                .toMap();
    }

    public Map<ClientId, BigDecimal> getOverdraftRestByClientIds(int shard, Collection<ClientId> clientIds) {
        Field<BigDecimal> overdraftRestField = CLIENTS_OPTIONS.OVERDRAFT_LIM
                .sub(CLIENTS_OPTIONS.DEBT).as("overdraft_rest");
        return dslContextProvider.ppc(shard)
                .select(CLIENTS_OPTIONS.CLIENT_ID, overdraftRestField)
                .from(CLIENTS_OPTIONS)
                .where(CLIENTS_OPTIONS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .fetchMap(r -> ClientId.fromLong(r.get(CLIENTS_OPTIONS.CLIENT_ID)),
                        r -> r.get(overdraftRestField));
    }

    public void update(int shard, Collection<AppliedChanges<ClientsOptions>> appliedChanges) {
        DSLContext context = dslContextProvider.ppc(shard);
        new UpdateHelper<>(context, CLIENTS_OPTIONS.CLIENT_ID)
                .processUpdateAll(clientsOptionsMapper, appliedChanges)
                .execute();
    }

    /**
     * Добавляет/обновляет параметры клиента, связанные с балансом. Добавление происходит безусловно.
     * Обновление произойдет только если внешний {@code balanceTid} для клиента
     * больше внутреннего(хранится в базе Директа); состояние записи не поменяется, если условие не выполнено.
     * Для {@code nextPayDate} null-значения кодируются 'нулевой' mysql-датой '0000-00-00'
     */
    public void insertOrUpdateBalanceClientOptions(int shard, ClientId clientId, long balanceTid,
                                                   BigDecimal overdraftLim, BigDecimal debt,
                                                   @Nullable LocalDate nextPayDate, @Nullable Boolean balanceBanned,
                                                   Boolean nonResident, @Nullable Boolean isBusinessUnit,
                                                   @Nullable Boolean isBrand) {
        DSLContext dslContext = dslContextProvider.ppc(shard);

        Field<LocalDate> nextPayDateField = nextPayDate == null
                ? mysqlZeroLocalDate()
                : DSL.value(nextPayDate);
        ClientsOptionsStatusbalancebanned balanceBannedValue = balanceBanned == null
                ? null
                : balanceBanned ? ClientsOptionsStatusbalancebanned.Yes : ClientsOptionsStatusbalancebanned.No;

        InsertOnDuplicateStep<ClientsOptionsRecord> insertStep = dslContext.insertInto(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.CLIENT_ID, clientId.asLong())
                .set(CLIENTS_OPTIONS.BALANCE_TID, balanceTid)
                .set(CLIENTS_OPTIONS.OVERDRAFT_LIM, overdraftLim)
                .set(CLIENTS_OPTIONS.DEBT, debt)
                .set(CLIENTS_OPTIONS.NEXT_PAY_DATE, nextPayDateField)
                .set(CLIENTS_OPTIONS.STATUS_BALANCE_BANNED, balanceBannedValue)
                .set(CLIENTS_OPTIONS.NON_RESIDENT, booleanToLong(nonResident))
                .set(CLIENTS_OPTIONS.IS_BUSINESS_UNIT,
                        DSL.nvl(booleanToLong(isBusinessUnit), CLIENTS_OPTIONS.IS_BUSINESS_UNIT))
                .set(CLIENTS_OPTIONS.IS_BRAND,
                        DSL.nvl(booleanToLong(isBrand), CLIENTS_OPTIONS.IS_BRAND));

        InsertOnDuplicateSetStep<ClientsOptionsRecord> onConflictStep = insertStep.onDuplicateKeyUpdate();
        setFieldValueIfBalanceTidIncreases(onConflictStep, CLIENTS_OPTIONS.OVERDRAFT_LIM);
        setFieldValueIfBalanceTidIncreases(onConflictStep, CLIENTS_OPTIONS.DEBT);
        setFieldValueIfBalanceTidIncreases(onConflictStep, CLIENTS_OPTIONS.NEXT_PAY_DATE);
        setFieldValueIfBalanceTidIncreases(onConflictStep, CLIENTS_OPTIONS.STATUS_BALANCE_BANNED);
        setFieldValueIfBalanceTidIncreases(onConflictStep, CLIENTS_OPTIONS.NON_RESIDENT);
        if (isBusinessUnit != null) {
            setFieldValueIfBalanceTidIncreases(onConflictStep, CLIENTS_OPTIONS.IS_BUSINESS_UNIT);
        }
        if (isBrand != null) {
            setFieldValueIfBalanceTidIncreases(onConflictStep, CLIENTS_OPTIONS.IS_BRAND);
        }

        //N.B.: обновление tid должно идти в самом конце запроса!
        setFieldValueIfBalanceTidIncreases(onConflictStep, CLIENTS_OPTIONS.BALANCE_TID).execute();
    }

    public boolean updateAutoOverdraftLimit(int shard, ClientId clientId, BigDecimal autoOverdraftLim) {
        return updateAutoOverdraftLimit(dslContextProvider.ppc(shard).configuration(), clientId, autoOverdraftLim);
    }

    public boolean updateAutoOverdraftLimit(Configuration conf, ClientId clientId, BigDecimal autoOverdraftLim) {
        return conf.dsl()
                .update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.AUTO_OVERDRAFT_LIM, autoOverdraftLim)
                .where(CLIENTS_OPTIONS.CLIENT_ID.eq(clientId.asLong())
                        .and(CLIENTS_OPTIONS.AUTO_OVERDRAFT_LIM.ne(autoOverdraftLim)))
                .execute() > 0;
    }

    public boolean updateCanManagePricePackage(int shard, ClientId clientId, Boolean canManagePricePackage) {
        return dslContextProvider.ppc(shard).configuration().dsl()
                .update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.CAN_MANAGE_PRICE_PACKAGES, booleanToLong(canManagePricePackage))
                .where(CLIENTS_OPTIONS.CLIENT_ID.eq(clientId.asLong()))
                .execute() > 0;
    }

    public boolean updateCanApprovePricePackage(int shard, ClientId clientId, Boolean canApprovePricePackage) {
        return dslContextProvider.ppc(shard).configuration().dsl()
                .update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.CAN_APPROVE_PRICE_PACKAGES, booleanToLong(canApprovePricePackage))
                .where(CLIENTS_OPTIONS.CLIENT_ID.eq(clientId.asLong()))
                .execute() > 0;
    }

    public void updateSocialAdvertising(int shard, Collection<ClientId> clientIds, boolean socialAdvertising) {
        dslContextProvider.ppc(shard).configuration().dsl()
                .update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.SOCIAL_ADVERTISING, booleanToLong(socialAdvertising))
                .where(CLIENTS_OPTIONS.CLIENT_ID.in(clientIds))
                .execute();
    }

    public Map<Long, List<Long>> getCommonMetrikaCountersByClientId(int shard, Collection<ClientId> clientIds) {
        List<ClientsOptions> clientsOptionsList = getClientsOptions(shard, clientIds);
        return StreamEx.of(clientsOptionsList)
                .mapToEntry(ClientsOptions::getId, ClientsOptions::getCommonMetrikaCounters)
                .nonNullValues()
                .mapValues(counters -> counters.stream()
                        .distinct()
                        .collect(Collectors.toList()))
                .toMap();
    }

    public void updateClientsFlags(int shard, Collection<AppliedChanges<ClientsOptions>> clientsOptionsChanges) {
        updateClientsFlags(dslContextProvider.ppc(shard), clientsOptionsChanges);
    }

    public void updateClientsFlags(DSLContext dslContext,
                                   Collection<AppliedChanges<ClientsOptions>> clientsOptionsChanges) {
        JooqUpdateBuilder<ClientsOptionsRecord, ClientsOptions> updateBuilder =
                new JooqUpdateBuilder<>(CLIENTS_OPTIONS.CLIENT_ID, clientsOptionsChanges);

        updateBuilder.processProperty(ClientsOptions.CLIENT_FLAGS, CLIENTS_OPTIONS.CLIENT_FLAGS,
                ClientOptionsMapping::clientFlagsToDb);

        dslContext
                .update(CLIENTS_OPTIONS)
                .set(updateBuilder.getValues())
                .where(CLIENTS_OPTIONS.CLIENT_ID.in(updateBuilder.getChangedIds()))
                .execute();
    }

    public boolean updateCashBackBonus(int shard, ClientId clientId, BigDecimal cashBackBonus,
                                       BigDecimal awaitingBonus) {
        return updateCashBackBonus(dslContextProvider.ppc(shard).configuration(), clientId, cashBackBonus,
                awaitingBonus);
    }

    public boolean updateCashBackBonus(Configuration conf, ClientId clientId, BigDecimal cashBackBonus,
                                       BigDecimal awaitingBonus) {
        return conf.dsl()
                .update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.CASHBACK_BONUS, cashBackBonus)
                .set(CLIENTS_OPTIONS.CASHBACK_AWAITING_BONUS, awaitingBonus)
                .where(CLIENTS_OPTIONS.CLIENT_ID.eq(clientId.asLong()))
                .execute() > 0;
    }

    /**
     * Вернуть клиентов у которых не подключен автоовердрафт и которым не отправляли письма о доступности
     * "Порога отключения" (автоовердрафта)
     */
    public Set<ClientId> getClientsAutoOverdraftNotNotified(int shard, Collection<ClientId> clientIds) {
        return dslContextProvider.ppc(shard)
                .select(CLIENTS_OPTIONS.CLIENT_ID)
                .from(CLIENTS_OPTIONS)
                .where(CLIENTS_OPTIONS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .and(or(CLIENTS_OPTIONS.CLIENT_FLAGS.isNull(),
                        not(setField(CLIENTS_OPTIONS.CLIENT_FLAGS).contains(ClientFlags.AUTO_OVERDRAFT_NOTIFIED.getTypedValue()))))
                .and(not(IS_AUTO_OVERDRAFT_SWITCHED_ON))
                .fetchSet(record -> ClientId.fromLong(record.get(CLIENTS.CLIENT_ID)));
    }

    /**
     * Устанавливает конкретный флаг в наборе флагов CLIENTS_OPTIONS.CLIENT_FLAGS
     * Остальные значения флагов не меняются
     */
    public void setClientFlag(int shard, ClientId clientId, ClientFlags clientFlag) {
        String flag = clientFlag.getTypedValue();
        dslContextProvider.ppc(shard)
                .update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.CLIENT_FLAGS,
                        DSL.iif(CLIENTS_OPTIONS.CLIENT_FLAGS.isNull().or(CLIENTS_OPTIONS.CLIENT_FLAGS.eq("")),
                                flag, CLIENTS_OPTIONS.CLIENT_FLAGS.concat(",", flag)))
                .where(CLIENTS_OPTIONS.CLIENT_ID.eq(clientId.asLong()))
                .execute();
    }

    public void updatePhoneVerificationStatus(int shard, ClientId clientId,
                                              PhoneVerificationStatus phoneVerificationStatus) {
        dslContextProvider.ppc(shard)
                .update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.PHONE_VERIFICATION_STATUS,
                        PhoneVerificationStatus.toSource(phoneVerificationStatus))
                .set(CLIENTS_OPTIONS.LAST_CHANGE_PHONE_VERIFICATION_STATUS, LocalDateTime.now())
                .where(CLIENTS_OPTIONS.CLIENT_ID.eq(clientId.asLong()))
                .execute();
    }
}
