package ru.yandex.direct.core.entity.client.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertValuesStep1;
import org.jooq.Record;
import org.jooq.Record2;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperWithSupplier;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.client.container.ClientsQueryFilter;
import ru.yandex.direct.core.entity.client.container.PrimaryManagersQueryFilter;
import ru.yandex.direct.core.entity.client.model.AgencyStatus;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientAutoOverdraftInfo;
import ru.yandex.direct.core.entity.client.model.ClientDataForNds;
import ru.yandex.direct.core.entity.client.model.ClientExperiment;
import ru.yandex.direct.core.entity.client.model.ClientPrimaryManager;
import ru.yandex.direct.core.entity.client.model.ClientWithOptions;
import ru.yandex.direct.core.entity.client.model.ClientWithUsers;
import ru.yandex.direct.core.entity.client.model.PhoneVerificationStatus;
import ru.yandex.direct.core.entity.client.model.TinType;
import ru.yandex.direct.core.entity.user.model.ApiEnabled;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.repository.ApiUserRepository;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbschema.ppc.enums.ClientsAllowCreateScampBySubclient;
import ru.yandex.direct.dbschema.ppc.enums.ClientsApiOptionsApiEnabled;
import ru.yandex.direct.dbschema.ppc.enums.ClientsOptionsStatusbalancebanned;
import ru.yandex.direct.dbschema.ppc.enums.ClientsRole;
import ru.yandex.direct.dbschema.ppc.enums.ClientsWorkCurrency;
import ru.yandex.direct.dbschema.ppc.enums.UsersRepType;
import ru.yandex.direct.dbschema.ppc.enums.UsersStatusblocked;
import ru.yandex.direct.dbschema.ppc.tables.records.ClientsOptionsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.ClientsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.rbac.ClientPerm;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.RbacSubrole;

import static java.math.BigDecimal.ZERO;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.setField;
import static ru.yandex.direct.common.jooqmapperex.FieldMapperFactoryEx.booleanField;
import static ru.yandex.direct.common.jooqmapperex.FieldMapperFactoryEx.timestampField;
import static ru.yandex.direct.common.jooqmapperex.FieldMapperFactoryEx.yesNoField;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromLongFieldToBoolean;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromYesNoEnumFieldToBoolean;
import static ru.yandex.direct.common.util.GuavaCollectors.toMultimap;
import static ru.yandex.direct.common.util.RepositoryUtils.booleanToLong;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_API_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_TO_FETCH_NDS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENT_FIRM_COUNTRY_CURRENCY;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppc.tables.Clients.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.tables.ClientsOptions.CLIENTS_OPTIONS;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class ClientRepository {
    private final RbacService rbacService; //  ситуация аналогична UserRepository
    private final DslContextProvider ppcDslContextProvider;
    private final OldJooqMapperWithSupplier<Client> clientWithOptionsJooqMapper;
    private final JooqReaderWithSupplier<ClientPrimaryManager> primaryManagerReader;
    private final JooqReaderWithSupplier<ClientAutoOverdraftInfo> clientAutoOverdraftInfoReader;
    private final JooqReaderWithSupplier<Client> clientDataForNdsReader;
    private final JooqReaderWithSupplier<ClientExperiment> clientExperimentReader;
    private final ShardHelper shardHelper;

    @Autowired
    public ClientRepository(DslContextProvider ppcDslContextProvider, RbacService rbacService, ShardHelper shardHelper) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.rbacService = rbacService;
        this.shardHelper = shardHelper;

        clientWithOptionsJooqMapper = new OldJooqMapperBuilder<>(Client::new, ClientMapping::postProcess)
                .map(field(CLIENTS.CLIENT_ID, Client.ID))
                .map(field(CLIENTS.NAME, Client.NAME))
                .map(field(CLIENTS.CHIEF_UID, Client.CHIEF_UID))
                .map(field(CLIENTS.AGENCY_CLIENT_ID, Client.AGENCY_CLIENT_ID))
                .map(field(CLIENTS.AGENCY_UID, Client.AGENCY_USER_ID))
                .map(convertibleField(CLIENTS.ROLE, Client.ROLE)
                        .convertFromDbBy(RbacRole::fromSource)
                        .convertToDbBy(RbacRole::toSource))
                .map(field(CLIENTS.COUNTRY_REGION_ID, Client.COUNTRY_REGION_ID))
                .map(convertibleField(CLIENTS.WORK_CURRENCY, Client.WORK_CURRENCY)
                        .convertToDbBy(ClientMapping::workCurrencyToDb)
                        .convertFromDbBy(ClientMapping::workCurrencyFromDb)
                        .withDatabaseDefault())
                .map(timestampField(CLIENTS.CREATE_DATE, Client.CREATE_DATE)
                        .withDatabaseDefault())
                .map(field(CLIENTS.DELETED_REPS, Client.DELETED_REPS))
                .map(field(CLIENTS.AGENCY_URL, Client.AGENCY_URL))
                .map(convertibleField(CLIENTS.AGENCY_STATUS, Client.AGENCY_STATUS)
                        .convertFromDbBy(AgencyStatus::fromSource)
                        .convertToDbBy(AgencyStatus::toSource)
                        .withDatabaseDefault())
                .map(field(CLIENTS.PRIMARY_MANAGER_UID, Client.PRIMARY_MANAGER_UID))
                .map(convertibleField(CLIENTS.PRIMARY_MANAGER_SET_BY_IDM, Client.IS_IDM_PRIMARY_MANAGER)
                        .convertFromDbBy(RepositoryUtils::booleanFromLong)
                        .convertToDbBy(RepositoryUtils::booleanToLong))
                .map(field(CLIENTS.PRIMARY_BAYAN_MANAGER_UID, Client.PRIMARY_BAYAN_MANAGER_UID))
                .map(field(CLIENTS.PRIMARY_GEO_MANAGER_UID, Client.PRIMARY_GEO_MANAGER_UID))
                .map(booleanField(CLIENTS.IS_FAVICON_BLOCKED, Client.FAVICON_BLOCKED)
                        .withDatabaseDefault())
                .map(yesNoField(CLIENTS.ALLOW_CREATE_SCAMP_BY_SUBCLIENT, Client.ALLOW_CREATE_SCAMP_BY_SUBCLIENT,
                        ClientsAllowCreateScampBySubclient.class)
                        .withDatabaseDefault())
                .map(field(CLIENTS.CONNECT_ORG_ID, Client.CONNECT_ORG_ID))
                .map(field(CLIENTS_OPTIONS.OVERDRAFT_LIM, Client.OVERDRAFT_LIMIT)
                        .withDefaultValueForModel(ZERO)
                        .withDatabaseDefault())
                .map(field(CLIENTS_OPTIONS.DEBT, Client.DEBT)
                        .withDefaultValueForModel(ZERO)
                        .withDatabaseDefault())
                .map(field(CLIENTS_OPTIONS.CASHBACK_BONUS, Client.CASH_BACK_BONUS))
                .map(field(CLIENTS_OPTIONS.CASHBACK_AWAITING_BONUS, Client.CASH_BACK_AWAITING_BONUS))
                .map(field(CLIENTS_OPTIONS.AUTO_OVERDRAFT_LIM, Client.AUTO_OVERDRAFT_LIMIT)
                        .withDefaultValueForModel(ZERO)
                        .withDatabaseDefault())
                .map(convertibleField(CLIENTS_OPTIONS.STATUS_BALANCE_BANNED, Client.STATUS_BALANCE_BANNED)
                        .convertFromDbBy(ClientsOptionsStatusbalancebanned.Yes::equals)
                        .convertToDbBy(value -> nvl(value, false)
                                ? ClientsOptionsStatusbalancebanned.Yes
                                : ClientsOptionsStatusbalancebanned.No)
                        .withDefaultValueForModel(Boolean.FALSE)
                        .withDatabaseDefault())
                .map(booleanField(CLIENTS_OPTIONS.HIDE_MARKET_RATING, Client.HIDE_MARKET_RATING)
                        .withDefaultValueForModel(false)
                        .withDatabaseDefault())
                .map(booleanField(CLIENTS_OPTIONS.NON_RESIDENT, Client.NON_RESIDENT)
                        .withDefaultValueForModel(false)
                        .withDatabaseDefault())
                .map(setField(CLIENTS_OPTIONS.CLIENT_FLAGS, ClientOptionsMapping.CLIENTS_OPTIONS_FLAGS_ENTRIES))
                .map(booleanField(CLIENTS_OPTIONS.IS_BUSINESS_UNIT, Client.IS_BUSINESS_UNIT)
                        .withDefaultValueForModel(false))
                .map(booleanField(CLIENTS_OPTIONS.IS_BRAND, Client.IS_BRAND)
                        .withDefaultValueForModel(false))
                .map(booleanField(CLIENTS_OPTIONS.IS_USING_QUASI_CURRENCY, Client.USES_QUASI_CURRENCY)
                        .disableWritingToDb())
                .map(booleanField(CLIENTS_OPTIONS.SOCIAL_ADVERTISING, Client.SOCIAL_ADVERTISING)
                        .disableWritingToDb())
                .map(convertibleField(CLIENTS_OPTIONS.DEFAULT_DISALLOWED_PAGE_IDS, Client.DEFAULT_DISALLOWED_PAGE_IDS)
                        .convertFromDbBy(ClientMapping::disallowedPageIdsFromDbFormat)
                        .convertToDbBy(ClientMapping::disallowedPageIdsToDbFormat))
                .map(field(CLIENTS_OPTIONS.TIN, Client.TIN))
                .map(convertibleField(CLIENTS_OPTIONS.TIN_TYPE, Client.TIN_TYPE)
                        .convertFromDbBy(TinType::fromSource)
                        .convertToDbBy(TinType::toSource)
                        .withDatabaseDefault())
                .map(convertibleField(CLIENTS_OPTIONS.PHONE_VERIFICATION_STATUS, Client.PHONE_VERIFICATION_STATUS)
                        .convertFromDbBy(PhoneVerificationStatus::fromSource)
                        .convertToDbBy(PhoneVerificationStatus::toSource)
                        .withDatabaseDefault())
                .map(convertibleField(CLIENTS_OPTIONS.DEFAULT_ALLOWED_DOMAINS, Client.DEFAULT_ALLOWED_DOMAINS)
                        .convertFromDbBy(ClientMapping::defaultAllowedDomainsFromDbFormat)
                        .disableWritingToDb())
                .build();

        primaryManagerReader = selectPrimaryManagerMapper();

        clientAutoOverdraftInfoReader = createClientAutoOverdraftInfoReader();

        clientDataForNdsReader = createClientDataForNdsReader();

        clientExperimentReader = createClientExperimentReader();

    }

    private static JooqReaderWithSupplier<ClientPrimaryManager> selectPrimaryManagerMapper() {
        return JooqReaderWithSupplierBuilder.builder(ClientPrimaryManager::new)
                .readProperty(ClientPrimaryManager.SUBJECT_CLIENT_ID,
                        fromField(CLIENTS.CLIENT_ID).by(ClientId::fromLong))
                .readProperty(ClientPrimaryManager.PRIMARY_MANAGER_UID, fromField(CLIENTS.PRIMARY_MANAGER_UID))
                .readProperty(ClientPrimaryManager.IS_IDM_PRIMARY_MANAGER,
                        fromField(CLIENTS.PRIMARY_MANAGER_SET_BY_IDM).by(RepositoryUtils::booleanFromLong))
                .build();
    }

    private static JooqReaderWithSupplier<ClientAutoOverdraftInfo> createClientAutoOverdraftInfoReader() {
        return JooqReaderWithSupplierBuilder.builder(ClientAutoOverdraftInfo::new)
                .readProperty(ClientAutoOverdraftInfo.CLIENT_ID, fromField(Tables.CLIENTS_OPTIONS.CLIENT_ID))
                .readProperty(ClientAutoOverdraftInfo.DEBT, fromField(Tables.CLIENTS_OPTIONS.DEBT))
                .readProperty(ClientAutoOverdraftInfo.STATUS_BALANCE_BANNED,
                        fromYesNoEnumFieldToBoolean(Tables.CLIENTS_OPTIONS.STATUS_BALANCE_BANNED))
                .readProperty(ClientAutoOverdraftInfo.OVERDRAFT_LIMIT, fromField(Tables.CLIENTS_OPTIONS.OVERDRAFT_LIM))
                .readProperty(ClientAutoOverdraftInfo.AUTO_OVERDRAFT_LIMIT, fromField(Tables.CLIENTS_OPTIONS.AUTO_OVERDRAFT_LIM))
                .build();
    }

    private static JooqReaderWithSupplier<Client> createClientDataForNdsReader() {
        return JooqReaderWithSupplierBuilder.builder(Client::new)
                .readProperty(ClientDataForNds.ID, fromField(Tables.CLIENTS.CLIENT_ID))
                .readProperty(ClientDataForNds.AGENCY_CLIENT_ID, fromField(Tables.CLIENTS.AGENCY_CLIENT_ID))
                .readProperty(ClientDataForNds.NON_RESIDENT, fromLongFieldToBoolean(Tables.CLIENTS_OPTIONS.NON_RESIDENT))
                .build();
    }

    private static JooqReaderWithSupplier<ClientExperiment> createClientExperimentReader() {
        return JooqReaderWithSupplierBuilder.builder(ClientExperiment::new)
                .readProperty(ClientExperiment.EXPERIMENT_ID, fromField(Tables.CLIENTS_OPTIONS.EXPERIMENT_ID))
                .readProperty(ClientExperiment.SEGMENT_ID, fromField(Tables.CLIENTS_OPTIONS.SEGMENT_ID))
                .build();
    }

    @Nonnull
    public List<Client> get(int shard, Collection<ClientId> clientIds) {
        return getClientsByFilter(shard, ClientsQueryFilter.getByClientIds(clientIds));
    }

    @Nonnull
    public List<Client> getClientsByFilter(int shard, ClientsQueryFilter filter) {
        Condition condition = DSL.trueCondition();
        if (filter.getClientIds() != null) {
            condition = condition.and(CLIENTS.CLIENT_ID.in(mapList(filter.getClientIds(), ClientId::asLong)));
        }
        if (filter.getLastClientId() != null) {
            condition = condition.and(CLIENTS.CLIENT_ID.greaterThan(filter.getLastClientId().asLong()));
        }
        if (filter.getConnectOrgIdRequired()) {
            condition = condition.and(CLIENTS.CONNECT_ORG_ID.isNotNull());
        }
        if (filter.getConnectOrgIds() != null) {
            condition = condition.and(CLIENTS.CONNECT_ORG_ID.in(filter.getConnectOrgIds()));
        }
        SelectConditionStep<Record> conditionStep = ppcDslContextProvider.ppc(shard)
                .select(clientWithOptionsJooqMapper.getFieldsToRead())
                .from(CLIENTS)
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .where(condition);
        if (filter.getLimitOffset() != null) {
            conditionStep
                    .orderBy(CLIENTS.CLIENT_ID)
                    .offset(filter.getLimitOffset().offset())
                    .limit(filter.getLimitOffset().limit());
        }
        return conditionStep
                .fetch(clientWithOptionsJooqMapper::fromDb);
    }

    /**
     * Вернуть множество субклиентов, которые являются супер-субклиентами по версии PPC
     */
    public Set<ClientId> getSubclientsAllowedToCreateCamps(int shard, Collection<ClientId> clientIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(CLIENTS.CLIENT_ID)
                .from(CLIENTS)
                .where(
                        CLIENTS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong))
                                .and(CLIENTS.ALLOW_CREATE_SCAMP_BY_SUBCLIENT.eq(
                                        ClientsAllowCreateScampBySubclient.Yes)))
                .fetchSet(r -> ClientId.fromLong(r.get(CLIENTS.CLIENT_ID)));
    }

    /**
     * Является ли субклиент супер-субклиентом по версии PPC
     */
    public boolean isSubclientAllowedToCreateCamps(int shard, ClientId clientId) {
        return getSubclientsAllowedToCreateCamps(shard, singletonList(clientId))
                .contains(clientId);
    }

    /**
     * Возвращает валюту клиента, как она представлена в БД
     *
     * @param shard    shard
     * @param clientId client ID
     * @return {@link Currency}
     */
    @Nonnull
    public ClientsWorkCurrency getWorkCurrency(int shard, ClientId clientId) {
        return ppcDslContextProvider.ppc(shard)
                .select(CLIENTS.WORK_CURRENCY)
                .from(CLIENTS)
                .where(CLIENTS.CLIENT_ID.eq(clientId.asLong()))
                .fetchOptional(CLIENTS.WORK_CURRENCY)
                .orElse(ClientsWorkCurrency.YND_FIXED);
    }

    public List<ClientPrimaryManager> getIdmPrimaryManagers(int shard, PrimaryManagersQueryFilter filter) {
        Condition condition = CLIENTS.PRIMARY_MANAGER_UID.isNotNull()
                .and(CLIENTS.PRIMARY_MANAGER_SET_BY_IDM.eq(RepositoryUtils.TRUE));

        if (filter.getLastClientId() != null) {
            Long lastClientId = filter.getLastClientId().asLong();
            condition = condition.and(CLIENTS.CLIENT_ID.greaterThan(lastClientId));
        }

        SelectConditionStep<Record> conditionStep = ppcDslContextProvider.ppc(shard)
                .select(primaryManagerReader.getFieldsToRead())
                .from(CLIENTS)
                .where(condition);

        if (filter.getLimitOffset() != null) {
            conditionStep
                    .orderBy(CLIENTS.CLIENT_ID, CLIENTS.PRIMARY_MANAGER_UID)
                    .offset(filter.getLimitOffset().offset())
                    .limit(filter.getLimitOffset().limit());
        }

        return conditionStep.fetch(primaryManagerReader::fromDb);
    }

    public Map<ClientId, ClientsWorkCurrency> getWorkCurrencies(int shard, Collection<ClientId> clientIds) {
        Field<ClientsWorkCurrency> currencyField = CLIENTS.WORK_CURRENCY.nvl(ClientsWorkCurrency.YND_FIXED).as("cur");
        Map<ClientId, ClientsWorkCurrency> currencyMap = ppcDslContextProvider.ppc(shard)
                .select(CLIENTS.CLIENT_ID, currencyField)
                .from(CLIENTS)
                .where(CLIENTS.CLIENT_ID.in(clientIds))
                .fetchMap(r -> ClientId.fromLong(r.get(CLIENTS.CLIENT_ID)),
                        r -> r.get(currencyField));
        // клиентам, валюты которых не нашли в базе, сопоставляем YND_FIXED
        clientIds.forEach(clientId -> currencyMap.putIfAbsent(clientId, ClientsWorkCurrency.YND_FIXED));
        return currencyMap;
    }

    public Map<ClientId, RbacRole> getRbacRoles(int shard, Collection<ClientId> clientIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(CLIENTS.CLIENT_ID, CLIENTS.ROLE)
                .from(CLIENTS)
                .where(CLIENTS.CLIENT_ID.in(clientIds))
                .fetchMap(r -> ClientId.fromLong(r.get(CLIENTS.CLIENT_ID)),
                        r -> RbacRole.fromSource(r.get(CLIENTS.ROLE)));
    }

    public List<ClientWithUsers> getClientData(int shard, Collection<ClientId> clientIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(clientWithOptionsJooqMapper.getFieldsToRead())
                .select(asList(USERS.UID, USERS.LOGIN, USERS.FIO, USERS.CLIENT_ID, USERS.STATUS_BLOCKED))
                .from(USERS)
                .leftJoin(CLIENTS).on(USERS.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .where(USERS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .fetchGroups(USERS.CLIENT_ID)
                .values().stream()
                .map(this::convertClientWithUsersRecord)
                .collect(Collectors.toList());
    }

    private ClientWithUsers convertClientWithUsersRecord(Result<Record> records) {
        Record record = records.get(0);
        Client client = clientWithOptionsJooqMapper.fromDb(record);
        Long clientId = record.getValue(USERS.CLIENT_ID);
        client.setId(clientId); // override if client not exists
        // Для заблокированных внутренних пользователей записи в clients может не быть
        Long chiefUid = rbacService.getChiefByClientIdOptional(ClientId.fromLong(clientId))
                .orElse(record.getValue(USERS.UID));

        List<User> users = records.stream()
                .map(u -> new User()
                        .withUid(u.getValue(USERS.UID))
                        .withLogin(u.getValue(USERS.LOGIN))
                        .withClientId(ClientId.fromLong(u.getValue(USERS.CLIENT_ID)))
                        .withFio(u.getValue(USERS.FIO))
                        .withChiefUid(chiefUid)
                        .withStatusBlocked(u.getValue(USERS.STATUS_BLOCKED) == UsersStatusblocked.Yes)
                )
                .sorted(Comparator.comparing(User::getLogin))
                .collect(Collectors.toList());

        return new ClientWithUsers(client, users);
    }

    public List<ClientId> getClientsWithRole(Integer shard, ClientsRole role) {
        return ppcDslContextProvider.ppc(shard)
                .select(CLIENTS.CLIENT_ID)
                .from(CLIENTS)
                .where(CLIENTS.ROLE.eq(role))
                .fetch(r -> ClientId.fromLong(r.get(CLIENTS.CLIENT_ID)));
    }

    public List<Long> getClientChiefUidsWithRoles(Integer shard, Collection<ClientsRole> roles) {
        List<Long> chiefUids = ppcDslContextProvider.ppc(shard)
                .select(CLIENTS.CHIEF_UID)
                .from(CLIENTS)
                .where(CLIENTS.ROLE.in(roles))
                .fetch(CLIENTS.CHIEF_UID);
        return filterList(chiefUids, Objects::nonNull);
    }

    public Map<ClientId, Long> getChiefUidsByClientIds(int shard, Collection<ClientId> clientIds) {
        if (clientIds.isEmpty()) {
            return emptyMap();
        }
        return ppcDslContextProvider.ppc(shard)
                .select(CLIENTS.CLIENT_ID, CLIENTS.CHIEF_UID)
                .from(CLIENTS)
                .where(CLIENTS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .fetchMap(r -> ClientId.fromLong(r.get(CLIENTS.CLIENT_ID)),
                        r -> r.get(CLIENTS.CHIEF_UID));
    }

    /**
     * Вернуть страну клиента
     */
    @Nonnull
    public Optional<Long> getCountryRegionIdByClientId(int shard, ClientId clientId) {
        return Optional.ofNullable(getCountryRegionIdByClientId(shard, singletonList(clientId)).get(clientId.asLong()));
    }

    /**
     * Вернуть мапу (id клиента -> страна клиента)
     */
    public Map<Long, Long> getCountryRegionIdByClientId(int shard, Collection<ClientId> clientIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(CLIENTS.CLIENT_ID, CLIENTS.COUNTRY_REGION_ID)
                .from(CLIENTS)
                .where(CLIENTS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .fetchMap(CLIENTS.CLIENT_ID, CLIENTS.COUNTRY_REGION_ID);
    }

    /**
     * Вернуть валюты в которых может платить клиент в разных странах
     *
     * @return Возвращает отображение идентификатора страны в коллекцию допустимых валют
     */
    @Nonnull
    public Multimap<Long, CurrencyCode> getAllowedForPayCurrencies(int shard, ClientId clientId) {
        return ppcDslContextProvider.ppc(shard)
                .select(CLIENT_FIRM_COUNTRY_CURRENCY.COUNTRY_REGION_ID, CLIENT_FIRM_COUNTRY_CURRENCY.CURRENCY)
                .from(CLIENT_FIRM_COUNTRY_CURRENCY)
                .where(CLIENT_FIRM_COUNTRY_CURRENCY.CLIENT_ID.eq(clientId.asLong()))
                .fetch()
                .stream()
                .collect(toMultimap(Record2::value1, r -> CurrencyCode.valueOf(r.value2().getLiteral())));
    }

    /**
     * Заполняет необходимые таблицы при создании клиента
     */
    public void addClient(int shard, ClientWithOptions clientWithOptions, boolean addToFetchNds) {
        DSLContext dslContext = ppcDslContextProvider.ppc(shard);

        dslContext.transaction(conf -> {

            conf.dsl().insertInto(CLIENTS)
                    .columns(CLIENTS.CLIENT_ID, CLIENTS.CHIEF_UID, CLIENTS.WORK_CURRENCY, CLIENTS.PERMS,
                            CLIENTS.ROLE, CLIENTS.COUNTRY_REGION_ID, CLIENTS.NAME)
                    .values(/* clientId */ clientWithOptions.getClientId().asLong(),
                            /* chiefUid */ clientWithOptions.getUid(),
                            /* currency */ ClientsWorkCurrency.valueOf(clientWithOptions.getCurrency().name()),
                            /* perms */ "",
                            /* role */ RbacRole.toSource(clientWithOptions.getRole()),
                            /* region */ nvl(clientWithOptions.getCountryRegionId(), 0L),
                            /* name */ clientWithOptions.getName())
                    .execute();

            // Используется insert ... select для того, чтобы вставить в поле CLIENTS_OPTIONS.NEXT_PAY_DATE нулевую
            // дату (0000-00-00). Данное поле является по сути nullable, но по факту нет. Null в данном поле
            // эмулируется с помощью нулевой даты. Но так как данное поле mapping-ся в значение типа java.sql.Date
            // для которого данное значение не является допустимым, то приходится хитрить
            conf.dsl().insertInto(CLIENTS_OPTIONS)
                    .columns(
                            CLIENTS_OPTIONS.CLIENT_ID,
                            CLIENTS_OPTIONS.BALANCE_TID,
                            CLIENTS_OPTIONS.OVERDRAFT_LIM,
                            CLIENTS_OPTIONS.DEBT,
                            CLIENTS_OPTIONS.NEXT_PAY_DATE,
                            CLIENTS_OPTIONS.HIDE_MARKET_RATING,
                            CLIENTS_OPTIONS.CLIENT_FLAGS,
                            CLIENTS_OPTIONS.IS_USING_QUASI_CURRENCY)
                    .select(
                            dslContext.select(
                                    DSL.val(clientWithOptions.getClientId().asLong()),
                                    DSL.val(0L),
                                    DSL.val(ZERO),
                                    DSL.val(ZERO),
                                    DSL.val(0L).cast(CLIENTS_OPTIONS.NEXT_PAY_DATE.getDataType()),
                                    DSL.val(clientWithOptions.isHideMarketRating() ? 1L : 0L),
                                    DSL.val(clientWithOptions.getFlags()),
                                    DSL.val(booleanToLong(clientWithOptions.isUsesQuasiCurrency()))))
                    .execute();

            if (!clientWithOptions.doDisableApi()) {
                insertDefaultClientsApiOptions(conf.dsl(), clientWithOptions.getClientId(), true);
            }

            if (addToFetchNds) {
                conf.dsl().insertInto(CLIENTS_TO_FETCH_NDS)
                        .columns(CLIENTS_TO_FETCH_NDS.CLIENT_ID)
                        .values(clientWithOptions.getClientId().asLong())
                        .execute();
            }
        });
    }

    public void insertDefaultClientsApiOptions(int shard, ClientId clientId, boolean updateExisting) {
        insertDefaultClientsApiOptions(ppcDslContextProvider.ppc(shard), clientId, updateExisting);
    }

    /**
     * Установить дефолтные параметры доступа к апи клиента
     */
    private void insertDefaultClientsApiOptions(DSLContext dslContext, ClientId clientId, boolean updateExisting) {
        var step = dslContext.insertInto(CLIENTS_API_OPTIONS)
                .columns(CLIENTS_API_OPTIONS.CLIENT_ID, CLIENTS_API_OPTIONS.API_ENABLED)
                .values(clientId.asLong(), ClientsApiOptionsApiEnabled.Default);
        if (updateExisting) {
            step.onDuplicateKeyUpdate()
                    .set(CLIENTS_API_OPTIONS.API_ENABLED, ClientsApiOptionsApiEnabled.Default)
                    .execute();
        } else {
            step.onDuplicateKeyIgnore().execute();
        }
    }

    /**
     * Вернуть параметра доступа к api для клиента
     */
    @Nonnull
    public ApiEnabled getClientApiOptionsEnabledStatus(int shard, ClientId clientId) {
        return getClientsApiOptionsEnabledStatus(shard, Collections.singleton(clientId)).get(clientId);
    }

    /**
     * Вернуть параметра доступа к api для клиентов
     */
    @Nonnull
    public Map<ClientId, ApiEnabled> getClientsApiOptionsEnabledStatus(int shard, Collection<ClientId> clientIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(CLIENTS_API_OPTIONS.CLIENT_ID, CLIENTS_API_OPTIONS.API_ENABLED)
                .from(CLIENTS_API_OPTIONS)
                .where(CLIENTS_API_OPTIONS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .fetchMap(s -> ClientId.fromLong(s.get(CLIENTS_API_OPTIONS.CLIENT_ID)),
                        s -> ApiUserRepository.convertApiEnabled(s.get(CLIENTS_API_OPTIONS.API_ENABLED)));
    }

    /**
     * Обновить набор таблиц с информацией по клиенту
     * <p>
     * Обновляются только те поля, которые помечены изменёнными в {@link AppliedChanges}
     *
     * @param appliedChanges коллекция применённых изменений
     */
    public void update(int shard, Collection<AppliedChanges<Client>> appliedChanges) {
        if (appliedChanges.isEmpty()) {
            return;
        }
        updateClients(shard, appliedChanges);
        updateClientsOptions(shard, appliedChanges);
    }

    private void updateClients(int shard, Collection<AppliedChanges<Client>> appliedChanges) {
        JooqUpdateBuilder<ClientsRecord, Client> updateBuilder =
                new JooqUpdateBuilder<>(CLIENTS.CLIENT_ID, appliedChanges);
        updateBuilder.processProperty(Client.PRIMARY_MANAGER_UID, CLIENTS.PRIMARY_MANAGER_UID);
        updateBuilder.processProperty(Client.IS_IDM_PRIMARY_MANAGER, CLIENTS.PRIMARY_MANAGER_SET_BY_IDM,
                RepositoryUtils::booleanToLong);
        updateBuilder.processProperty(Client.CONNECT_ORG_ID, CLIENTS.CONNECT_ORG_ID);
        updateBuilder.processProperty(Client.DELETED_REPS, CLIENTS.DELETED_REPS);

        updateBuilder.processProperty(Client.AGENCY_USER_ID, CLIENTS.AGENCY_UID);
        updateBuilder.processProperty(Client.AGENCY_CLIENT_ID, CLIENTS.AGENCY_CLIENT_ID);
        updateBuilder.processProperty(Client.ROLE, CLIENTS.ROLE, RbacRole::toSource);

        ppcDslContextProvider.ppc(shard)
                .update(CLIENTS)
                .set(updateBuilder.getValues())
                .where(CLIENTS.CLIENT_ID.in(updateBuilder.getChangedIds()))
                .execute();
    }

    private void updateClientsOptions(int shard, Collection<AppliedChanges<Client>> appliedChanges) {
        Set<Long> clientIds = listToSet(appliedChanges, c -> c.getModel().getId());
        ensureClientsOptionsExist(shard, clientIds);

        JooqUpdateBuilder<ClientsOptionsRecord, Client> ub =
                new JooqUpdateBuilder<>(CLIENTS_OPTIONS.CLIENT_ID, appliedChanges);
        ub.processProperty(Client.HIDE_MARKET_RATING, CLIENTS_OPTIONS.HIDE_MARKET_RATING, val -> val ? 1L : 0L);
        ub.processProperties(ClientOptionsMapping.CLIENTS_OPTION_FLAGS, CLIENTS_OPTIONS.CLIENT_FLAGS,
                ClientOptionsMapping::extractOptionFlags);

        ppcDslContextProvider.ppc(shard)
                .update(CLIENTS_OPTIONS)
                .set(ub.getValues())
                .where(CLIENTS_OPTIONS.CLIENT_ID.in(ub.getChangedIds()))
                .execute();
    }

    private void ensureClientsOptionsExist(int shard, Set<Long> clientIds) {
        Set<Long> existed = ppcDslContextProvider.ppc(shard)
                .select(CLIENTS_OPTIONS.CLIENT_ID)
                .from(CLIENTS_OPTIONS)
                .where(CLIENTS_OPTIONS.CLIENT_ID.in(clientIds))
                .fetchSet(CLIENTS_OPTIONS.CLIENT_ID);
        InsertValuesStep1<ClientsOptionsRecord, Long> insert =
                ppcDslContextProvider.ppc(shard).insertInto(CLIENTS_OPTIONS, CLIENTS_OPTIONS.CLIENT_ID);
        for (Long clientId : Sets.difference(clientIds, existed)) {
            insert = insert.values(clientId);
        }
        insert.onDuplicateKeyIgnore().execute();
    }

    /**
     * @return true, если клиент является представителем одного из бизнес-юнитов Яндекса.
     * В таблицу {@link ru.yandex.direct.dbschema.ppc.Tables#CLIENTS} не ходит, что иногда полезно
     * (т.к. для некоторых клиентов записи в этой таблице может не быть,
     * из-за чего {@link #get(int, Collection)} вернет null,
     * даже если в {@link ru.yandex.direct.dbschema.ppc.Tables#CLIENTS_OPTIONS} запись есть)
     */
    @Nullable
    public Boolean isBusinessUnitClient(int shard, ClientId clientId) {
        Long dbValue = ppcDslContextProvider.ppc(shard)
                .select(CLIENTS_OPTIONS.IS_BUSINESS_UNIT)
                .from(CLIENTS_OPTIONS)
                .where(CLIENTS_OPTIONS.CLIENT_ID.eq(clientId.asLong()))
                .fetchOne(CLIENTS_OPTIONS.IS_BUSINESS_UNIT);
        return RepositoryUtils.booleanFromLong(dbValue);
    }

    /**
     * @return true, если клиент заблокирован в балансе
     * В таблицу {@link ru.yandex.direct.dbschema.ppc.Tables#CLIENTS} не ходит, что иногда полезно
     * (т.к. для некоторых клиентов записи в этой таблице может не быть,
     * из-за чего {@link #get(int, Collection)} вернет null,
     * даже если в {@link ru.yandex.direct.dbschema.ppc.Tables#CLIENTS_OPTIONS} запись есть)
     */
    @Nullable
    public Boolean isBannedClient(int shard, ClientId clientId) {
        ClientsOptionsStatusbalancebanned dbValue = ppcDslContextProvider.ppc(shard)
                .select(CLIENTS_OPTIONS.STATUS_BALANCE_BANNED)
                .from(CLIENTS_OPTIONS)
                .where(CLIENTS_OPTIONS.CLIENT_ID.eq(clientId.asLong()))
                .fetchOne(CLIENTS_OPTIONS.STATUS_BALANCE_BANNED);
        return dbValue != null && dbValue == ClientsOptionsStatusbalancebanned.Yes;
    }

    /**
     * Прописать клиенты права
     */
    public void setPerms(int shard, ClientId clientId, Set<ClientPerm> perms) {
        ppcDslContextProvider.ppc(shard)
                .update(CLIENTS)
                .set(CLIENTS.PERMS, ClientPerm.format(perms))
                .where(CLIENTS.CLIENT_ID.eq(clientId.asLong()))
                .execute();
    }

    public void updateRole(int shard, ClientId clientId, RbacRole role, @Nullable RbacSubrole subrole) {
        ppcDslContextProvider.ppc(shard)
                .update(CLIENTS)
                .set(CLIENTS.ROLE, RbacRole.toSource(role))
                .set(CLIENTS.SUBROLE, RbacSubrole.toSource(subrole))
                .where(CLIENTS.CLIENT_ID.eq(clientId.asLong()))
                .execute();
    }

    public void updateChief(int shard, ClientId clientId, Long oldChiefUid, Long newChiefUid) {
        ppcDslContextProvider.ppc(shard).transaction(config -> {
            DSLContext txContext = DSL.using(config);

            txContext.update(USERS)
                    .set(USERS.REP_TYPE, UsersRepType.main)
                    .where(USERS.UID.eq(oldChiefUid))
                    .execute();

            txContext.update(USERS)
                    .set(USERS.REP_TYPE, UsersRepType.chief)
                    .where(USERS.UID.eq(newChiefUid))
                    .execute();

            txContext.update(CLIENTS)
                    .set(CLIENTS.CHIEF_UID, newChiefUid)
                    .where(CLIENTS.CLIENT_ID.eq(clientId.asLong()))
                    .execute();

        });
    }

    /**
     * По списку clientId агентств вернуть множество clientId их субклиентов.
     */
    public Set<ClientId> getSubclientIdsByAgencyClientIds(int shard, Collection<Long> agencyClientIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(CLIENTS.CLIENT_ID)
                .from(CLIENTS)
                .where(CLIENTS.AGENCY_CLIENT_ID.in(agencyClientIds))
                .fetchSet(record -> ClientId.fromLong(record.get(CLIENTS.CLIENT_ID)));
    }

    /**
     * Возвращает мапу clientId -> nullable primaryManagerUid
     */
    public Map<Long, Long> getPrimaryManagerUids(Collection<Long> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .stream()
                .map(e -> getPrimaryManagerUids(e.getKey(), e.getValue()))
                .flatMap(map -> map.entrySet().stream())
                .toMap(e -> e.getKey(), e -> e.getValue());
    }

    public Optional<ClientExperiment> getExperiment(int shard, Long clientId) {
        return ppcDslContextProvider.ppc(shard)
                .select(clientExperimentReader.getFieldsToRead())
                .from(CLIENTS_OPTIONS)
                .where(CLIENTS_OPTIONS.CLIENT_ID.eq(clientId).and(CLIENTS_OPTIONS.EXPERIMENT_ID.isNotNull()).and(CLIENTS_OPTIONS.SEGMENT_ID.isNotNull()))
                .fetchOptional(clientExperimentReader::fromDb);
    }

    public void saveClientExperiment(int shard, Long clientId, ClientExperiment experiment) {
        ppcDslContextProvider.ppc(shard)
                .update(CLIENTS_OPTIONS)
                .set(CLIENTS_OPTIONS.EXPERIMENT_ID, experiment.getExperimentId())
                .set(CLIENTS_OPTIONS.SEGMENT_ID, experiment.getSegmentId())
                .where(CLIENTS_OPTIONS.CLIENT_ID.eq(clientId))
                .execute();
    }

    private Map<Long, Long> getPrimaryManagerUids(int shard, Collection<Long> clientIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(CLIENTS.CLIENT_ID, CLIENTS.PRIMARY_MANAGER_UID)
                .from(CLIENTS)
                .where(CLIENTS.CLIENT_ID.in(clientIds))
                .fetch()
                .intoMap(CLIENTS.CLIENT_ID, CLIENTS.PRIMARY_MANAGER_UID);
    }

    public List<ClientAutoOverdraftInfo> getClientsAutoOverdraftInfo(int shard, Collection<ClientId> clientIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(clientAutoOverdraftInfoReader.getFieldsToRead())
                .from(CLIENTS_OPTIONS)
                .where(CLIENTS_OPTIONS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .fetch(clientAutoOverdraftInfoReader::fromDb);
    }

    public List<ClientDataForNds> getClientsDataForNds(int shard, Collection<ClientId> clientIds) {
        if (clientIds.isEmpty()) {
            return List.of();
        }
        return ppcDslContextProvider.ppc(shard)
                .select(clientDataForNdsReader.getFieldsToRead())
                .from(CLIENTS)
                .join(CLIENTS_OPTIONS)
                .on(CLIENTS_OPTIONS.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .where(CLIENTS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .fetch(clientDataForNdsReader::fromDb);
    }
}
