package ru.yandex.direct.core.entity.client.repository;

import java.util.Collection;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.client.model.AggregationType;
import ru.yandex.direct.core.entity.client.model.ClientSpent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.clientIdProperty;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_STAT;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class ClientSpentRepository {
    private final JooqMapperWithSupplier<ClientSpent> jooqMapper;

    private final DslContextProvider dslContextProvider;

    @Autowired
    public ClientSpentRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        jooqMapper = JooqMapperWithSupplierBuilder.builder(ClientSpent::new)
                .map(clientIdProperty(ClientSpent.CLIENT_ID, CLIENTS_STAT.CLIENT_ID))
                .map(convertibleProperty(ClientSpent.TYPE, CLIENTS_STAT.TYPE,
                        AggregationType::fromSource, AggregationType::toSource))
                .map(property(ClientSpent.ACTIVE28_DAYS_SUM, CLIENTS_STAT.ACTIVE_28DAYS_SUM_RUB))
                .map(property(ClientSpent.DAILY_SPENT_ESTIMATE, CLIENTS_STAT.DAILY_SPENT_RUB_ESTIMATE))
                .map(property(ClientSpent.REST_SUM, CLIENTS_STAT.SUM_REST_RUB))
                .map(property(ClientSpent.TOTAL_SUM, CLIENTS_STAT.TOTAL_SUM_RUB))
                .build();
    }

    public int replaceClientsSpending(int shard, Collection<ClientSpent> clientsSpending) {
        var insertHelper = new InsertHelper<>(dslContextProvider.ppc(shard), CLIENTS_STAT);
        insertHelper.addAll(jooqMapper, clientsSpending);
        insertHelper.onDuplicateKeyUpdate()
                .set(CLIENTS_STAT.ACTIVE_28DAYS_SUM_RUB, MySQLDSL.values(CLIENTS_STAT.ACTIVE_28DAYS_SUM_RUB))
                .set(CLIENTS_STAT.DAILY_SPENT_RUB_ESTIMATE, MySQLDSL.values(CLIENTS_STAT.DAILY_SPENT_RUB_ESTIMATE))
                .set(CLIENTS_STAT.SUM_REST_RUB, MySQLDSL.values(CLIENTS_STAT.SUM_REST_RUB))
                .set(CLIENTS_STAT.TOTAL_SUM_RUB, MySQLDSL.values(CLIENTS_STAT.TOTAL_SUM_RUB));

        return insertHelper.executeIfRecordsAdded();
    }

    public Optional<ClientSpent> getClientSpent(int shard, ClientId clientId) {
        final var result = dslContextProvider
                .ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(CLIENTS_STAT)
                .where(CLIENTS_STAT.CLIENT_ID.eq(clientId.asLong()))
                .fetch(jooqMapper::fromDb);

        if (result.size() == 0) {
            return Optional.empty();
        }

        return Optional.of(result.get(0));
    }
}
