package ru.yandex.direct.core.entity.client.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.client.model.ManagerHierarchyInfo;
import ru.yandex.direct.dbschema.ppc.tables.records.ManagerHierarchyRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.emptyList;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static ru.yandex.direct.dbschema.ppc.Tables.MANAGER_HIERARCHY;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class ManagerHierarchyRepository {

    private static final JooqReaderWithSupplier<ManagerHierarchyInfo> MANAGER_HIERARCHY_READER =
            createMapper();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public ManagerHierarchyRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    private static JooqReaderWithSupplier<ManagerHierarchyInfo> createMapper() {
        return JooqReaderWithSupplierBuilder.builder(ManagerHierarchyInfo::new)
                .readProperty(ManagerHierarchyInfo.ID, fromField(MANAGER_HIERARCHY.MANAGER_CLIENT_ID))
                .readProperty(ManagerHierarchyInfo.MANAGER_CLIENT_ID,
                        fromField(MANAGER_HIERARCHY.MANAGER_CLIENT_ID).by(ClientId::fromLong))
                .readProperty(ManagerHierarchyInfo.MANAGER_UID, fromField(MANAGER_HIERARCHY.MANAGER_UID))
                .readProperty(ManagerHierarchyInfo.CHIEFS_CLIENT_ID,
                        fromField(MANAGER_HIERARCHY.CHIEFS_CLIENT_ID)
                                .by(ManagerHierarchyRepository::clientIdsFromJson))
                .readProperty(ManagerHierarchyInfo.SUBORDINATES_CLIENT_ID,
                        fromField(MANAGER_HIERARCHY.SUBORDINATES_CLIENT_ID)
                                .by(ManagerHierarchyRepository::clientIdsFromJson))
                .readProperty(ManagerHierarchyInfo.SUBORDINATES_UID,
                        fromField(MANAGER_HIERARCHY.SUBORDINATES_UID)
                                .by(ManagerHierarchyRepository::idsFromJson))
                .build();
    }

    @Nonnull
    public List<ManagerHierarchyInfo> getManagerData(int shard, Collection<ClientId> clientIds) {
        return dslContextProvider.ppc(shard)
                .select(MANAGER_HIERARCHY_READER.getFieldsToRead())
                .from(MANAGER_HIERARCHY)
                .where(MANAGER_HIERARCHY.MANAGER_CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .fetch(MANAGER_HIERARCHY_READER::fromDb);
    }

    public void updateManagerData(int shard, Collection<AppliedChanges<ManagerHierarchyInfo>> appliedChanges) {
        JooqUpdateBuilder<ManagerHierarchyRecord, ManagerHierarchyInfo> ub =
                new JooqUpdateBuilder<>(MANAGER_HIERARCHY.MANAGER_CLIENT_ID, appliedChanges);

        ub.processProperty(ManagerHierarchyInfo.SUBORDINATES_CLIENT_ID, MANAGER_HIERARCHY.SUBORDINATES_CLIENT_ID,
                ManagerHierarchyRepository::clientIdsToJson);
        ub.processProperty(ManagerHierarchyInfo.SUBORDINATES_UID, MANAGER_HIERARCHY.SUBORDINATES_UID,
                ManagerHierarchyRepository::idsToJson);

        dslContextProvider.ppc(shard)
                .update(MANAGER_HIERARCHY)
                .set(ub.getValues())
                .where(MANAGER_HIERARCHY.MANAGER_CLIENT_ID.in(ub.getChangedIds()))
                .execute();
    }

    public void deleteManagerData(int shard, ClientId clientId) {
        dslContextProvider.ppc(shard)
                .deleteFrom(MANAGER_HIERARCHY)
                .where(MANAGER_HIERARCHY.MANAGER_CLIENT_ID.eq(clientId.asLong()))
                .execute();
    }

    @Nullable
    private static String idsToJson(@Nullable List<Long> ids) {
        return JsonUtils.toJsonCollectionEmptyToNull(ids);
    }

    private static List<Long> idsFromJson(@Nullable String idsJson) {
        return isEmpty(idsJson)
                ? emptyList()
                : JsonUtils.fromJson(idsJson, new TypeReference<List<Long>>() {
        });
    }

    @Nullable
    private static String clientIdsToJson(@Nullable List<ClientId> clientIds) {
        List<Long> ids = mapList(clientIds, ClientId::asLong);
        return idsToJson(ids);
    }

    private static List<ClientId> clientIdsFromJson(@Nullable String idsJson) {
        List<Long> ids = idsFromJson(idsJson);
        return mapList(ids, ClientId::fromLong);
    }
}
