package ru.yandex.direct.core.entity.client.service;

import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.agency.service.AgencyService;
import ru.yandex.direct.core.entity.application.model.AgencyOptions;
import ru.yandex.direct.core.entity.client.exception.BalanceErrorException;
import ru.yandex.direct.core.entity.client.exception.CantRegisterClientException;
import ru.yandex.direct.core.entity.client.exception.PassportErrorException;
import ru.yandex.direct.core.entity.client.model.AddAgencyClientRequest;
import ru.yandex.direct.core.entity.client.model.AddAgencyClientResponse;
import ru.yandex.direct.core.entity.client.model.ClientWithOptions;
import ru.yandex.direct.core.entity.client.service.validation.ClientDefectTranslations;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.security.MethodNotAllowedException;
import ru.yandex.direct.core.service.RequestInfoProvider;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.core.service.integration.balance.RegisterClientRequest;
import ru.yandex.direct.core.service.integration.balance.RegisterClientResult;
import ru.yandex.direct.core.service.integration.passport.PassportService;
import ru.yandex.direct.core.service.integration.passport.RegisterUserRequest;
import ru.yandex.direct.core.service.integration.passport.RegisterUserResult;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.PassportUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для добавления новых subclient-ов в систему
 * <p>
 * Управляет хождением во внешние системы + непосредственно сохранением subclient-а
 * в базе Direct-а
 */
@ParametersAreNonnullByDefault
@Service
public class AddAgencyClientService {
    private final UserService userService;
    private final AddAgencyClientValidator validator;
    private final PassportService passportService;
    private final BalanceService balanceService;
    private final ClientService clientService;
    private final AgencyService agencyService;
    private final AgencyLimitedRepsService agencyLimitedRepsService;
    private final RbacService rbacService;
    private final FeatureService featureService;

    @Autowired
    public AddAgencyClientService(
            UserService userService,
            AddAgencyClientValidator validator,
            PassportService passportService,
            BalanceService balanceService,
            ClientService clientService,
            AgencyService agencyService,
            AgencyLimitedRepsService agencyLimitedRepsService,
            RbacService rbacService,
            FeatureService featureService) {
        this.userService = Objects.requireNonNull(userService, "userService");
        this.validator = Objects.requireNonNull(validator, "validator");
        this.passportService = Objects.requireNonNull(passportService, "passportService");
        this.balanceService = Objects.requireNonNull(balanceService, "balanceService");
        this.clientService = Objects.requireNonNull(clientService, "clientService");
        this.agencyService = Objects.requireNonNull(agencyService, "agencyService");
        this.agencyLimitedRepsService = Objects.requireNonNull(agencyLimitedRepsService, "agencyLimitedRepsService");
        this.rbacService = Objects.requireNonNull(rbacService, "rbacService");
        this.featureService = Objects.requireNonNull(featureService, "featureService");
    }

    public Result<AddAgencyClientResponse> processRequest(
            RequestInfoProvider requestInfoProvider,
            UidAndClientId operator,
            UidAndClientId agency,
            UidAndClientId agencyMainChief,
            AddAgencyClientRequest request) {
        if (!userService.canAgencyCreateSubclient(agencyMainChief)) {
            throw new MethodNotAllowedException(ClientDefectTranslations.INSTANCE.forbiddenCreateSubclientDetails());
        }

        AgencyOptions agencyOpt = agencyService.getAgencyOptions(agency.getClientId());

        if (!request.getSharedAccountEnabled().isPresent()) {
            request.setSharedAccountEnabled(agencyOpt.isDefaultClientsWithWallet());
        }

        ValidationResult<AddAgencyClientRequest, Defect> validationResult = validator.validate(
                agency,
                agencyOpt,
                request);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }

        ResponseBuilder builder = new ResponseBuilder(request);

        // Регистрируем клиента в паспорте
        RegisterUserResult registerUserResult = passportService.registerUser(
                requestInfoProvider.getRequestId(),
                builder.toPassportRegisterUserRequest(),
                requestInfoProvider.getRemoteIpAddress());
        if (!registerUserResult.getStatus().isSuccess()) {
            throw PassportErrorException.newInstance(registerUserResult.getStatus());
        }
        builder.apply(registerUserResult);

        // Регистрируем клиента в балансе
        RegisterClientResult registerClientResult = balanceService.registerNewClient(
                operator.getUid(), agency, builder.toBalanceRegisterClientRequest());
        if (!registerClientResult.getStatus().isSuccess()) {
            throw BalanceErrorException.newInstance();
        }
        builder.apply(registerClientResult);

        // Список представителей с доступом к клиенту
        Set<UidAndClientId> agencyReps = new HashSet<>(2);
        agencyReps.add(agency);
        Set<String> agencyEnabledFeatures = featureService.getEnabledForClientId(agency.getClientId());
        boolean isNewLimRepSchema = agencyEnabledFeatures.contains(FeatureName.NEW_LIM_REP_SCHEMA.getName());
        if (isNewLimRepSchema) {
            agencyReps.addAll(rbacService
                            .getChiefOfGroupForLimitedAgencyReps(agencyReps)
                            .values());
        }

        // Все данные успешно собраны, регистрируем клиента в базе direct-а
        boolean isSuccess = clientService.registerSubclient(
                operator.getUid(),
                agencyReps,
                builder.toSubclient(),
                agencyEnabledFeatures);
        if (!isSuccess) {
            throw new CantRegisterClientException();
        }

        agencyLimitedRepsService.updateAgencyLimitedRepsSubclientsInBalance(
                operator.getUid(), mapList(agencyReps, UidAndClientId::getUid));

        return Result.successful(builder.build());
    }

    private static final class ResponseBuilder {
        private final AddAgencyClientRequest source;

        private String login;
        private String password;
        private Long uid;
        private String email;
        private ClientId clientId;

        ResponseBuilder(AddAgencyClientRequest source) {
            this.source = source;
            this.login = PassportUtils.normalizeLogin(source.getLogin());
        }

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }

        @Nonnull
        RegisterUserRequest toPassportRegisterUserRequest() {
            return new RegisterUserRequest()
                    .withLogin(login)
                    .withFirstName(source.getFirstName())
                    .withLastName(source.getLastName());
        }

        public void apply(RegisterUserResult result) {
            uid = result.getUid();
            email = result.getEmail();
            password = result.getPassword();
        }

        @Nonnull
        RegisterClientRequest toBalanceRegisterClientRequest() {
            return new RegisterClientRequest()
                    .withClientUid(uid)
                    .withName(getName())
                    .withEmail(email)
                    .withCurrency(source.getCurrency())
                    .withAgency(true);
        }

        public void apply(RegisterClientResult result) {
            clientId = result.getClientId();
        }

        @Nonnull
        ClientWithOptions toSubclient() {
            return new ClientWithOptions()
                    .withRole(RbacRole.CLIENT)
                    .withLogin(login)
                    .withUid(uid)
                    .withClientId(clientId)
                    .withEmail(source.getNotificationEmail())
                    .withName(getName())
                    .withCurrency(source.getCurrency())
                    .withNotificationEmail(source.getNotificationEmail())
                    .withNotificationLang(source.getNotificationLang())
                    .withSendNews(source.isSendNews())
                    .withSendAccNews(source.isSendAccNews())
                    .withSendWarn(source.isSendWarn())
                    .withHideMarketRating(source.isHideMarketRating())
                    .withNoTextAutocorrection(source.isNoTextAutocorrection())
                    .withAllowEditCampaigns(source.getAllowEditCampaigns())
                    .withAllowImportXls(source.getAllowImportXls())
                    .withAllowTransferMoney(source.getAllowTransferMoney())
                    .withSharedAccountDisabled(!source.getSharedAccountEnabled().orElse(true))
                    .withUsesQuasiCurrency(source.getCurrency() == CurrencyCode.KZT);
        }

        @Nonnull
        public AddAgencyClientResponse build() {
            return new AddAgencyClientResponse()
                    .withLogin(source.getLogin())
                    .withEmail(email)
                    .withPassword(password)
                    .withClientId(clientId);
        }

        @Nonnull
        private String getName() {
            return source.getFirstName() + " " + source.getLastName();
        }
    }
}
