package ru.yandex.direct.core.entity.client.service;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.application.model.AgencyOptions;
import ru.yandex.direct.core.entity.client.model.AddAgencyClientRequest;
import ru.yandex.direct.core.entity.client.service.validation.AgencyClientCurrencyValidatorFactory;
import ru.yandex.direct.core.entity.client.service.validation.NotificationEmailValidator;
import ru.yandex.direct.core.entity.user.validator.FirstLastNameValidator;
import ru.yandex.direct.core.entity.user.validator.LoginValidator;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.client.service.AddAgencyClientValidator.DefectDefinitions.inconsistentStateAllowEditCampaignAndAllowImportXls;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

/**
 * Валидатор запросов на создание клиентов агентств
 */
@Service
@Lazy
@ParametersAreNonnullByDefault
public class AddAgencyClientValidator {
    private final LoginValidator loginValidator;
    private final FirstLastNameValidator firstNameValidator;
    private final FirstLastNameValidator lastNameValidator;
    private final NotificationEmailValidator notificationEmailValidator;
    private final AgencyClientCurrencyValidatorFactory currencyValidatorFactory;

    @Autowired
    AddAgencyClientValidator(
            LoginValidator loginValidator,
            FirstLastNameValidator firstNameValidator,
            FirstLastNameValidator lastNameValidator,
            NotificationEmailValidator notificationEmailValidator,
            AgencyClientCurrencyValidatorFactory currencyValidatorFactory) {
        this.loginValidator = Objects.requireNonNull(
                loginValidator, "loginValidator");
        this.firstNameValidator = Objects.requireNonNull(
                firstNameValidator, "firstNameValidator");
        this.lastNameValidator = Objects.requireNonNull(
                lastNameValidator, "lastNameValidator");
        this.notificationEmailValidator = Objects.requireNonNull(
                notificationEmailValidator, "notificationEmailValidator");
        this.currencyValidatorFactory = Objects.requireNonNull(
                currencyValidatorFactory, "currencyValidatorFactory");
    }

    /**
     * Объединение из Validate.pm::_validate_create_new_subclient и Direct::Validation::Client::check_country_and_currency
     */
    public ValidationResult<AddAgencyClientRequest, Defect> validate(
            UidAndClientId agency,
            AgencyOptions agencyOpt, AddAgencyClientRequest request) {
        ModelItemValidationBuilder<AddAgencyClientRequest> vb = ModelItemValidationBuilder.of(request);

        vb.item(AddAgencyClientRequest.LOGIN)
                .check(notNull())
                .checkBy(loginValidator, When.isValid());

        vb.item(AddAgencyClientRequest.FIRST_NAME)
                .check(notNull())
                .checkBy(firstNameValidator, When.isValid());

        vb.item(AddAgencyClientRequest.LAST_NAME)
                .check(notNull())
                .checkBy(lastNameValidator, When.isValid());

        vb.item(AddAgencyClientRequest.NOTIFICATION_EMAIL)
                .check(notNull())
                .checkBy(notificationEmailValidator, When.isValid());

        vb.item(AddAgencyClientRequest.CURRENCY)
                .check(notNull())
                .checkBy(
                        currencyValidatorFactory.newInstance(agency.getClientId()),
                        // Проверяем валюту, только если все остальное ок, так как проверка достаточно тяжелая
                        When.isValid());

        // если запрещёно редактирование кампаний, то и импорт XLS должен быть запрещён, иначе Rbac выдаст ошибку
        vb.item(AddAgencyClientRequest.ALLOW_IMPORT_XLS)
                .check(Constraint.fromPredicate(
                        x -> !x,
                        inconsistentStateAllowEditCampaignAndAllowImportXls()
                ), When.isTrue(request.getAllowEditCampaigns() == Boolean.FALSE));

        vb.item(AddAgencyClientRequest.SHARED_ACCOUNT_ENABLED)
                .check(validateSharedAccountEnabled(agencyOpt),
                        When.isTrue(request.getSharedAccountEnabled().isPresent()));

        return vb.getResult();
    }

    private Constraint<Optional<Boolean>, Defect> validateSharedAccountEnabled(AgencyOptions agencyOpt) {
        return v -> {
            //Значение SHARED_ACCOUNT_ENABLED = No разрешено только если allow_clients_without_wallet != 0
            if (v.orElse(Boolean.TRUE)) {
                return null;
            }
            if (!agencyOpt.isAllowClientsWithoutWallet()) {
                return noRights();
            }
            return null;
        };
    }

    public enum VoidDefectIds implements DefectId<Void> {
        INCONSISTENT_STATE_ALLOW_EDIT_CAMPAIGN_AND_ALLOW_IMPORT_XLS,
    }

    public static class DefectDefinitions {
        public static Defect<Void> inconsistentStateAllowEditCampaignAndAllowImportXls() {
            return new Defect<>(VoidDefectIds.INCONSISTENT_STATE_ALLOW_EDIT_CAMPAIGN_AND_ALLOW_IMPORT_XLS);
        }
    }
}
