package ru.yandex.direct.core.entity.client.service;

import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.WalletService;
import ru.yandex.direct.core.entity.client.exception.BalanceErrorException;
import ru.yandex.direct.core.entity.client.exception.CantRegisterClientException;
import ru.yandex.direct.core.entity.client.model.ClientWithOptions;
import ru.yandex.direct.core.entity.client.service.validation.AddClientValidationService;
import ru.yandex.direct.core.entity.user.service.BlackboxUserService;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.core.service.integration.balance.RegisterClientRequest;
import ru.yandex.direct.core.service.integration.balance.RegisterClientResult;
import ru.yandex.direct.core.service.integration.balance.RegisterClientStatus;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.LoginOrUid;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.PassportUtils;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;

import static java.util.Objects.requireNonNull;
import static ru.yandex.direct.core.entity.user.service.BlackboxUserService.getValidEmail;
import static ru.yandex.direct.i18n.Language.EN;
import static ru.yandex.direct.i18n.Language.RU;
import static ru.yandex.direct.i18n.Language.TR;
import static ru.yandex.direct.i18n.Language.UK;
import static ru.yandex.direct.regions.Region.RUSSIA_REGION_ID;
import static ru.yandex.direct.regions.Region.TURKEY_REGION_ID;
import static ru.yandex.direct.regions.Region.UKRAINE_REGION_ID;

/**
 * Сервис для добавления новых клиентов в систему
 * <p>
 * Управляет хождением во внешние системы + непосредственно сохранением клиента
 * в базе Direct-а
 */
@ParametersAreNonnullByDefault
@Service
public class AddClientService {

    private static final Logger logger = LoggerFactory.getLogger(AddClientService.class);

    private static final Map<Long, Language> LANGUAGE_BY_REGION_ID = ImmutableMap.of(
            RUSSIA_REGION_ID, RU,
            UKRAINE_REGION_ID, UK,
            TURKEY_REGION_ID, TR
    );

    private final BalanceService balanceService;
    private final ClientService clientService;
    private final WalletService walletService;
    private final GeoTreeFactory geoTreeFactory;
    private final AddClientValidationService addClientValidationService;
    private final BlackboxUserService blackboxUserService;

    @Autowired
    public AddClientService(
            BalanceService balanceService,
            ClientService clientService,
            WalletService walletService,
            GeoTreeFactory geoTreeFactory,
            AddClientValidationService addClientValidationService,
            BlackboxUserService blackboxUserService) {
        this.balanceService = requireNonNull(balanceService, "balanceService");
        this.clientService = requireNonNull(clientService, "clientService");
        this.walletService = requireNonNull(walletService, "walletService");
        this.geoTreeFactory = geoTreeFactory;
        this.addClientValidationService = addClientValidationService;
        this.blackboxUserService = blackboxUserService;
    }

    /**
     * Создает нового клиента и возвращает его ClientID и паспортный UID.
     *
     * @param loginOrUid Логин или uid
     * @param fio        ФИО
     * @param region     Код региона (страны)
     * @param currency   Валюта
     * @param rbacRole   Роль
     */
    public Result<UidAndClientId> processRequest(
            LoginOrUid loginOrUid, String fio, long region, CurrencyCode currency, RbacRole rbacRole) {
        return processRequest(loginOrUid, fio, region, currency, rbacRole, AddClientOptions.defaultOptions());
    }

    /**
     * Создает нового клиента и возвращает его ClientID и паспортный UID.
     *
     * @param loginOrUid       Логин или uid
     * @param fio              ФИО
     * @param region           Код региона (страны)
     * @param currency         Валюта
     * @param rbacRole         Роль
     * @param addClientOptions параметры создания клиента
     */
    public Result<UidAndClientId> processRequest(
            LoginOrUid loginOrUid, String fio, long region, CurrencyCode currency,
            RbacRole rbacRole, AddClientOptions addClientOptions) {
        ValidationResult<Object, Defect> vr =
                addClientValidationService.validateAddClientRequest(loginOrUid, fio, region, currency);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }
        Result<BlackboxCorrectResponse> blackboxResult = blackboxUserService.getAndCheckUser(loginOrUid);
        return createClient(blackboxResult, loginOrUid, fio, region, currency, rbacRole, addClientOptions);
    }

    /**
     * Создает нового клиента и возвращает его ClientID и паспортный UID.
     *
     * @param loginOrUid       Логин или uid
     * @param fio              ФИО
     * @param region           Код региона (страны)
     * @param currency         Валюта
     * @param rbacRole         Роль
     * @param addClientOptions параметры создания клиента
     */
    public Result<UidAndClientId> processRequest(
            Result<BlackboxCorrectResponse> blackboxResult,
            LoginOrUid loginOrUid, String fio, long region, CurrencyCode currency,
            RbacRole rbacRole, AddClientOptions addClientOptions) {
        ValidationResult<Object, Defect> vr =
                addClientValidationService.validateAddClientRequest(loginOrUid, fio, region, currency);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }
        return createClient(blackboxResult, loginOrUid, fio, region, currency, rbacRole, addClientOptions);
    }

    private Result<UidAndClientId> createClient(Result<BlackboxCorrectResponse> blackboxResult,
                                                LoginOrUid loginOrUid, String fio, long region,
                                                CurrencyCode currency, RbacRole rbacRole,
                                                AddClientOptions addClientOptions) {
        if (!blackboxResult.isSuccessful()) {
            return Result.broken(blackboxResult.getValidationResult());
        }
        BlackboxCorrectResponse user = blackboxResult.getResult();

        Long country = getGeoTree().upRegionToType(region, Region.REGION_TYPE_COUNTRY);
        ResponseBuilder builder = new ResponseBuilder(user, currency, fio, country, rbacRole);

        // Регистрируем клиента в балансе
        RegisterClientResult registerClientResult = balanceService.registerNewClient(builder.getUid(), null,
                builder.toBalanceRegisterClientRequest(), addClientOptions.getUseExistingIfRequired());
        if (!registerClientResult.getStatus().isSuccess()) {
            if (addClientOptions.getUseExistingIfRequired() &&
                    registerClientResult.getStatus() == RegisterClientStatus.INVALID_CURRENCY) {
                return Result.broken(ValidationResult.failed(loginOrUid.get(), CommonDefects.invalidValue()));
            }
            throw BalanceErrorException.newInstance();
        }
        builder.apply(registerClientResult);

        // Все данные успешно собраны, регистрируем клиента в базе direct-а
        ClientWithOptions clientWithOptions = builder.toClientWithOptions(
                addClientOptions.getPaymentBeforeModerationAllowed(),
                addClientOptions.getDisableApi());
        boolean isSuccess = clientService.registerClient(clientWithOptions);

        if (!isSuccess) {
            throw new CantRegisterClientException();
        }

        UidAndClientId uidAndClientId = builder.build();
        if (addClientOptions.isWithWallet()) {
            walletService.createWalletForNewClient(uidAndClientId.getClientId(), uidAndClientId.getUid());
        }

        return Result.successful(uidAndClientId);
    }

    private GeoTree getGeoTree() {
        return geoTreeFactory.getGlobalGeoTree();
    }

    private static final class ResponseBuilder {
        private final String login;
        private final long uid;
        private final RbacRole rbacRole;
        private final String email;
        private final CurrencyCode currencyCode;
        private final String name;
        private final Language language;
        private final Long countryRegionId;

        private ClientId clientId;

        ResponseBuilder(BlackboxCorrectResponse source, CurrencyCode currencyCode, String fio,
                        Long country, RbacRole rbacRole) {
            this.login = PassportUtils.normalizeLogin(source.getLogin().get());
            this.currencyCode = currencyCode;
            this.email = getValidEmail(source);
            Preconditions.checkState(Predicates.validEmail().test(this.email));

            this.uid = source.getUid().get().getUid();
            this.rbacRole = rbacRole;
            this.name = fio;
            this.countryRegionId = country;
            this.language = LANGUAGE_BY_REGION_ID.getOrDefault(country, EN);
        }

        public long getUid() {
            return uid;
        }

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }

        @Nonnull
        RegisterClientRequest toBalanceRegisterClientRequest() {
            return new RegisterClientRequest()
                    .withClientUid(uid)
                    .withName(name)
                    .withEmail(email)
                    .withCurrency(currencyCode)
                    .withAgency(true)
                    .withCountryRegionId(countryRegionId);
        }

        public void apply(RegisterClientResult result) {
            clientId = result.getClientId();
        }

        @Nonnull
        ClientWithOptions toClientWithOptions(boolean paymentBeforeModerationAllowed,
                                              boolean disableApi) {
            return new ClientWithOptions()
                    .withRole(rbacRole)
                    .withLogin(login)
                    .withUid(uid)
                    .withClientId(clientId)
                    .withEmail(email)
                    .withName(name)
                    .withCurrency(currencyCode)
                    .withCountryRegionId(countryRegionId)
                    .withNotificationEmail(email)
                    .withNotificationLang(language)
                    .withSendNews(true)
                    .withSendAccNews(true)
                    .withSendWarn(true)
                    .withHideMarketRating(false)
                    .withNoTextAutocorrection(false)
                    .withAllowEditCampaigns(true)
                    .withAllowImportXls(true)
                    .withAllowTransferMoney(true)
                    .withSharedAccountDisabled(false)
                    .withUsesQuasiCurrency(currencyCode == CurrencyCode.KZT)
                    .withPaymentBeforeModeration(paymentBeforeModerationAllowed)
                    .withDisableApi(disableApi);
        }

        @Nonnull
        public UidAndClientId build() {
            return UidAndClientId.of(uid, clientId);
        }
    }
}
