package ru.yandex.direct.core.entity.client.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.AgencyClientRelation;
import ru.yandex.direct.core.entity.client.repository.AgencyClientRelationRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис управляющий отношениями между агенствами и клиентами
 */
@Service
@ParametersAreNonnullByDefault
public class AgencyClientRelationService {
    private final AgencyClientRelationRepository agencyClientRelationRepository;
    private final ShardHelper shardHelper;
    private final ClientRepository clientRepository;

    @Autowired
    public AgencyClientRelationService(AgencyClientRelationRepository agencyClientRelationRepository,
                                       ShardHelper shardHelper, ClientRepository clientRepository) {
        this.agencyClientRelationRepository = agencyClientRelationRepository;
        this.shardHelper = shardHelper;
        this.clientRepository = clientRepository;
    }

    static boolean canAgencyBindClient(
            ClientId agencyId, List<AgencyClientRelation> relations, boolean allowedToCreateCamps) {
        Optional<AgencyClientRelation> agencyRelation = relations.stream()
                .filter(r -> Objects.equals(r.getAgencyClientId(), agencyId))
                .findFirst();
        // Если клиент привязан к данному агенству или был привязан к данному агенству, то смотрим
        // только на поле isBinded. Т.е если он был отвязан от данного агенства, то его уже нельзя
        // привязывать к нему
        if (agencyRelation.isPresent()) {
            return agencyRelation.get().getBinded();
        } else {
            // Иначе мы должны только проверить, что клиент не привязан ни к одну из других агенств в настоящий
            // момент, хотя мог быть ранее привязан к какому-либо другому агенству и впоследствии был отвязан от него
            return !(relations.stream().anyMatch(AgencyClientRelation::getBinded) || allowedToCreateCamps);
        }
    }

    /**
     * Получить id неархивных субклиентов агенства из переданого списка id субклиентов
     *
     * @param agencyChiefClientId главный представитель агенства
     * @param clientIds           коллекция id проверяемых субклиентов
     * @return множество id неархивных субклиентов
     */
    @Nonnull
    public Set<Long> getUnArchivedAgencyClients(Long agencyChiefClientId, Collection<Long> clientIds) {
        Set<Long> uniqueClientIds = new HashSet<>(clientIds);
        return shardHelper.groupByShard(uniqueClientIds, ShardKey.CLIENT_ID).stream()
                .flatMapKeyValue(
                        (shard, ids) -> agencyClientRelationRepository.getUnarchivedAgencyClientsIds(
                                shard, agencyChiefClientId, ids).stream())
                .toSet();
    }

    public boolean isUnArchivedAgencyClient(@Nonnull Long agencyChiefClientId, @Nonnull Long clientId) {
        return getUnArchivedAgencyClients(agencyChiefClientId, singletonList(clientId)).contains(clientId);
    }

    @Nonnull
    public Collection<AgencyClientRelation> massGetByClients(Collection<ClientId> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong).stream()
                .map(e -> agencyClientRelationRepository.getByClients(e.getKey(), e.getValue()))
                .toFlatList(Function.identity());
    }

    public List<AgencyClientRelation> getUnarchivedBindedAgencies(Collection<ClientId> clientIds) {
        return massGetByClients(clientIds).stream()
                .filter(AgencyClientRelation::getBinded)
                .filter(v -> !v.getArchived())
                .collect(Collectors.toList());
    }

    /**
     * Вернуть список только доступных для привязки к агенству клиентов
     *
     * @param agencyId  Id агенства
     * @param clientIds Список Id-ков клиентов
     * @see #canAgencyBindClient
     */
    public Set<ClientId> getAllowableToBindClients(ClientId agencyId, Collection<ClientId> clientIds) {
            if (clientIds.isEmpty()) {
            return emptySet();
        }

        List<Long> longClientIds = mapList(clientIds, ClientId::asLong);
        Map<ClientId, List<AgencyClientRelation>> clientRelations =
                shardHelper.groupByShard(longClientIds, ShardKey.CLIENT_ID)
                        .stream()
                        .mapValues(ids -> mapList(ids, ClientId::fromLong))
                        .flatMapKeyValue(
                                (shard, clids) -> agencyClientRelationRepository.getByClients(shard, clids)
                                        .stream())
                        .groupingBy(AgencyClientRelation::getClientClientId);

        Set<ClientId> superSubclients = shardHelper.groupByShard(longClientIds, ShardKey.CLIENT_ID)
                .stream()
                .mapValues(ids -> mapList(ids, ClientId::fromLong))
                .flatMapKeyValue(
                        (shard, clids) -> clientRepository.getSubclientsAllowedToCreateCamps(shard, clids)
                                .stream())
                .toSet();

        return clientIds.stream()
                .filter(
                        clid -> canAgencyBindClient(
                                agencyId,
                                clientRelations.getOrDefault(clid, Collections.emptyList()),
                                superSubclients.contains(clid)))
                .collect(toSet());
    }

    /**
     * Аналог get_allow_agency_bind_client
     * <p>
     * проверяем может ли агентство обслуживать клиента:
     * либо должен быть доступ в agency_client_relations
     * либо у клиента не должно быть других агентств кроме текущего (и не должно быть возможности создавать самоходные кампании)
     * </p>
     *
     * @param agencyId Id агенства
     * @param clientId Id клиента
     * @return true, если может, иначе false
     */
    public boolean canAgencyBindClient(ClientId agencyId, ClientId clientId) {
        return getAllowableToBindClients(agencyId, singleton(clientId)).contains(clientId);
    }

    /**
     * Получить id агентств, в которых обслуживание указанного субклиента завершено
     *
     * @param clientId id субклиента
     * @return множество id агентств
     */
    @Nonnull
    public Set<ClientId> getUnbindedAgencies(ClientId clientId) {
        return getUnbindedAgencies(singletonList(clientId)).getOrDefault(clientId, emptySet());
    }

    /**
     * Получить id агентств, в которых обслуживание указанных субклиентов завершено
     *
     * @param clientIds коллекция id проверяемых субклиентов
     * @return множество id агентств
     */

    public Map<ClientId, Set<ClientId>> getUnbindedAgencies(Collection<ClientId> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong).stream()
                .map(e -> agencyClientRelationRepository.getByClients(e.getKey(), e.getValue()))
                .flatMap(Collection::stream)
                .filter(e -> !e.getBinded())
                .mapToEntry(AgencyClientRelation::getClientClientId, AgencyClientRelation::getAgencyClientId)
                .grouping(toSet());
    }

    /**
     * Привязать заданных клиентов к заданном агенству
     *
     * @param agencyId  Id агенства
     * @param clientIds Список Id-ков клиентов
     */
    public void bindClients(ClientId agencyId, Set<ClientId> clientIds) {
        // Возможно следует дать возможность настраивать размер chunk-а. Пока hardcode-им
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .chunkedBy(100)
                .forEach((shard, clids) -> agencyClientRelationRepository.bindClients(
                        shard, agencyId, clids));
    }

    /**
     * Получить связки клиентов агентства(в том числе и архивные)
     * Поиск по всем шардам
     *
     * @param agencyId clientId агентства
     */
    public List<AgencyClientRelation> getAgencyClients(ClientId agencyId) {
        List<AgencyClientRelation> result = new ArrayList<>();
        shardHelper.forEachShard(shard -> {
            result.addAll(agencyClientRelationRepository.getByAgencyId(shard, agencyId));
        });
        return result;
    }
}
