package ru.yandex.direct.core.entity.client.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.model.Representative;

import static ru.yandex.direct.feature.FeatureName.BALANCE_LIMITED_AGENCY_REP;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Service
public class AgencyLimitedRepsService {
    private final BalanceService balanceService;
    private final FeatureService featureService;
    private final RbacService rbacService;

    @Autowired
    public AgencyLimitedRepsService(
            BalanceService balanceService,
            FeatureService featureService,
            RbacService rbacService) {
        this.balanceService = Objects.requireNonNull(balanceService, "balanceService");
        this.featureService = Objects.requireNonNull(featureService, "featureService");
        this.rbacService = Objects.requireNonNull(rbacService, "rbacService");
    }

    /**
     * Отправить в баланс список доступных представителю агентства субклиентов агентства.
     * Нужно для того, чтобы ограниченные представители видели в балансе только своих клиентов
     * Неограниченные представители видят всех субклиентов агентства
     */
    public void updateAgencyLimitedRepsSubclientsInBalance(Long operatorUid, Collection<Long> agencyUids) {
        if (agencyUids.isEmpty()) {
            return;
        }
        // находим ограниченных представителей
        List<Representative> limitedRepresentatives = rbacService.getAgencyLimitedRepresentatives(agencyUids);
        Set<ClientId> agencyClientIds =
                StreamEx.of(limitedRepresentatives).map(Representative::getClientId).toSet();

        // если фича для агентства выключена, то не делаем запрос в баланс
        Map<ClientId, Boolean> isFeatureEnabled =
                featureService.isEnabledForClientIds(agencyClientIds, BALANCE_LIMITED_AGENCY_REP.getName());

        List<Representative> filteredLimitedReps = filterList(limitedRepresentatives,
                r -> Boolean.TRUE.equals(isFeatureEnabled.get(r.getClientId())));
        List<Long> filteredRepUids = mapList(filteredLimitedReps, Representative::getUserId);

        // находим субклиентов агентства, к которым представители имеют доступ
        Map<Long, List<Long>> subclientIdsByRepUids = rbacService.getSubclientsOfAgencyReps(filteredRepUids);

        for (Representative representative : filteredLimitedReps) {

            UidAndClientId repUidAndClientId =
                    UidAndClientId.of(representative.getUserId(), representative.getClientId());
            List<Long> subclientIds = subclientIdsByRepUids.get(representative.getUserId());

            if (!CollectionUtils.isEmpty(subclientIds)) {
                balanceService.setAgencyLimitedRepSubclients(
                        operatorUid, repUidAndClientId, subclientIds);
            }
            // TODO: для ограниченных представителей без клиентов передавать в баланс [null]
            // https://st.yandex-team.ru/BALANCE-28458#1529074162000
        }
    }
}
