package ru.yandex.direct.core.entity.client.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.ClientBrand;
import ru.yandex.direct.core.entity.client.repository.ClientBrandsRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

@Service
@ParametersAreNonnullByDefault
public class ClientBrandsService {
    private static final int INSERT_CHUNK_SIZE = 1_000;
    private static final int DELETE_CHUNK_SIZE = 1_000;

    private final ShardHelper shardHelper;
    private final ClientBrandsRepository clientBrandsRepository;

    @Autowired
    public ClientBrandsService(ShardHelper shardHelper, ClientBrandsRepository clientBrandsRepository) {
        this.shardHelper = shardHelper;
        this.clientBrandsRepository = clientBrandsRepository;
    }

    /**
     * Записать в ppc.client_brands бренды, предварительно удалив все другие бренды клиентов, упоминающихся в
     * переданном списке на запись.
     * <p>
     * Запись делается чанками по {@value INSERT_CHUNK_SIZE}
     *
     * @return количество вставленных в таблицу строк
     */
    public int replaceClientBrands(Collection<ClientBrand> clientBrands) {
        return shardHelper
                .groupByShard(clientBrands, ShardKey.CLIENT_ID, ClientBrand::getClientId)
                .chunkedBy(INSERT_CHUNK_SIZE).stream()
                .mapToInt(el -> clientBrandsRepository.replaceClientBrands(el.getKey(), el.getValue()))
                .sum();
    }

    /**
     * Получить общее количество записей в таблице ppc.client_brands во всех шардах
     */
    public int getClientBrandsCount() {
        return shardHelper.dbShards().stream()
                .mapToInt(clientBrandsRepository::getClientBrandsCount).sum();
    }


    /**
     * Получить из таблицы ppc.client_brands во всех шардах список из всех записей, синхронизированные раньше заданного времени
     */
    public List<ClientBrand> getEntriesOlderThanDateTime(LocalDateTime borderDateTime) {
        return shardHelper.dbShards().stream()
                .map(shard -> clientBrandsRepository.getEntriesOlderThanDateTime(shard, borderDateTime))
                .flatMap(Collection::stream).collect(Collectors.toList());
    }

    /**
     * Удалить из таблицы ppc.client_brands во всех шардах все записи, синхронизированные раньше заданного времени
     * <p>
     * Удаление в каждом шарде делается чанками по {@value DELETE_CHUNK_SIZE}
     *
     * @return количество удаленных из таблицы строк
     */
    public int deleteEntriesOlderThanDateTime(LocalDateTime borderDateTime) {
        return shardHelper.dbShards().stream()
                .mapToInt(shard -> clientBrandsRepository.deleteEntriesOlderThanDateTime(shard, borderDateTime))
                .sum();
    }

    /**
     * Удалить из таблицы ppc.client_brands все записи во всех шардах, принадлежащие клиентам с заданным ClientId
     * <p>
     * Удаление делается чанками по {@value DELETE_CHUNK_SIZE}
     *
     * @return количество удаленных из таблицы строк
     */
    public int deleteEntriesByClientId(Collection<ClientId> clientIds) {
        return shardHelper
                .groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong)
                .chunkedBy(DELETE_CHUNK_SIZE).stream()
                .mapToInt(el -> clientBrandsRepository.deleteEntriesByClientId(el.getKey(), el.getValue()))
                .sum();
    }
}
