package ru.yandex.direct.core.entity.client.service;

import java.util.List;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.client.service.validation.ClientChiefValidationService;
import ru.yandex.direct.core.entity.turbolanding.service.TurboLandingService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.core.entity.vcard.repository.VcardRepository;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.singleton;
import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;
import static ru.yandex.direct.utils.CommonUtils.notEquals;

@Service
public class ClientChiefService {

    private final ShardHelper shardHelper;
    private final ClientRepository clientRepository;
    private final UserRepository userRepository;
    private final VcardRepository vcardRepository;
    private final CampaignRepository campaignRepository;
    private final BalanceService balanceService;
    private final ClientChiefValidationService validationService;
    private final TurboLandingService turboLandingService;

    @Autowired
    public ClientChiefService(ShardHelper shardHelper, ClientRepository clientRepository, UserRepository userRepository,
                              VcardRepository vcardRepository, CampaignRepository campaignRepository,
                              BalanceService balanceService, ClientChiefValidationService validationService,
                              TurboLandingService turboLandingService) {
        this.validationService = validationService;
        this.clientRepository = clientRepository;
        this.userRepository = userRepository;
        this.shardHelper = shardHelper;
        this.balanceService = balanceService;
        this.vcardRepository = vcardRepository;
        this.campaignRepository = campaignRepository;
        this.turboLandingService = turboLandingService;
    }

    /**
     * Изменяет главного представителя клиента.
     * Если новый совпадает со старым, то ничего не делает, просто возвращает Success.
     *
     * @param newChiefUid Идентификатор нового главного представителя. На момент вызова метода пользователь должен
     *                    существовать и быть представителем заданного клиента, в противном случае вернётся ошибка
     *                    валидации.
     * @param clientId    Идентификатор клиента. Ожидается, что существование клиента с таким ID уже проверено, в
     *                    противном случае метод вернёт исключение IllegalArgumentException.
     */
    public Result<User> changeChief(Long newChiefUid, ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        checkArgument(shard != NO_SHARD);
        Client client = clientRepository.get(shard, singleton(clientId)).stream().findFirst()
                .orElseThrow(IllegalArgumentException::new);
        Long oldChiefUid = client.getChiefUid();
        List<User> users = userRepository.fetchByUids(shard, List.of(newChiefUid, oldChiefUid));
        User newChief = StreamEx.of(users)
                .findAny(u -> Objects.equals(u.getId(), newChiefUid))
                .orElse(null);
        if (Objects.equals(oldChiefUid, newChiefUid)) {
            return Result.successful(newChief);
        }

        List<Long> allRepUids = userRepository.getUidsByClientIds(shard, singleton(clientId)).get(clientId);
        ValidationResult<User, Defect> validationResult =
                validationService.validateChangeChief(newChief, clientId, allRepUids);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        // Баланс идемпотентный - его обновляем первым
        balanceService.changeChief(clientId, newChiefUid);

        User oldChief = StreamEx.of(users)
                .findAny(u -> Objects.equals(u.getId(), oldChiefUid))
                .orElseThrow();
        updateUserMinorFields(shard, newChief, oldChief);

        updateRelatedTables(shard, allRepUids, newChiefUid);
        clientRepository.updateChief(shard, clientId, oldChiefUid, newChiefUid);
        turboLandingService.refreshTurbolandingMetrikaGrants(newChiefUid, clientId);
        User actualChief = userRepository.fetchByUids(shard, singleton(newChiefUid)).stream().findFirst()
                .orElse(null);
        return Result.successful(actualChief);
    }

    private void updateUserMinorFields(int shard, User newChief, User oldChief) {
        ModelChanges<User> modelChanges = new ModelChanges<>(newChief.getId(), User.class);
        //Сохранять город при смене главного представителя: DIRECT-22085
        modelChanges.process(oldChief.getGeoId(), User.GEO_ID);
        //Включаем будущему главному представителю отправку уведомлений о клиенте
        modelChanges.process(true, User.SEND_CLIENT_LETTERS);
        modelChanges.process(true, User.SEND_CLIENT_SMS);
        AppliedChanges<User> appliedChanges = modelChanges.applyTo(newChief);
        userRepository.update(shard, singleton(appliedChanges));
    }

    private void updateRelatedTables(int shard, List<Long> allRepresentativeUids, Long newChiefUid) {
        Set<Long> otherRepUids = StreamEx.of(allRepresentativeUids)
                .filter(l -> notEquals(l, newChiefUid))
                .toSet();
        vcardRepository.updateVcardUids(shard, otherRepUids, newChiefUid);
        vcardRepository.updateOrgDetailsUids(shard, otherRepUids, newChiefUid);
        campaignRepository.updateCampOwners(shard, otherRepUids, newChiefUid);
        userRepository.updateUserRelatedTables(shard, otherRepUids, newChiefUid);
    }

}
