package ru.yandex.direct.core.entity.client.service;

import java.time.Duration;
import java.util.Collection;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.client.repository.ClientCurrencyConversionTeaserRepository;
import ru.yandex.direct.core.entity.currency.model.CurrencyConversionState;
import ru.yandex.direct.core.entity.currency.repository.CurrencyConvertQueueRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;

/**
 * Сервис для проверки, затронут ли определённый клиент переводом в валюту.
 */
@ParametersAreNonnullByDefault
@Service
public class ClientCurrencyConversionTeaserService {
    private final ClientCurrencyConversionTeaserRepository clientCurrencyConversionTeaserRepository;
    private final CurrencyConvertQueueRepository currencyConvertQueueRepository;
    private final ShardHelper shardHelper;
    private final Duration stopOperationBeforeConvert;

    @Autowired
    public ClientCurrencyConversionTeaserService(
            ClientCurrencyConversionTeaserRepository clientCurrencyConversionTeaserRepository,
            CurrencyConvertQueueRepository currencyConvertQueueRepository,
            ShardHelper shardHelper,
            DirectConfig directConfig) {
        this.clientCurrencyConversionTeaserRepository = clientCurrencyConversionTeaserRepository;
        this.currencyConvertQueueRepository = currencyConvertQueueRepository;
        this.shardHelper = shardHelper;
        this.stopOperationBeforeConvert = directConfig.getDuration("client_currency_conversion.stop_operation_before");
    }

    /**
     * Переводится ли клиент в валюту прямо сейчас или в ближайшем будущем.
     *
     * @param excludeStates если у клиента в очереди на перевод есть задание в одном
     *                      из таких статусов, считать его завершённым. Задания в статусе
     *                      DONE всегда считаются завершёнными.
     */
    public boolean isClientConvertingSoon(ClientId clientId, Collection<CurrencyConversionState> excludeStates) {
        int shard = shardHelper.getShardByClientId(clientId);
        Collection<ClientId> clientsConvertingSoon = currencyConvertQueueRepository.fetchConvertingClients(
                shard,
                singletonList(clientId),
                stopOperationBeforeConvert,
                excludeStates);

        return clientsConvertingSoon.contains(clientId);
    }

    /**
     * Переводится ли клиент в валюту прямо сейчас или в ближайшем будущем.
     */
    public boolean isClientConvertingSoon(ClientId clientId) {
        return isClientConvertingSoon(clientId, emptySet());
    }

    /**
     * true, если клиент в у. е. и его заблокировали до того момента, пока он не зайдёт в интерфейс
     * и не согласится с тем, что его переведут в валюту.
     */
    public boolean doesClientHaveToConvert(ClientId clientId) {
        Set<ClientId> clientsThatHaveToConvert = clientCurrencyConversionTeaserRepository.getClientsThatHaveToConvert(
                shardHelper.getShardByClientId(clientId), singletonList(clientId));
        return clientsThatHaveToConvert.contains(clientId);
    }
}
