package ru.yandex.direct.core.entity.client.service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.enums.YandexDomain;
import ru.yandex.direct.core.entity.region.RegionDesc;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.regions.GeoTreeType;
import ru.yandex.direct.regions.Region;

import static java.util.Collections.singleton;
import static ru.yandex.direct.core.entity.region.TranslatableRegion.LOCALE_NAMES_GETTERS;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;
import static ru.yandex.direct.regions.Region.RUSSIA_REGION_ID;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class ClientGeoService {
    private final GeoTreeFactory geoTreeFactory;
    private final ClientService clientService;

    @Nullable
    public static String getRegionName(@Nullable RegionDesc regionDesc, @Nonnull Language language) {
        if (regionDesc == null) {
            return null;
        }
        var langNameGetter = LOCALE_NAMES_GETTERS.get(language);
        if (langNameGetter == null) {
            throw new IllegalArgumentException("unknown language provided");
        }
        return langNameGetter.apply(regionDesc);
    }

    @Autowired
    public ClientGeoService(GeoTreeFactory geoTreeFactory, ClientService clientService) {
        this.geoTreeFactory = geoTreeFactory;
        this.clientService = clientService;
    }

    @Nonnull
    public GeoTree getClientTranslocalGeoTree(ClientId clientId) {
        return getClientTranslocalGeoTree(clientService.getCountryRegionIdByClientId(clientId).orElse(null));
    }

    @Nonnull
    public GeoTree getClientTranslocalGeoTree(@Nullable Long clientCountryId) {
        return geoTreeFactory.getTranslocalGeoTree(Optional.ofNullable(clientCountryId).orElse(GLOBAL_REGION_ID));
    }

    public GeoTree getClientTranslocalGeoTree(@Nullable YandexDomain yandexDomain) {
        long clientCountryId = YandexDomain.RU == yandexDomain ? RUSSIA_REGION_ID : GLOBAL_REGION_ID;
        return geoTreeFactory.getTranslocalGeoTree(clientCountryId);
    }

    /*
        Анализирует список регионов и в случае необходимости добавляет Крым
        Используется для сохранения регионов в БД.
        Аналог функции GeoTools::modify_translocal_region_before_save в perl
     */
    public List<Long> convertForSave(List<Long> geoIds, GeoTree geoTree) {

        List<Long> geoIdsLocal = new ArrayList<>(geoIds);
        Set<Long> regionSet = new HashSet<>(geoIds);

        List<Long> plusRegions = filterList(geoIds, geoId -> geoId >= 0);

        if (isCrimeaIsNeitherInPlusRegionsNorInMinusRegion(regionSet) && noneRegionInCrimea(plusRegions)
                && (isAtLeastOneRegionIsFromRussiaForRussianGeoTree(plusRegions, geoTree)
                || isAtLeastOneRegionIsFromUkraineForGlobalGeoTree(plusRegions, geoTree)
                && !regionSet.contains(-Region.UKRAINE_REGION_ID))) {
            // добавляем Крым
            geoIdsLocal.add(Region.CRIMEA_REGION_ID);
        }

        // вычитание "Крыма" из любых регионов кроме "всего мира" не имеет смысла - убираем
        boolean wantRemoveCrimeaFromWorld = plusRegions.stream().noneMatch(v -> v.equals(Region.GLOBAL_REGION_ID));
        if (wantRemoveCrimeaFromWorld) {
            geoIdsLocal = filterList(geoIdsLocal, v -> v != -Region.CRIMEA_REGION_ID);
        }

        return geoIdsLocal;
    }

    /*
        Аналог функции GeoTools::modify_translocal_region_before_show в perl
     */
    public List<Long> convertForWeb(List<Long> geoIds, GeoTree geoTree) {

        List<Long> geoIdsLocal = new ArrayList<>(geoIds);
        Set<Long> regionSet = new HashSet<>(geoIds);

        List<Long> plusRegions = filterList(geoIds, geoId -> geoId >= 0);
        List<Long> minusRegions = mapList(filterList(geoIds, geoId -> geoId < 0), v -> -v);

        if (isCrimeaIsNeitherInPlusRegionsNorInMinusRegion(regionSet)
                && noneRegionInCrimea(plusRegions)) {

            if (isAtLeastOneRegionIsFromRussiaForRussianGeoTree(plusRegions, geoTree)) {
                insertMinusCrimeaIntoRightPlace(geoTree, geoIdsLocal, RUSSIA_REGION_ID);
            } else if (isAtLeastOneRegionIsFromUkraineForGlobalGeoTree(plusRegions, geoTree)) {
                insertMinusCrimeaIntoRightPlace(geoTree, geoIdsLocal, Region.UKRAINE_REGION_ID);
            }
        }

        boolean plusRegionsAreInCrimea = plusRegions.stream().anyMatch(v -> !v.equals(Region.CRIMEA_REGION_ID)
                && geoTree.isAnyRegionIncludedIn(singleton(Region.CRIMEA_REGION_ID), singleton(v)));

        boolean haveNotMinusRegionsFromCrimea = minusRegions.stream().noneMatch(v -> !v.equals(Region.CRIMEA_REGION_ID)
                && geoTree.isAnyRegionIncludedIn(singleton(v), singleton(Region.CRIMEA_REGION_ID)));

        // Выкидываем Крым в случае, если имеющийся плюс-регион сам находится в Крыму и Крым не вычитается.
        if (plusRegionsAreInCrimea && haveNotMinusRegionsFromCrimea) {
            geoIdsLocal = filterList(geoIdsLocal, v -> !v.equals(Region.CRIMEA_REGION_ID));
        }

        return geoIdsLocal;
    }

    /**
     * Добавляет минус Крым после соответствующего плюс региона - после России для русского гео дерево и после
     * Украины - для глобального.
     * <p>
     * Учитывает, что Украины может не быть, а может быть СНГ - минус Крым будет добавлен после СНГ
     *
     * @param geoTree     гео дерево
     * @param geoIdsLocal список гео, куда добавится Крым
     * @param regionId    идентификатор региона, после которого нужно добавить минус Крым
     */
    private void insertMinusCrimeaIntoRightPlace(GeoTree geoTree, List<Long> geoIdsLocal, long regionId) {
        Region currentRegion = geoTree.getRegion(regionId);

        while (currentRegion != null && currentRegion.getId() != GLOBAL_REGION_ID) {
            int indexOfRegion = geoIdsLocal.indexOf(currentRegion.getId());
            if (indexOfRegion != -1) {
                geoIdsLocal.add(indexOfRegion + 1, -Region.CRIMEA_REGION_ID);
                return;
            }
            currentRegion = currentRegion.getParent();
        }

        geoIdsLocal.add(-Region.CRIMEA_REGION_ID);
    }

    /*
        Есть ли хотябы один из регионов подрегиона полуострова Крым.
        Подсчет идет для геодерева для API
    */
    private boolean noneRegionInCrimea(List<Long> plusRegions) {
        return plusRegions.stream()
                .noneMatch(v -> geoTreeFactory.getApiGeoTree()
                        .isAnyRegionIncludedIn(singleton(Region.CRIMEA_REGION_ID),
                                singleton(v)));

    }

    /*
        Крыма нет ни в плюс-регионах, ни в минус регионах
    */
    private boolean isCrimeaIsNeitherInPlusRegionsNorInMinusRegion(Set<Long> regionSet) {
        return !regionSet.contains(Region.CRIMEA_REGION_ID) && !regionSet.contains(-Region.CRIMEA_REGION_ID);
    }

    /*
        В случае русского (RUSSIAN) текущего дерева - среди выбранных регионов есть хотябы один регион из России
    */
    private boolean isAtLeastOneRegionIsFromRussiaForRussianGeoTree(List<Long> plusRegions, GeoTree geoTree) {
        return geoTree.getGeoTreeType() == GeoTreeType.RUSSIAN
                && (plusRegions.stream().anyMatch(v -> geoTree.isAnyRegionIncludedIn(
                singleton(Region.RUSSIA_REGION_ID), singleton(v))));
    }

    /*
        В случае украинского (GLOBAL) текущего дерева - среди выбранных регионов есть хотябы один регион из Украины
    */
    boolean isAtLeastOneRegionIsFromUkraineForGlobalGeoTree(List<Long> plusRegions, GeoTree geoTree) {
        return geoTree.getGeoTreeType() == GeoTreeType.GLOBAL
                && (plusRegions.stream().anyMatch(v -> geoTree.isAnyRegionIncludedIn(
                singleton(Region.UKRAINE_REGION_ID), singleton(v))));
    }

}
