package ru.yandex.direct.core.entity.client.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.ClientLimits;
import ru.yandex.direct.core.entity.client.model.ClientSpecialLimits;
import ru.yandex.direct.core.entity.client.repository.ApiSpecialUserOptionsRepository;
import ru.yandex.direct.core.entity.client.repository.ClientLimitsRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class ClientLimitsService {
    private static final String REPORTS_LIMIT_KEYNAME = "api_reports_limit";

    private final ClientLimitsRepository clientLimitsRepository;
    private final ApiSpecialUserOptionsRepository apiSpecialUserOptionsRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public ClientLimitsService(ClientLimitsRepository clientLimitsRepository,
                               ApiSpecialUserOptionsRepository apiSpecialUserOptionsRepository,
                               ShardHelper shardHelper) {
        this.clientLimitsRepository = clientLimitsRepository;
        this.apiSpecialUserOptionsRepository = apiSpecialUserOptionsRepository;
        this.shardHelper = shardHelper;
    }

    /**
     * Получить лимиты клиента для clientId
     *
     * @return Лимиты клиента для указанного clientId
     */
    @Nonnull
    public ClientLimits getClientLimits(ClientId clientId) {
        return massGetClientLimits(List.of(clientId)).get(0);
    }

    /**
     * Получить лимиты клиентов для коллекции clientId
     *
     * @return коллекцию лимитов клиентов, для всех clientId присутствующих в запросе, но в произвольном порядке
     */
    @Nonnull
    public List<ClientLimits> massGetClientLimits(Collection<ClientId> clientIds) {
        List<ClientLimits> results =
                shardHelper.groupByShard(mapList(clientIds, ClientId::asLong), ShardKey.CLIENT_ID).stream()
                        .flatMapKeyValue((shard, ids) -> clientLimitsRepository.fetchByClientIds(shard, ids).stream())
                        .toList();
        Set<ClientId> present = listToSet(results, ClientLimits::getClientId);

        clientIds.stream()
                .filter(clientId -> !present.contains(clientId))
                .map(this::createEmptyClientLimits)
                .forEach(results::add);
        return results;
    }

    private ClientLimits createEmptyClientLimits(ClientId clientId) {
        ClientLimits clientLimits = new ClientLimits();
        clientLimits.withClientId(clientId);
        return clientLimits;
    }

    /**
     * Получить специальные лимиты клиентов для коллекции clientId
     *
     * @return коллекцию лимитов клиентов, для всех clientId, присутствующих в запросе
     */
    @Nonnull
    public List<ClientSpecialLimits> massGetClientSpecialLimits(List<ClientId> clientIds) {
        Map<ClientId, Long> clientToOptionValueMap =
                shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong).stream()
                        .flatMapKeyValue((shard, ids) -> apiSpecialUserOptionsRepository
                                .getSingleSpecialUserOptionForMultipleClients(shard, ids, REPORTS_LIMIT_KEYNAME)
                                .entrySet().stream())
                        .collect(toMap(Map.Entry::getKey, Map.Entry::getValue));

        return clientIds.stream()
                .map(clientId -> new ClientSpecialLimits()
                        .withClientId(clientId.asLong())
                        .withStatReportsCountLimit(clientToOptionValueMap.get(clientId)))
                .collect(toList());
    }
}
