package ru.yandex.direct.core.entity.client.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.ClientMeasurerSettings;
import ru.yandex.direct.core.entity.client.model.ClientMeasurerSystem;
import ru.yandex.direct.core.entity.client.repository.ClientMeasurerSettingsRepository;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardSupport;

@Service
@ParametersAreNonnullByDefault
public class ClientMeasurerSettingsService {
    private final ShardSupport shardSupport;
    private final ClientMeasurerSettingsRepository clientMeasurerSettingsRepository;
    private final MediascopeClientSettingsService mediascopeClientSettingsService;

    public ClientMeasurerSettingsService(
            ShardSupport shardSupport,
            ClientMeasurerSettingsRepository clientMeasurerSettingsRepository,
            MediascopeClientSettingsService mediascopeClientSettingsService
    ) {
        this.shardSupport = shardSupport;
        this.clientMeasurerSettingsRepository = clientMeasurerSettingsRepository;
        this.mediascopeClientSettingsService = mediascopeClientSettingsService;
    }

    public String extendSettings(ClientMeasurerSystem measurerSystem, String settings) {
        if (measurerSystem == ClientMeasurerSystem.MEDIASCOPE) {
            return mediascopeClientSettingsService.extendSettings(settings);
        }

        return settings;
    }

    public void insertOrUpdate(Long clientId, List<ClientMeasurerSettings> clientMeasurerSettingsList) {
        if (clientMeasurerSettingsList.isEmpty()) {
            return;
        }

        encryptSettings(clientMeasurerSettingsList);
        var shardId = shardSupport.getShard(ShardKey.CLIENT_ID, clientId);
        clientMeasurerSettingsRepository.insertOrUpdate(shardId, clientMeasurerSettingsList);
    }

    public void update(int shard, ClientMeasurerSettings clientMeasurerSettings) {
        encryptSettings(List.of(clientMeasurerSettings));
        clientMeasurerSettingsRepository.update(shard, clientMeasurerSettings);
    }

    public List<ClientMeasurerSettings> getByClientId(Long clientId) {
        var shardId = shardSupport.getShard(ShardKey.CLIENT_ID, clientId);
        var measurerSettings = clientMeasurerSettingsRepository.getByClientId(shardId, clientId);
        decryptSettings(measurerSettings);
        return measurerSettings;
    }

    public ClientMeasurerSettings getByClientIdAndSystem(Long clientId, ClientMeasurerSystem measurerSystem) {
        var shardId = shardSupport.getShard(ShardKey.CLIENT_ID, clientId);
        return getByClientIdsAndSystem(shardId, List.of(clientId), measurerSystem).get(clientId);
    }

    public Map<Long, ClientMeasurerSettings> getByClientIdsAndSystem(
            int shard, List<Long> clientIds, ClientMeasurerSystem measurerSystem
    ) {
        var measurerSettingsMap =
                clientMeasurerSettingsRepository.getByClientIdsAndSystem(shard, clientIds, measurerSystem);
        decryptSettings(measurerSettingsMap.values());
        return measurerSettingsMap;
    }

    public List<ClientMeasurerSettings> getByMeasurerSystem(int shard, ClientMeasurerSystem clientMeasurerSystem) {
        var measurerSettings = clientMeasurerSettingsRepository.getByMeasurerSystem(shard, clientMeasurerSystem);
        decryptSettings(measurerSettings);
        return measurerSettings;
    }

    public void deleteByClientIdAndSystem(Long clientId, ClientMeasurerSystem clientMeasurerSystem) {
        var shardId = shardSupport.getShard(ShardKey.CLIENT_ID, clientId);
        clientMeasurerSettingsRepository.deleteByClientIdAndSystem(shardId, clientId, clientMeasurerSystem);
    }

    private void encryptSettings(Collection<ClientMeasurerSettings> clientMeasurerSettingsList) {
        clientMeasurerSettingsList.stream()
                .filter(s -> s.getClientMeasurerSystem() == ClientMeasurerSystem.MEDIASCOPE)
                .forEach(s -> s.setSettings(mediascopeClientSettingsService.encryptSettings(s.getSettings())));
    }

    private void decryptSettings(Collection<ClientMeasurerSettings> clientMeasurerSettingsList) {
        clientMeasurerSettingsList.stream()
                .filter(s -> s.getClientMeasurerSystem() == ClientMeasurerSystem.MEDIASCOPE)
                .forEach(s -> s.setSettings(mediascopeClientSettingsService.decryptSettings(s.getSettings())));
    }
}
