package ru.yandex.direct.core.entity.client.service;

import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.AgencyNds;
import ru.yandex.direct.core.entity.client.model.ClientDataForNds;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.repository.AgencyNdsRepository;
import ru.yandex.direct.core.entity.client.repository.ClientNdsRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;


@Service
@ParametersAreNonnullByDefault
public class ClientNdsService {
    private final ShardHelper shardHelper;
    private final ClientNdsRepository clientNdsRepository;
    private final ClientRepository clientRepository;
    private final AgencyNdsRepository agencyNdsRepository;

    @Autowired
    public ClientNdsService(ShardHelper shardHelper, ClientNdsRepository clientNdsRepository,
                            ClientRepository clientRepository, AgencyNdsRepository agencyNdsRepository) {
        this.shardHelper = shardHelper;
        this.clientNdsRepository = clientNdsRepository;
        this.clientRepository = clientRepository;
        this.agencyNdsRepository = agencyNdsRepository;
    }

    @Nullable
    public ClientNds getClientNds(ClientId clientId) {
        return Iterables.getFirst(massGetClientNds(singletonList(clientId)), null);
    }

    /**
     * Получение фактического НДС для клиента
     * Если клиент агентский и является резидентом, то используем НДС агентства
     * В остальных случаях получаем НДС самого клиента.
     *
     * @param clientId    идентификатор клиента
     * @param agencyId    идентификатор агентства, если оно есть
     * @param nonResident является ли клиент налоговым резидентом (client_options.non_resident)
     */
    @Nullable
    public ClientNds getEffectiveClientNds(ClientId clientId, @Nullable ClientId agencyId, boolean nonResident) {
        ClientId targetClientId = getEffectiveNdsClientId(clientId, agencyId, nonResident);
        return massGetClientNds(singletonList(targetClientId)).stream().findFirst().orElse(null);
    }

    /**
     * Получает фактический НДС для набора клиентов.
     */
    public List<ClientNds> massGetEffectiveClientNdsByIds(Collection<ClientId> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .stream()
                .flatMapKeyValue((shard, ids) ->
                        massGetEffectiveClientNds(clientRepository.getClientsDataForNds(shard, ids))
                                .stream()
                )
                .toList();
    }

    /**
     * Получает фактический НДС для набора клиентов.
     */
    public <T extends ClientDataForNds> List<ClientNds> massGetEffectiveClientNds(Collection<T> clients) {
        if (clients.isEmpty()) {
            return emptyList();
        }
        Set<ClientId> clientIds = clients.stream()
                .map(client -> getEffectiveNdsClientId(ClientId.fromLong(client.getId()),
                        ClientId.fromNullableLong(client.getAgencyClientId()),
                        client.getNonResident()))
                .collect(toSet());
        Map<Long, ClientNds> clientNdsMap = listToMap(massGetClientNds(clientIds), ClientNds::getClientId);
        Map<Long, AgencyNds> agencyNdsMap = listToMap(massGetAgencyNds(clientIds), AgencyNds::getClientId);
        return clients.stream()
                .map(client -> {
                    ClientId targetClientId = getEffectiveNdsClientId(ClientId.fromLong(client.getId()),
                            ClientId.fromNullableLong(client.getAgencyClientId()),
                            client.getNonResident());
                    // У клиента может не оказаться записи в client_nds, в этом случае возвращаем null
                    ClientNds effectiveClientNds = clientNdsMap.get(targetClientId.asLong());
                    AgencyNds effectiveAgencyNds = agencyNdsMap.get(targetClientId.asLong());
                    return getClientNds(client.getId(), effectiveClientNds, effectiveAgencyNds);
                })
                .filter(Objects::nonNull)
                .collect(toList());
    }

    /**
     * Возвращает ClientNds, выбирая существующий НДС либо в случае отсутствия использует НДС агентства
     */
    private ClientNds getClientNds(Long clientId, @Nullable ClientNds clientNds, @Nullable AgencyNds agencyNds) {
        if (clientNds != null) {
            return new ClientNds()
                    .withClientId(clientId)
                    .withDateFrom(clientNds.getDateFrom())
                    .withDateTo(clientNds.getDateTo())
                    .withNds(clientNds.getNds());
        }
        if (agencyNds != null) {
            return new ClientNds()
                    .withClientId(clientId)
                    .withDateFrom(agencyNds.getDateFrom())
                    .withDateTo(agencyNds.getDateTo())
                    .withNds(agencyNds.getNds());
        }
        return null;
    }

    /**
     * Возвращает тот clientId, чей НДС нужно брать в качестве фактического НДС для клиента с указанными свойствами.
     * Если клиент агентский и является резидентом, то метод возвращает agencyId.
     * В остальных случаях возвращается clientId.
     */
    private ClientId getEffectiveNdsClientId(
            ClientId clientId, @Nullable ClientId agencyId, @Nullable Boolean nonResident
    ) {
        return nonResident != null && nonResident ? clientId : nvl(agencyId, clientId);
    }

    /**
     * Получает график НДС для клиента.
     *
     * @param clientId идентификатор клиента
     * @return коллекция НДС
     */
    @Nullable
    public Collection<ClientNds> getClientNdsHistory(ClientId clientId) {
        return clientNdsRepository.fetchHistoryByClientId(shardHelper.getShardByClientId(clientId), clientId);
    }

    /**
     * Получает НДС для коллекции клиентов.
     *
     * @param clientIds коллекция идентификаторов клиентов
     * @return коллекция НДС
     */
    @Nonnull
    public Collection<ClientNds> massGetClientNds(Collection<ClientId> clientIds) {
        LocalDate today = LocalDate.now();
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).stream()
                .flatMapKeyValue((shard, ids) -> clientNdsRepository.fetchByClientIds(shard, ids, today).stream())
                .toList();
    }

    /**
     * Получает НДС для коллекции агентских клиентов
     *
     * @param clientIds коллекция идентификаторов клиентов
     * @return коллекция НДС
     */
    @Nonnull
    public Collection<AgencyNds> massGetAgencyNds(Collection<ClientId> clientIds) {
        LocalDate today = LocalDate.now();
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).stream()
                .flatMapKeyValue((shard, ids) -> agencyNdsRepository.fetchByClientIds(shard, ids, today).stream())
                .toList();
    }

    /**
     * Получает графики НДС для коллекции клиентов
     */
    @Nonnull
    public Map<Long, Collection<ClientNds>> massGetClientNdsHistory(Collection<Long> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).stream()
                .mapKeyValue(clientNdsRepository::fetchHistoryByClientIds)
                .flatMap(map -> map.entrySet().stream())
                .toMap(Map.Entry::getKey, Map.Entry::getValue);
    }

}
