package ru.yandex.direct.core.entity.client.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.enums.YandexDomain;
import ru.yandex.direct.core.entity.campaign.model.YandexOffice;
import ru.yandex.direct.core.entity.client.model.ContactPhone;
import ru.yandex.direct.core.entity.client.model.office.GeoCity;
import ru.yandex.direct.core.entity.client.model.office.OfficeContact;
import ru.yandex.direct.core.entity.client.model.office.OfficeName;
import ru.yandex.direct.core.entity.client.model.office.RegionContact;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.yandexoffice.YandexOfficesRepository;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.Region;

import static ru.yandex.direct.core.entity.client.model.office.OfficeName.BLR;
import static ru.yandex.direct.core.entity.client.model.office.OfficeName.NEW_OFFICES;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

/**
 * Сервис, возвращающий данные офиса для клиента
 */
@Service
@ParametersAreNonnullByDefault
public class ClientOfficeService {

    private static final Set<GeoCity> MOSCOW_OFFICE_GEO_CITIES =
            ImmutableSet.of(GeoCity.RU, GeoCity.MOSCOW, GeoCity.CENTER, GeoCity.VOLGA);
    private static final List<GeoCity> GEO_CITIES_LIST = ImmutableList.copyOf(GeoCity.values());

    private final ClientGeoService clientGeoService;
    private final YandexOfficesRepository yandexOfficesRepository;

    @Autowired
    public ClientOfficeService(ClientGeoService clientGeoService, YandexOfficesRepository yandexOfficesRepository) {
        this.clientGeoService = clientGeoService;
        this.yandexOfficesRepository = yandexOfficesRepository;
    }

    /**
     * Получить {@link GeoCity}, нужен для определения офиса {@link #getOfficeContactForFooter}
     * Логика перенесена из перла (см. YandexOffice.pm::get_officecity_by_geo_footer).
     * Данные города можно посмотреть в {@link GeoCity}, а цепочка if'ов заменено на stream-api
     */
    @Nullable
    GeoCity getGeoCity(Long regionId, GeoTree geoTree) {
        Set<Long> region = Collections.singleton(regionId);

        Predicate<GeoCity> isRegionServicedByGeoCity =
                geoCity -> geoTree.isAnyRegionIncludedIn(region, geoCity.getRegionIds())
                        && !geoTree.isAnyRegionIncludedIn(region, geoCity.getMinusRegionIds());

        return GEO_CITIES_LIST.stream()
                .filter(isRegionServicedByGeoCity)
                .findFirst()
                .orElse(null);
    }

    public Map<Long, YandexOffice> getOfficesByIds(Collection<Long> officeIds) {
        return yandexOfficesRepository.getOfficesById(officeIds);
    }

    /**
     * Возвращает контактные номера менеджера кампании. Отображается на showCamp.
     * Оригинал был в {@code b-campaign-state.bemtree.js}
     */
    public List<ContactPhone> getManagerContactPhones(User manager, @Nullable YandexOffice managerOffice) {
        var officeName = ifNotNull(managerOffice, YandexOffice::getOfficeNick);

        OfficeName office = StreamEx.of(OfficeName.values())
                .findFirst(o -> o.getName().equals(officeName))
                .orElse(null);

        if (managerOffice == null || office == null) {
            return List.of();
        }

        List<ContactPhone> contactPhones = new ArrayList<>();

        if (office.getFederalPhone() != null) {
            contactPhones.add(new ContactPhone()
                    .withPhone(office.getFederalPhone()));
        }

        boolean isNewOffice = NEW_OFFICES.contains(office);

        if (office == BLR) {
            contactPhones.add(new ContactPhone()
                    .withPhone(RegionContact.BLR_ADD.getPhone()));
        } else if (!isNewOffice) {
            contactPhones.add(new ContactPhone()
                    .withPhone(managerOffice.getOfficePhone()));
        }

        // Считаем, что в phone у менеджера либо доп код, либо что-то странное
        if (manager.getPhone() != null && manager.getPhone().length() <= 5) {
            try {
                Long managerPhoneExtension = Long.parseLong(manager.getPhone());
                contactPhones.get(contactPhones.size() - 1)
                        .withExtension(managerPhoneExtension);
            } catch (NumberFormatException ignored) {
            }
        }

        return contactPhones;
    }

    /**
     * Функция, возвращающая по коду региона и домену {@link OfficeContact}, описывающий офис, который обслуживает
     * данный регион.
     * Нужен для футера интерфейса
     * Метод перенесен из перла YandexOffice.pm::get_officecity_by_geo_footer. Если правите тут, правьте и там.
     */
    public OfficeContact getOfficeContactForFooter(Long regionId, Language language,
                                                   @Nullable YandexDomain yandexDomain,
                                                   @Nullable Long countryRegionId) {
        GeoCity geoCity = getGeoCity(regionId, clientGeoService.getClientTranslocalGeoTree(yandexDomain));

        if (YandexDomain.BY == yandexDomain || GeoCity.BY == geoCity) {
            return OfficeContact.BLR;
        } else if (YandexDomain.UA == yandexDomain) {
            return OfficeContact.UKR;
        } else if (YandexDomain.TR == yandexDomain) {
            return OfficeContact.TR;
        } else if ((countryRegionId != null && Region.KAZAKHSTAN_REGION_ID == countryRegionId)
                || YandexDomain.KZ == yandexDomain
                || GeoCity.KZ == geoCity) {
            return OfficeContact.KAZ;
        } else if (Language.EN == language) {
            return OfficeContact.EN;
        } else if (Language.UK == language) {
            return GeoCity.KIEV == geoCity ? OfficeContact.KIEV : OfficeContact.UKR;
        } else if (Language.RU == language) {
            if (GeoCity.UA == geoCity || GeoCity.ODESSA == geoCity) {
                return OfficeContact.UKR;
            } else if (GeoCity.KIEV == geoCity) {
                return OfficeContact.KIEV;
            } else if (MOSCOW_OFFICE_GEO_CITIES.contains(geoCity)) {
                return OfficeContact.MSK;
            } else if (GeoCity.SPB == geoCity || GeoCity.NORTH == geoCity) {
                return OfficeContact.SPB;
            } else if (GeoCity.EBURG == geoCity || GeoCity.URAL == geoCity) {
                return OfficeContact.EKB;
            } else if (GeoCity.NOVOSIB == geoCity || GeoCity.SIBERIA == geoCity) {
                return OfficeContact.NSK;
            } else if (GeoCity.N_NOVGOROD == geoCity) {
                return OfficeContact.N_NOVGOROD;
            } else if (GeoCity.SOUTH == geoCity) {
                return OfficeContact.ROSTOV;
            } else if (GeoCity.TATARSTAN == geoCity) {
                return OfficeContact.KAZAN;
            }
        }

        return OfficeContact.OTHER;
    }
}
