package ru.yandex.direct.core.entity.client.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.container.PrimaryManagersQueryFilter;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientPrimaryManager;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.client.service.validation.PrimaryManagerValidationService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.client.container.PrimaryManagersQueryFilter.getNextIdmManagersPageFilter;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class ClientPrimaryManagerService {

    private static final String DEFAULT_LOGIN = "-";

    private final PrimaryManagerValidationService validationService;
    private final ClientRepository clientRepository;
    private final CampaignRepository campaignRepository;
    private final UserService userService;
    private final ShardHelper shardHelper;

    @Autowired
    public ClientPrimaryManagerService(PrimaryManagerValidationService validationService,
                                       ClientRepository clientRepository,
                                       CampaignRepository campaignRepository,
                                       UserService userService, ShardHelper shardHelper) {
        this.validationService = validationService;
        this.clientRepository = clientRepository;
        this.campaignRepository = campaignRepository;
        this.userService = userService;
        this.shardHelper = shardHelper;
    }

    public List<ClientPrimaryManager> getAllIdmPrimaryManagers() {
        ArrayList<ClientPrimaryManager> managers = new ArrayList<>();
        PrimaryManagersQueryFilter filter = PrimaryManagersQueryFilter.allIdmPrimaryManagers();
        for (Integer shard : shardHelper.dbShards()) {
            managers.addAll(
                    clientRepository.getIdmPrimaryManagers(shard, filter));
        }
        fillClientPrimaryManagers(managers);
        return managers;
    }

    /**
     * Возвращает последовательность ролей главного менеджера заданной длинны начиная со следующей после последней
     * возвращённой. Возвращает только роли заданные через Idm. Набор отсортирован сначала по шарду,
     * потом по SubjectClientId, потом по PrimaryManagerUid. Если последняя возвращённая роль не задана, то возвращаются
     * роли с начала первого шарда.
     */
    public List<ClientPrimaryManager> getNextPageIdmPrimaryManagers(@Nullable ClientId lastClientId, int pageSize) {
        int continueShard = nvl(ifNotNull(lastClientId, shardHelper::getShardByClientId), 0);
        Iterator<Integer> shards = StreamEx.of(shardHelper.dbShards())
                .filter(shard -> continueShard <= shard)
                .sorted()
                .iterator();
        ArrayList<ClientPrimaryManager> managers = new ArrayList<>(pageSize);
        int limit = pageSize - managers.size();
        while (shards.hasNext() && 0 < limit) {
            PrimaryManagersQueryFilter filter = getNextIdmManagersPageFilter(lastClientId, limit);
            List<ClientPrimaryManager> shardManagers = clientRepository.getIdmPrimaryManagers(shards.next(), filter);
            managers.addAll(shardManagers);
            lastClientId = null;
            limit = pageSize - managers.size();
        }
        fillClientPrimaryManagers(managers);
        return managers;
    }

    private void fillClientPrimaryManagers(ArrayList<ClientPrimaryManager> managers) {
        List<Long> uids = mapList(managers, ClientPrimaryManager::getPrimaryManagerUid);
        Collection<User> users = userService.massGetUser(uids);
        Map<Long, User> userByUid = listToMap(users, User::getUid);
        for (ClientPrimaryManager manager : managers) {
            User user = userByUid.get(manager.getPrimaryManagerUid());
            String passportLogin = Optional.ofNullable(user).map(User::getLogin).orElse(DEFAULT_LOGIN);
            String domainLogin = Optional.ofNullable(user).map(User::getDomainLogin).orElse(DEFAULT_LOGIN);
            manager.withPassportLogin(passportLogin)
                    .withDomainLogin(domainLogin);
        }
    }

    /**
     * Изменяет информацию о текущем главном менеджере клиента.
     * Uid главного менеджера может принимать значение null, тогда в базе будет сохранено значение NULL - т.е. главный
     * менеджер станет не задан.
     * Так же, если операция изменения или удаления главного менеджера делается над менеджером заданным из IDM, то
     * вместе с главным менеджером клиента, изменяются и сервисирующие менеджеры во всех кампаниях клиента.
     */
    public Result<Long> updatePrimaryManager(ClientPrimaryManager newManager) {
        ValidationResult<ClientPrimaryManager, Defect> validationResult =
                validationService.updateValidation(newManager);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        ClientId subjClientId = newManager.getSubjectClientId();
        int shard = shardHelper.getShardByClientId(subjClientId);
        Client actualClient = clientRepository.get(shard, List.of(subjClientId)).get(0);
        if (needUpdateManagerForAllClientCampaigns(actualClient, newManager)) {
            campaignRepository.setManagerForAllClientCampaigns(shard, subjClientId, newManager.getPrimaryManagerUid());
        }
        ModelChanges<Client> modelChanges = new ModelChanges<>(subjClientId.asLong(), Client.class);
        modelChanges.process(newManager.getPrimaryManagerUid(), Client.PRIMARY_MANAGER_UID);
        modelChanges.process(newManager.getIsIdmPrimaryManager(), Client.IS_IDM_PRIMARY_MANAGER);
        AppliedChanges<Client> appliedChanges = modelChanges.applyTo(actualClient);
        clientRepository.update(shard, List.of(appliedChanges));
        return Result.successful(subjClientId.asLong(), validationResult);
    }

    private boolean needUpdateManagerForAllClientCampaigns(Client actualClient, ClientPrimaryManager newManager) {
        if (actualClient.getAgencyClientId() != null) {
            //если клиент агентский - ManagerUid кампаниям не проставляется
            return false;
        }
        boolean actualManagerIsIdm = actualClient.getIsIdmPrimaryManager();
        //новый uid главного менеджера совпадает со старым установленным из IDM,
        //значит всё и так уже должно быть в правильном состоянии, ничего не делаем
        if (Objects.equals(newManager.getPrimaryManagerUid(), actualClient.getPrimaryManagerUid())
                && actualManagerIsIdm) {
            return false;
        }
        //сбрасываем главного менеджера, если он был задан из IDM, то сбрасываем и сервисирующего менеджера
        //во всех кампаниях клиента
        if (newManager.getPrimaryManagerUid() == null) {
            return actualManagerIsIdm;
        }
        //задаём нового главного менеджера, если он задан из IDM, то устанавливаем его и сервисирующим менеджером
        //во всех кампаниях клиента
        return newManager.getIsIdmPrimaryManager();
    }

}
