package ru.yandex.direct.core.entity.client.service;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.NotImplementedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.repository.BsResyncQueueRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.container.ClientsQueryFilter;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientExperiment;
import ru.yandex.direct.core.entity.client.model.ClientWithOptions;
import ru.yandex.direct.core.entity.client.model.ClientWithUsers;
import ru.yandex.direct.core.entity.client.model.ClientsOptions;
import ru.yandex.direct.core.entity.client.model.CreateAgencySubclientStatus;
import ru.yandex.direct.core.entity.client.repository.ClientManagersRepository;
import ru.yandex.direct.core.entity.client.repository.ClientOptionsRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.currency.model.CurrencyConversionState;
import ru.yandex.direct.core.entity.currency.repository.CurrencyConvertQueueRepository;
import ru.yandex.direct.core.entity.currency.repository.ForceCurrencyConvertRepository;
import ru.yandex.direct.core.entity.user.model.ApiEnabled;
import ru.yandex.direct.core.entity.user.model.DeletedClientRepresentative;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbschema.ppc.enums.ClientsRole;
import ru.yandex.direct.dbschema.ppc.enums.ClientsWorkCurrency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.rbac.ClientPerm;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.RbacSubrole;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;
import static ru.yandex.direct.regions.Region.RUSSIA_REGION_ID;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class ClientService implements EntityService<Client, Long> {
    private static final Logger logger = LoggerFactory.getLogger(ClientService.class);

    private static final ImmutableMap<RbacSubrole, RbacRole> ROLE_BY_SUBROLE =
            ImmutableMap.<RbacSubrole, RbacRole>builder()
                    .put(RbacSubrole.SUPERPLACER, RbacRole.PLACER)
                    .put(RbacSubrole.SUPERMEDIA, RbacRole.MEDIA)
                    .put(RbacSubrole.SUPERTEAMLEADER, RbacRole.MANAGER)
                    .put(RbacSubrole.TEAMLEADER, RbacRole.MANAGER)
                    .build();

    private final ShardHelper shardHelper;
    private final UserRepository userRepository;
    private final GeoTreeFactory geoTreeFactory;
    private final ClientRepository clientRepository;
    private final ClientOptionsRepository clientOptionsRepository;
    private final ClientManagersRepository clientManagersRepository;
    private final RbacService rbacService;
    private final AgencyClientRelationService agencyClientRelationService;
    private final BsResyncQueueRepository bsResyncQueueRepository;
    private final CampaignRepository campaignRepository;
    private final CurrencyConvertQueueRepository currencyConvertQueueRepository;
    private final ForceCurrencyConvertRepository forceCurrencyConvertRepository;
    private final Duration stopOperationBeforeConvert;

    @Autowired
    public ClientService(
            ShardHelper shardHelper,
            UserRepository userRepository,
            GeoTreeFactory geoTreeFactory, ClientRepository clientRepository,
            ClientOptionsRepository clientOptionsRepository,
            ClientManagersRepository clientManagersRepository,
            RbacService rbacService,
            AgencyClientRelationService agencyClientRelationService,
            BsResyncQueueRepository bsResyncQueueRepository, CampaignRepository campaignRepository,
            CurrencyConvertQueueRepository currencyConvertQueueRepository,
            ForceCurrencyConvertRepository forceCurrencyConvertRepository,
            DirectConfig directConfig) {
        this.shardHelper = shardHelper;
        this.userRepository = userRepository;
        this.geoTreeFactory = geoTreeFactory;
        this.clientRepository = clientRepository;
        this.clientOptionsRepository = clientOptionsRepository;
        this.clientManagersRepository = clientManagersRepository;
        this.rbacService = rbacService;
        this.agencyClientRelationService = agencyClientRelationService;
        this.bsResyncQueueRepository = bsResyncQueueRepository;
        this.campaignRepository = campaignRepository;
        this.currencyConvertQueueRepository = currencyConvertQueueRepository;
        this.forceCurrencyConvertRepository = forceCurrencyConvertRepository;
        this.stopOperationBeforeConvert = directConfig.getDuration("client_currency_conversion.stop_operation_before");
    }

    @Override
    public MassResult<Long> add(
            ClientId clientId, Long operatorUid, List<Client> entities, Applicability applicability) {
        // есть отдельный сервис для добавления клиентов, возможно, нужно будет заюзать его
        throw new NotImplementedException("Adding clients is not yet implemented.");
    }

    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer, List<Client> entities, Applicability applicability) {
        throw new NotImplementedException("Copying clients is not yet implemented.");
    }

    /**
     * Вернуть номер shard-а для заданного клиента
     *
     * @param clientId Идентификатор клиента в балансе
     */
    public int getShardByClientIdStrictly(ClientId clientId) {
        return shardHelper.getShardByClientIdStrictly(clientId);
    }

    @Nullable
    public Client getClient(ClientId clientId) {
        return Iterables.getFirst(massGetClient(singletonList(clientId)), null);
    }

    @Override
    public List<Client> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        return massGetClient(StreamEx.of(ids).map(ClientId::fromLong).toList());
    }

    @Nonnull
    public List<Client> getClients(int shard, Collection<ClientId> clientIds) {
        return clientRepository.get(shard, clientIds);
    }

    @Nonnull
    public List<Client> massGetClient(Collection<ClientId> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).stream()
                .flatMapKeyValue((shard, ids) -> clientRepository.get(shard, ids).stream())
                .toList();
    }

    /**
     * Вернуть словарь из id клиента в клиента.
     * Не даёт гарантий, что все входные id будут в ключах ответа
     *
     * @param clientIds — коллекция id клиентов
     * @return словарь id клиента → клиент
     */
    @Nonnull
    public Map<ClientId, Client> massGetClientsByClientIds(Collection<ClientId> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).stream()
                .flatMapKeyValue((shard, ids) -> clientRepository.get(shard, ids).stream())
                .toMap(c -> ClientId.fromLong(c.getId()), Function.identity());
    }

    @Nonnull
    public Map<ClientId, RbacRole> massGetRolesByClientId(Collection<ClientId> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong).stream()
                .map(e -> clientRepository.getRbacRoles(e.getKey(), e.getValue()))
                .flatMapToEntry(Function.identity())
                .toMap();
    }

    @Nonnull
    public Map<ClientId, Long> massGetChiefUidsByClientIds(Collection<ClientId> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong)
                .stream()
                .map(e -> clientRepository.getChiefUidsByClientIds(e.getKey(), e.getValue()))
                .flatMapToEntry(Function.identity())
                .toMap();
    }

    /**
     * Постранично возвращает клиентов связанных с организациями (ConnectOrgId!=null). Страница представляет собой
     * отсортированный список заданной длинны начиная со следующего клиента после последнего возвращённого на предыдущей
     * итерации. Список отсортирован сначала по шарду, потом по ClientId. Если последний возвращённый клиент не задан,
     * то возвращаются клиенты с начала первого шарда.
     */
    public List<Client> getNextPageConnectOrgClients(@Nullable Integer lastShard, @Nullable ClientId lastClientId,
                                                     int pageSize) {
        int continueShard = nvl(lastShard, 0);
        Iterator<Integer> shards = StreamEx.of(shardHelper.dbShards())
                .filter(shard -> continueShard <= shard)
                .sorted()
                .iterator();
        ArrayList<Client> clients = new ArrayList<>(pageSize);
        int limit = pageSize - clients.size();
        while (shards.hasNext() && 0 < limit) {
            ClientsQueryFilter filter = ClientsQueryFilter.getNextConnectOrgClientsPageFilter(lastClientId, limit);
            List<Client> shardClients = clientRepository.getClientsByFilter(shards.next(), filter);
            clients.addAll(shardClients);
            lastClientId = null;
            limit = pageSize - clients.size();
        }
        return clients;
    }

    public List<Client> getClientsByConnectOrgId(List<Long> connectOrgIds) {
        List<Client> clientsWithConnectOrgId = new ArrayList<>();
        ClientsQueryFilter connectOrgIdsFilter = ClientsQueryFilter.getByConnectOrgIds(connectOrgIds);
        shardHelper.forEachShard(shard -> {
            List<Client> clients = clientRepository.getClientsByFilter(shard, connectOrgIdsFilter);
            clientsWithConnectOrgId.addAll(clients);
        });
        return clientsWithConnectOrgId;
    }

    @Nonnull
    public Map<Long, Client> massGetClientsByUids(Collection<Long> uids) {
        Map<Long, Long> uidToClientId = shardHelper.getClientIdsByUids(uids);

        Set<ClientId> uniqClientIds = StreamEx.ofValues(uidToClientId)
                .map(ClientId::fromLong)
                .toSet();

        Map<Long, Client> clientByClientId = listToMap(massGetClient(uniqClientIds), Client::getId);

        return EntryStream.of(uidToClientId)
                .mapValues(clientByClientId::get)
                .toMap();
    }

    @Nonnull
    public Set<ClientId> clientIdsWithApiEnabled(Collection<Long> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).stream()
                .mapValues(ids -> mapList(ids, ClientId::fromLong))
                .flatMapKeyValue(this::getClientsApiOptionsEnabledStatus)
                .toSet();
    }

    private Stream<ClientId> getClientsApiOptionsEnabledStatus(Integer shard, List<ClientId> ids) {
        Map<ClientId, ApiEnabled> clientsApiOptionsEnabledStatus =
                clientRepository.getClientsApiOptionsEnabledStatus(shard, ids);
        return ids.stream()
                .filter(id -> clientsApiOptionsEnabledStatus.get(id) != ApiEnabled.NO);
    }

    /**
     * Проверить имеет ли субклиент права на редактирование кампаний.
     * Проверяем сначала флаг в PPC.clients, после этого в rbac
     */
    public boolean isSuperSubclient(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        if (clientRepository.isSubclientAllowedToCreateCamps(shard, clientId)) {
            return true;
        }
        return rbacService.isSuperSubclient(clientId);
    }

    /**
     * Возвращает валюту клиента
     *
     * @param clientId client ID
     * @return {@link Currency}
     */
    public Currency getWorkCurrency(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return getWorkCurrency(shard, clientId);
    }

    public Currency getWorkCurrency(int shard, ClientId clientId) {
        ClientsWorkCurrency workCurrency = clientRepository.getWorkCurrency(shard, clientId);
        return Currencies.getCurrency(workCurrency.getLiteral());
    }

    @Nonnull
    public Map<ClientId, Currency> massGetWorkCurrency(Collection<ClientId> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong).stream()
                .map(e -> clientRepository.getWorkCurrencies(e.getKey(), e.getValue()))
                .flatMapToEntry(Function.identity())
                .mapValues(dbCurrency -> Currencies.getCurrency(dbCurrency.getLiteral()))
                .toMap();
    }

    /**
     * Вернуть объединенные данные по клиенту (данные из таблиц clients, clients_options, users)
     *
     * @param clientIds Идентификаторы клиентов по которым надо вернуть данные
     */
    @Nonnull
    public List<ClientWithUsers> massGetClientWithUsers(Collection<Long> clientIds) {
        List<ClientWithUsers> result = new ArrayList<>();

        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).getShardedDataMap()
                .forEach((shard, ids) -> {
                    List<ClientId> clIds = mapList(ids, ClientId::fromLong);
                    result.addAll(clientRepository.getClientData(shard, clIds));
                });

        return result;
    }

    /**
     * Вернуть объединенные данные по клиенту (данные из таблиц clients, clients_options, users)
     * См. также massGetClientWithUsers
     */
    public ClientWithUsers getClientWithUsers(ClientId clientId) {
        List<ClientWithUsers> clients = massGetClientWithUsers(singletonList(clientId.asLong()));

        checkArgument(!clients.isEmpty(), "no such client: {}", clientId);
        return clients.get(0);
    }

    /**
     * Получить объединенные данные по клиенту с заданной ролью
     *
     * @param role роль пользователя в Директе
     */
    public List<ClientWithUsers> getClientsWithUsersByRole(RbacRole role) {
        ClientsRole dbRole = RbacRole.toSource(role);
        return shardHelper.dbShards().stream()
                .flatMap(shard -> {
                    List<ClientId> clientIds =
                            clientRepository.getClientsWithRole(shard, dbRole);
                    return clientRepository.getClientData(shard, clientIds).stream();
                })
                .collect(Collectors.toList());
    }

    /**
     * Получить uid клиентов с заданными ролями
     *
     * @param roles роли пользователей в Директе
     */
    public List<Long> getClientChiefUidsByRoles(Collection<RbacRole> roles) {
        List<ClientsRole> dbRoles = mapList(roles, RbacRole::toSource);
        return shardHelper.dbShards().stream()
                .flatMap(shard -> clientRepository.getClientChiefUidsWithRoles(shard, dbRoles).stream())
                .collect(Collectors.toList());
    }

    /**
     * Вернуть страну клиента
     */
    @Nonnull
    public Optional<Long> getCountryRegionIdByClientId(ClientId clientId) {
        return clientRepository.getCountryRegionIdByClientId(
                shardHelper.getShardByClientId(clientId),
                clientId);
    }

    /**
     * Вернуть страну клиента.
     * <p>
     * В случае, если у клиента указан невалидный регион (отсутствует в нашем гео-дереве),
     * вернуть {@link Region#RUSSIA_REGION_ID} = 225
     */
    @Nonnull
    public Long getCountryRegionIdByClientIdStrict(ClientId clientId) {
        Long regionId = getCountryRegionIdByClientId(clientId).orElse(GLOBAL_REGION_ID);

        GeoTree geoTree = geoTreeFactory.getGlobalGeoTree();
        if (!geoTree.hasRegion(regionId)) {
            return RUSSIA_REGION_ID;
        }

        return regionId;
    }

    /**
     * Вернуть валюты в которых может платить клиент в разных странах
     *
     * @return Возвращает отображение идентификатора страны в коллекцию допустимых валют
     */
    @Nonnull
    public Multimap<Long, CurrencyCode> getAllowedForPayCurrencies(ClientId clientId) {
        return clientRepository.getAllowedForPayCurrencies(shardHelper.getShardByClientId(clientId), clientId);
    }

    /**
     * Зарегистрировать клиента в Директе (Заполнить необходимые  таблиц + привязать в Rbac-е)
     *
     * @param client Новый клиент
     * @return true в случае успеха, иначе false
     */
    boolean registerClient(ClientWithOptions client) {
        try {
            // Сохраняем клиента в базе
            saveNewClient(client, true);
            setClientGrants(client, shardHelper.getShardByClientIdStrictly(client.getClientId()));

            return true;
        } catch (RuntimeException e) {
            logger.error("Can't register new client", e);
            return false;
        }
    }

    /**
     * Зарегистрировать клиента в Директе (Заполнить необходимые  таблиц + привязать в Rbac-е)
     *
     * @param operatorUid       Uid инициатора операции
     * @param agencyReps        Представители агенства для которого регистрируется клиент
     * @param clientWithOptions Новый клиент
     * @return true в случае успеха, иначе false
     */
    boolean registerSubclient(
            Long operatorUid, Collection<UidAndClientId> agencyReps, ClientWithOptions clientWithOptions,
            Set<String> agencyEnabledFeatures) {
        try {
            // Сохраняем клиента в базе
            saveNewClient(clientWithOptions, false);

            // Связываем клиента с агенством
            CreateAgencySubclientStatus status = createAgencySubclients(
                    operatorUid,
                    agencyReps,
                    singleton(UidAndClientId.of(clientWithOptions.getUid(), clientWithOptions.getClientId())),
                    agencyEnabledFeatures)
                    .getOrDefault(clientWithOptions.getClientId(), CreateAgencySubclientStatus.INTERNAL_ERROR);

            if (status.isSuccess()) {
                // полномочия можно выставлять только после связывания клиента с агенством
                setClientGrants(clientWithOptions,
                        shardHelper.getShardByClientIdStrictly(clientWithOptions.getClientId()));
            }

            return status.isSuccess();
        } catch (RuntimeException e) {
            logger.error("Can't register new subclient", e);
            return false;
        }
    }

    /**
     * Сохранить клиента в базе директа
     */
    void saveNewClient(ClientWithOptions client, boolean addToFetchNds) {
        // Закремпляем shard за клиентом
        int shard = shardHelper.allocShardForNewClient(
                client.getUid(), client.getLogin(), client.getClientId().asLong());

        clientRepository.addClient(shard, client, addToFetchNds);
        userRepository.addClient(shard, client);
    }

    void setClientGrants(ClientWithOptions client, int shard) {
        clientRepository.setPerms(shard, client.getClientId(), client.getPerms());
        rbacService.clearCaches();
    }

    /**
     * поставить клиенту флаг "это продукт внутренней рекламы"
     */
    public void setClientInternalAdProductPerm(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        clientRepository.setPerms(shard, clientId, singleton(ClientPerm.INTERNAL_AD_PRODUCT));
        rbacService.clearCaches();
    }

    public void updateClientRole(ClientId clientId, RbacRole role, @Nullable RbacSubrole subrole) {
        checkRoleSubrole(role, subrole);
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        clientRepository.updateRole(shard, clientId, role, subrole);
        rbacService.clearCaches();
    }

    public Optional<ClientExperiment> getExperiment(int shard, Long clientId) {
        return clientRepository.getExperiment(shard, clientId);
    }

    public void saveClientExperiment(int shard, Long clientId, ClientExperiment experiment) {
        clientRepository.saveClientExperiment(shard, clientId, experiment);
    }

    /**
     * проверяем соответствие роли и под-роли
     */
    private void checkRoleSubrole(RbacRole role, @Nullable RbacSubrole subrole) {
        checkState(role != null, "Role must be defined");

        if (subrole != null) {
            checkState(role == ROLE_BY_SUBROLE.get(subrole), "Incorrect subrole: %s for role: %s", subrole, role);
        }
    }

    /**
     * Привязать клиентов к заданному агенству как субклиентов. Аналог rbac_create_agency_subclient
     * расширенный возможностью создавать несколько subclient-ов за один раз
     *
     * @return Отображение Id клиента в результат выполнения операции
     */
    Map<ClientId, CreateAgencySubclientStatus> createAgencySubclients(
            Long operatorUid, Collection<UidAndClientId> agencyReps, Set<UidAndClientId> clients, Set<String> agencyEnabledFeatures) {
        if (clients.isEmpty()) {
            return emptyMap();
        }
        ClientId agencyClientId = agencyReps.stream().findAny().map(UidAndClientId::getClientId).get();

        Map<ClientId, CreateAgencySubclientStatus> result = new HashMap<>(clients.size());

        // Здесь есть возможность для гонок. По хорошему проверку и привязку надо делать в одной транзакции, но
        // пока оставим как perl-ом коде. Плюс не очень понятно как это делать в случае нескольких шардов. Придется
        // вытаскивать их наверх, а это не очень желательно
        Set<ClientId> registeredClients = StreamEx.of(clients).map(UidAndClientId::getClientId).toSet();
        Set<ClientId> allowableToBindClients = agencyClientRelationService.getAllowableToBindClients(
                agencyClientId, registeredClients);
        if (allowableToBindClients.size() < registeredClients.size()) {
            for (ClientId clientId : Sets.difference(registeredClients, allowableToBindClients)) {
                result.put(clientId, CreateAgencySubclientStatus.NOT_ALLOWED_BIND_TO_AGENCY);
            }
        }

        agencyClientRelationService.bindClients(agencyClientId, allowableToBindClients);

        Set<ClientId> bindedClients;
        if (agencyEnabledFeatures.contains(FeatureName.NEW_LIM_REP_SCHEMA.getName())) {
            bindedClients = rbacService.bindClientsToAgencyAndReps(operatorUid, agencyReps, allowableToBindClients);
        } else {
            bindedClients = rbacService.bindClientsToAgency(operatorUid, agencyReps, allowableToBindClients);
        }
        if (bindedClients.size() < allowableToBindClients.size()) {
            for (ClientId clientId : Sets.difference(allowableToBindClients, bindedClients)) {
                result.put(clientId, CreateAgencySubclientStatus.RBAC_CANT_BIND_TO_AGENCY);
            }
        }

        for (ClientId clientId : bindedClients) {
            result.put(clientId, CreateAgencySubclientStatus.OK);
        }

        return result;
    }

    /**
     * @see CurrencyConvertQueueRepository#isBalanceSideConvertFinished(int, ClientId)
     */
    public void setBalanceSideConvertFinished(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        currencyConvertQueueRepository.setBalanceSideConvertFinished(shard, clientId);
    }

    /**
     * @see CurrencyConvertQueueRepository#isBalanceSideConvertFinished(int, ClientId)
     */
    public boolean isBalanceSideConvertFinished(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return currencyConvertQueueRepository.isBalanceSideConvertFinished(shard, clientId);
    }

    private Set<ClientId> getConvertingClients(Collection<ClientId> clientIds,
                                               Collection<CurrencyConversionState> excludeStates) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).stream()
                .map(shardedData -> currencyConvertQueueRepository.fetchConvertingClients(shardedData.getKey(),
                        shardedData.getValue(), stopOperationBeforeConvert, excludeStates))
                .toFlatCollection(Function.identity(), HashSet::new);
    }

    private Map<ClientId, Boolean> isClientConvertingSoon(Collection<ClientId> clientIds,
                                                          CurrencyConversionState... excludeState) {
        Map<ClientId, Boolean> result = Maps.newHashMapWithExpectedSize(clientIds.size());
        Set<ClientId> convertingClients = getConvertingClients(clientIds, Arrays.asList(excludeState));
        clientIds.forEach(clientId -> result.put(clientId, convertingClients.contains(clientId)));
        return result;
    }

    /**
     * Проверяет, будет ли клиент в ближайшее время конвертироваться (или уже конвертируется).
     * 'ближайшее время' определяется значением {@link #stopOperationBeforeConvert}
     */
    public boolean isClientConvertingSoonAndNotWaitingOverdraftNow(ClientId clientId) {
        return isClientConvertingSoon(singletonList(clientId), CurrencyConversionState.OVERDRAFT_WAITING)
                .get(clientId);
    }

    /**
     * @see ForceCurrencyConvertRepository#setClientCanBeForciblyConvertedToCurrency(int, ClientId)
     */
    public void forceCurrencyConversion(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        forceCurrencyConvertRepository.setClientCanBeForciblyConvertedToCurrency(shard, clientId);
    }

    /**
     * @see ClientRepository#isBusinessUnitClient(int, ClientId)
     */
    @Nullable
    public Boolean isBusinessUnitClient(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return clientRepository.isBusinessUnitClient(shard, clientId);
    }

    /**
     * @see ClientRepository#isBannedClient(int, ClientId)
     */
    @Nullable
    public Boolean isBannedClient(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return clientRepository.isBannedClient(shard, clientId);
    }

    /**
     * Применить изменение к клиенту
     * <p>
     * Метод достаёт из репозитория клиента, для которого подготовлен набор изменений {@link ModelChanges} и
     * применяет его,
     * получая объект {@link AppliedChanges}.  {@code clientId} по которому достаётся клиент определяется методом
     * {@link ModelChanges#getId()}
     * <p>
     * ! Метод не записывает изменения в базу.
     *
     * @param changes набор изменений для примененения к клиенту
     * @return применённые изменения или null, если не удалсь найти клиента
     */
    @Nonnull
    public AppliedChanges<Client> applyChanges(ModelChanges<Client> changes) {
        Client client = getClient(ClientId.fromLong(changes.getId()));
        if (client == null) {
            throw new IllegalStateException(String.format("Couldn't find client #%d", changes.getId()));
        }
        return changes.applyTo(client);
    }


    /**
     * Записывает применённые изменения в базу
     * <p>
     * Перед тем как вызывать этот метод нужно убедиться, что изменения в {@code appliedChanges} корректные
     *
     * @param appliedChanges применённые изменения
     */
    public void update(AppliedChanges<Client> appliedChanges) {
        massUpdate(singleton(appliedChanges));
    }

    /**
     * Записывает коллекцию применённых изменения в базу
     * <p>
     * Валидация в этом методе не проводится. Перед тем как вызывать этот метод нужно убедиться,
     * что изменения в {@code appliedChanges} корректные
     *
     * @param appliedChanges коллекция применённых изменения
     */
    public void massUpdate(Collection<AppliedChanges<Client>> appliedChanges) {
        shardHelper.groupByShard(appliedChanges, ShardKey.CLIENT_ID, ac -> ac.getModel().getId())
                .forEach((shard, changesForShard) -> {
                    Set<ClientId> clientsWithChangedHideMarketRating = appliedChanges.stream()
                            .filter(c -> c.changed(Client.HIDE_MARKET_RATING))
                            .map(c -> c.getModel().getId())
                            .map(ClientId::fromLong)
                            .collect(toSet());
                    clientRepository.update(shard, changesForShard);
                    // Если изменился флаг hide_market_rating, добавляем все кампании клиента на переотправку
                    if (!clientsWithChangedHideMarketRating.isEmpty()) {
                        bsResyncCampaignsByClients(shard, clientsWithChangedHideMarketRating);
                    }
                });
    }

    private void bsResyncCampaignsByClients(int shard, Collection<ClientId> clientsWithChangedHideMarketRating) {
        Set<Long> cids = campaignRepository.getCampaignIdsByClientIds(shard, clientsWithChangedHideMarketRating);
        List<BsResyncItem> resyncItems = cids.stream()
                .map(cid -> new BsResyncItem(BsResyncPriority.UPDATE_DOMAIN_RATINGS, cid))
                .collect(toList());
        bsResyncQueueRepository.addToResync(shard, resyncItems);
    }

    @Nonnull
    public Map<ClientId, BigDecimal> getOverdraftRestByClientIds(
            Collection<ClientId> clientIds, Map<ClientId, Currency> workCurrencies, Map<ClientId, Percent> nds) {
        // получаем из базы
        Map<ClientId, BigDecimal> overdraftRests = shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).stream()
                .map(e -> clientOptionsRepository.getOverdraftRestByClientIds(e.getKey(), e.getValue()))
                .flatMapToEntry(Function.identity())
                .toMap();

        clientIds.forEach(clientId -> overdraftRests.putIfAbsent(clientId, BigDecimal.ZERO));

        // в базе овердрафт и долг хранятся с НДС, а клиентам показываем без НДС
        BiFunction<ClientId, BigDecimal, BigDecimal> removeNds = (ClientId clientId, BigDecimal rest) -> {
            Currency currency = workCurrencies.get(clientId);
            Percent ndsPercentage = nds.get(clientId);
            if (ndsPercentage == null || currency == null || currency.getCode() == CurrencyCode.YND_FIXED) {
                return rest;
            } else {
                Money withNds = Money.valueOf(rest, currency.getCode());
                Money withoutNds = withNds.subtractNds(ndsPercentage);
                return withoutNds.bigDecimalValue();
            }
        };
        return EntryStream.of(overdraftRests).mapToValue(removeNds).toMap();
    }

    public List<ClientsOptions> massGetClientOptions(List<Long> clientIds) {
        List<ClientsOptions> result = new ArrayList<>();

        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).getShardedDataMap()
                .forEach((shard, ids) -> {
                    List<ClientId> clIds = mapList(ids, ClientId::fromLong);
                    result.addAll(clientOptionsRepository.getClientsOptions(shard, clIds));
                });

        return result;
    }

    public ClientsOptions getClientOptions(ClientId clientId) {
        List<ClientsOptions> clientsOptions = massGetClientOptions(singletonList(clientId.asLong()));

        checkArgument(!clientsOptions.isEmpty(), "no such clients options: {}", clientId);
        return clientsOptions.get(0);
    }

    public void updateClientOptions(ClientId clientId, long balanceTid, BigDecimal overdraftLim, BigDecimal debt,
                                    @Nullable LocalDate nextPayDate, @Nullable Boolean balanceBanned,
                                    Boolean nonResident, @Nullable Boolean isBusinessUnit, @Nullable Boolean isBrand) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        clientOptionsRepository.insertOrUpdateBalanceClientOptions(shard, clientId, balanceTid, overdraftLim, debt,
                nextPayDate, balanceBanned, nonResident, isBusinessUnit, isBrand);
    }

    public void updateClientCashBackBonus(ClientId clientId, BigDecimal cashBackBonus, BigDecimal awaitingBonus) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        clientOptionsRepository.updateCashBackBonus(shard, clientId, cashBackBonus, awaitingBonus);
    }

    public void updateSocialAdvertisingByClientIds(Collection<ClientId> clientIds, boolean socialAdvertising) {
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .forEach((shard, shardClientIds) -> {
                    clientOptionsRepository.updateSocialAdvertising(shard, shardClientIds, socialAdvertising);
                });
    }

    public boolean isYaAgencyClient(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        //если в clients_options нет записи для клиента, он не считается клиентом яндекс.агенства
        return clientOptionsRepository.getIsYaAgencyClientByClientId(shard, Collections.singleton(clientId))
                .getOrDefault(clientId, false);
    }

    public Map<Long, Long> getClientsManagerFromCampaigns(Collection<Long> clientIds) {
        Map<Long, Long> result = new HashMap<>();
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .stream()
                .mapKeyValue(campaignRepository::getClientManagerUidByClientId)
                .forEach(result::putAll);
        return result;
    }

    /**
     * Получить для переданных clientIds - список всех менеджеров из кампаний.
     * Если у клиента нет менеджерских кампаний, то в результирующей мапе id клиента не будет.
     */
    public Map<Long, Set<Long>> getClientsManagersFromCampaigns(Collection<ClientId> clientIds) {
        Map<Long, Set<Long>> result = new HashMap<>();
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong)
                .stream()
                .mapValues(ids -> mapList(ids, ClientId::asLong))
                .mapKeyValue(campaignRepository::getClientManagersUidByClientId)
                .forEach(result::putAll);
        return result;
    }

    public List<Long> getClientsOfManagerWithoutCampaigns(Long managerUid) {
        return clientManagersRepository.getClientsOfManagerWithoutCampaigns(managerUid);
    }

    public void deleteFromClientManagers(Collection<Long> clientIds, Long managerUid) {
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .stream()
                .forKeyValue((shard, chunk) ->
                        clientManagersRepository.deleteFromClientManagers(shard, chunk, managerUid));
    }

    /**
     * Вернуть clientId которые могут оплачивать общий счет до модерации
     *
     * @param clientIds Идентификаторы клиентов по которым надо вернуть данные
     */
    @Nonnull
    public Set<Long> clientIdsWithPayBeforeModeration(Collection<Long> clientIds) {
        Set<Long> result = new HashSet<>();

        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).getShardedDataMap()
                .forEach((shard, ids) -> {
                    List<ClientId> clIds = mapList(ids, ClientId::fromLong);
                    result.addAll(
                            clientRepository.get(shard, clIds)
                                    .stream()
                                    .filter(Client::getCanPayBeforeModeration)
                                    .map(Client::getClientId)
                                    .collect(toSet())
                    );
                });

        return result;
    }

    public void updateClientConnectOrgId(Long clientId, @Nullable Long connectOrgId) {
        ModelChanges<Client> modelChanges = new ModelChanges<>(clientId, Client.class);
        modelChanges.process(connectOrgId, Client.CONNECT_ORG_ID);
        AppliedChanges<Client> appliedChanges = applyChanges(modelChanges);
        update(appliedChanges);
    }

    public void bindClientsToManager(Long managerUid, List<ClientId> clientIds) {
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .forEach((shard, ids) -> clientManagersRepository.bindClientsToManager(shard, managerUid, clientIds));
    }

    /**
     * Получить для переданных clientIds - список счетчиков метрики выбранных по умолчанию.
     */
    public Map<Long, List<Long>> getClientsCommonMetrikaCounters(Collection<ClientId> clientIds) {
        Map<Long, List<Long>> result = new HashMap<>();
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong)
                .stream()
                .mapKeyValue(clientOptionsRepository::getCommonMetrikaCountersByClientId)
                .forEach(result::putAll);
        return result;
    }

    /**
     * Получить удаленных представителей для заданного клиента
     */
    public Set<DeletedClientRepresentative> getClientDeletedReps(ClientId clientId) {
        var client = getClient(clientId);
        var deletedReps = ifNotNull(client, Client::getDeletedReps);
        return (deletedReps == null || deletedReps.isBlank()) ? Set.of() : Set.of(JsonUtils.fromJson(deletedReps,
                DeletedClientRepresentative[].class));
    }

}
