package ru.yandex.direct.core.entity.client.service;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.repository.BsResyncQueueRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.client.service.validation.UpdateClientValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class ClientUpdateOperation extends SimpleAbstractUpdateOperation<Client, Long> {

    private final ClientRepository clientRepository;
    private final ClientService clientService;
    private final BsResyncQueueRepository bsResyncQueueRepository;
    private final CampaignRepository campaignRepository;
    private final UpdateClientValidationService updateClientValidationService;
    private final ShardHelper shardHelper;
    private final long operatorUid;
    private Collection<ClientId> clientIdsWithChangedHideMarketRating;

    public ClientUpdateOperation(Applicability applicability, List<ModelChanges<Client>> modelChanges,
                                 ClientRepository clientRepository,
                                 ClientService clientService, CampaignRepository campaignRepository,
                                 BsResyncQueueRepository bsResyncQueueRepository,
                                 UpdateClientValidationService updateClientValidationService,
                                 ShardHelper shardHelper, long operatorUid) {
        super(applicability, modelChanges, id -> new Client().withId(id));
        this.clientRepository = clientRepository;
        this.clientService = clientService;
        this.campaignRepository = campaignRepository;
        this.bsResyncQueueRepository = bsResyncQueueRepository;
        this.updateClientValidationService = updateClientValidationService;
        this.operatorUid = operatorUid;
        this.shardHelper = shardHelper;
    }

    @Override
    protected ValidationResult<List<ModelChanges<Client>>, Defect> validateModelChanges(
            List<ModelChanges<Client>> modelChanges) {
        return updateClientValidationService.preValidate(modelChanges, operatorUid);
    }

    @Override
    protected Collection<Client> getModels(Collection<Long> clientIds) {
        return clientService.massGetClient(mapList(clientIds, ClientId::fromLong));
    }

    @Override
    protected ValidationResult<List<Client>, Defect> validateAppliedChanges(
            ValidationResult<List<Client>, Defect> validationResult) {
        // пока валидация не потребовалась, но метод создан и вызывается при обновлении клиента
        return validationResult;
    }

    @Override
    protected void beforeExecution(ExecutionStep<Client> executionStep) {
        Collection<AppliedChanges<Client>> appliedChangesCollection = executionStep.getAppliedChangesForExecution();
        // Параметр необходим для переотправки в БК кампаний клиентов в случае изменения флага hide_market_rating
        clientIdsWithChangedHideMarketRating = appliedChangesCollection.stream()
                .filter(c -> c.changed(Client.HIDE_MARKET_RATING))
                .map(c -> c.getModel().getId())
                .map(ClientId::fromLong)
                .collect(toSet());
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<Client>> validAppliedChanges) {
        shardHelper.groupByShard(validAppliedChanges, ShardKey.CLIENT_ID, ac -> ac.getModel().getId())
                .forEach((shard, changesForShard) -> clientRepository.update(shard, changesForShard));
        return mapList(validAppliedChanges, a -> a.getModel().getId());
    }

    @Override
    protected void afterExecution(ExecutionStep<Client> executionStep) {
        // Если изменился флаг hide_market_rating, добавляем все кампании клиента на переотправку
        // (потому что отправляем этот параметр в БК на уровене кампании)
        if (!clientIdsWithChangedHideMarketRating.isEmpty()) {
            shardHelper.groupByShard(clientIdsWithChangedHideMarketRating, ShardKey.CLIENT_ID).getShardedDataMap()
                    .forEach(this::bsResyncCampaignsByClients);
        }
    }

    private void bsResyncCampaignsByClients(int shard, Collection<ClientId> clientIdsWithChangedHideMarketRating) {
        Set<Long> cids = campaignRepository.getCampaignIdsByClientIds(shard, clientIdsWithChangedHideMarketRating);
        List<BsResyncItem> resyncItems = cids.stream()
                .map(cid -> new BsResyncItem(BsResyncPriority.UPDATE_DOMAIN_RATINGS, cid))
                .collect(toList());
        // Метод update допускает 1_000 изменений за раз. У каждого клиента может быть до 3_000 кампаний.
        // Т.е. в наихудщем случае мы будем должны здесь поставить в очередь 3_000_000 сообщений.
        bsResyncQueueRepository.addToResync(shard, resyncItems);
    }
}
