package ru.yandex.direct.core.entity.client.service;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.ManagerHierarchyInfo;
import ru.yandex.direct.core.entity.client.repository.ManagerHierarchyRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Collections.singletonList;

@Service
@ParametersAreNonnullByDefault
public class ManagerHierarchyService {

    private final ShardHelper shardHelper;
    private final ManagerHierarchyRepository managerHierarchyRepository;

    @Autowired
    public ManagerHierarchyService(ShardHelper shardHelper, ManagerHierarchyRepository managerHierarchyRepository) {
        this.shardHelper = shardHelper;
        this.managerHierarchyRepository = managerHierarchyRepository;
    }

    @Nullable
    public ManagerHierarchyInfo getManagerData(ClientId clientId) {
        return Iterables.getFirst(massGetManagerData(singletonList(clientId)), null);
    }

    @Nonnull
    public List<ManagerHierarchyInfo> massGetManagerData(Collection<ClientId> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).stream()
                .flatMapKeyValue((shard, ids) -> managerHierarchyRepository.getManagerData(shard, ids).stream())
                .toList();
    }

    public void updateManagerData(Collection<AppliedChanges<ManagerHierarchyInfo>> appliedChanges) {
        shardHelper.groupByShard(appliedChanges, ShardKey.CLIENT_ID, ac -> ac.getModel().getId())
                .forEach(managerHierarchyRepository::updateManagerData);
    }

    public void deleteManagerData(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        managerHierarchyRepository.deleteManagerData(shard, clientId);
    }
}
