package ru.yandex.direct.core.entity.client.service;

import java.time.Instant;
import java.util.Base64;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.client.model.MediascopeClientMeasurerSettings;
import ru.yandex.direct.mediascope.MediascopeClient;
import ru.yandex.direct.mediascope.model.request.MediascopeTokensRequest;
import ru.yandex.direct.mediascope.model.response.MediascopePrefixResponse;
import ru.yandex.direct.mediascope.model.response.MediascopeTokensResponse;
import ru.yandex.direct.mediascope.model.response.MediascopeUserInfoResponse;
import ru.yandex.direct.utils.crypt.Encrypter;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@Component
@ParametersAreNonnullByDefault
public class MediascopeClientSettingsService {
    private static final Logger logger = LoggerFactory.getLogger(MediascopeClientSettingsService.class);
    private static final Base64.Encoder BASE64_ENCODER = Base64.getEncoder();
    private static final Base64.Decoder BASE64_DECODER = Base64.getDecoder();

    private final Encrypter encrypter;
    private final MediascopeClient mediascopeClient;

    @Autowired
    public MediascopeClientSettingsService(
            @Qualifier("MediascopeClientSettingsEncrypter") Encrypter encrypter,
            MediascopeClient mediascopeClient) {
        this.mediascopeClient = mediascopeClient;
        this.encrypter = encrypter;
    }

    public String extendSettings(@Nullable String settings) {
        if (settings == null) {
            return null;
        }

        var mediascopeSettings = getTokens(settings);
        if (mediascopeSettings == null) {
            // this method is called before validation, so, return settings as is,
            // later validator will check it and throw error
            return settings;
        }

        var prefixResponse = mediascopeClient.getPrefix(mediascopeSettings.getAccessToken());
        mediascopeSettings.withTmsecprefix(ifNotNull(prefixResponse, MediascopePrefixResponse::getPrefix));

        var userInfoResponse = mediascopeClient.getUserInfo(mediascopeSettings.getAccessToken());
        mediascopeSettings.withUsername(ifNotNull(userInfoResponse, MediascopeUserInfoResponse::getUsername));

        return toJson(mediascopeSettings);
    }

    public String encryptSettings(String settings) {
        var mediascopeSettings = fromJson(settings, MediascopeClientMeasurerSettings.class);
        mediascopeSettings
                .withAccessToken(toBase64(encrypter.encryptText(mediascopeSettings.getAccessToken())))
                .withRefreshToken(toBase64(encrypter.encryptText(mediascopeSettings.getRefreshToken())));
        return toJson(mediascopeSettings);
    }

    public String decryptSettings(String settings) {
        var mediascopeSettings = fromJson(settings, MediascopeClientMeasurerSettings.class);
        mediascopeSettings
                .withAccessToken(encrypter.decryptText(fromBase64(mediascopeSettings.getAccessToken())))
                .withRefreshToken(encrypter.decryptText(fromBase64(mediascopeSettings.getRefreshToken())));
        return toJson(mediascopeSettings);
    }

    public Boolean isExpired(String settings) {
        var mediascopeSettings = fromJson(settings, MediascopeClientMeasurerSettings.class);
        return mediascopeSettings.getExpiresAt() < Instant.now().getEpochSecond();
    }

    private MediascopeClientMeasurerSettings getTokens(String settings) {
        try {
            // new format
            var tokensRequest = fromJson(settings, MediascopeTokensRequest.class);
            if (tokensRequest == null || tokensRequest.getCode() == null || tokensRequest.getCodeVerifier() == null) {
                // old format
                return fromJson(settings, MediascopeClientMeasurerSettings.class);
            }

            var tokensResponse = mediascopeClient.getNewTokens(tokensRequest);
            if (tokensResponse == null) {
                return null;
            }

            return fromTokensResponse(tokensResponse);
        } catch (IllegalArgumentException e) {
            logger.error("Invalid client measurer settings for mediascope");
        }

        return null;
    }

    private static MediascopeClientMeasurerSettings fromTokensResponse(MediascopeTokensResponse response) {
        return new MediascopeClientMeasurerSettings()
                .withAccessToken(response.getAccessToken())
                .withRefreshToken(response.getRefreshToken())
                .withExpiresAt(response.getExpiresIn() + Instant.now().getEpochSecond());
    }

    private static String toBase64(String str) {
        return new String(BASE64_ENCODER.encode(str.getBytes()));
    }

    private static String fromBase64(String str) {
        return new String(BASE64_DECODER.decode(str.getBytes()));
    }
}
