package ru.yandex.direct.core.entity.client.service.checker;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.stereotype.Service;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;

@Service
public class ClientAccessCheckerValidationService {

    ClientAccessCheckerTypeSupportFacade clientAccessCheckerTypeSupportFacade;

    public ClientAccessCheckerValidationService(
            ClientAccessCheckerTypeSupportFacade clientAccessCheckerTypeSupportFacade) {
        this.clientAccessCheckerTypeSupportFacade = clientAccessCheckerTypeSupportFacade;
    }

    /**
     * Метод валидации при проверке объектов на принадлежность клиента
     *
     * @param objects список объектов, подлежащих проверке
     * @param clientId    id клиента
     * @param objectClass тип объектов
     * @return результат основной валидации
     */
    public ValidationResult<List<Long>, Defect> validateModels(List<? extends ModelWithId> objects,
                                                               ClientId clientId,
                                                               Class<? extends ModelWithId> objectClass) {
        if (objects.isEmpty()) {
            return ValidationResult.success(Collections.emptyList());
        }
        List<Long> objectIds = objects.stream()
                .map(ModelWithId::getId)
                .collect(Collectors.toList());
        return validateIds(objectIds, clientId, objectClass);
    }

    /**
     * Основной метод валидации при проверке объектов на принадлежность клиенту
     *
     * @param objectIds список id объектов, подлежащих проверке
     * @param clientId       id клиента
     * @param objectClass    тип объектов
     * @return результат валидации
     */
    public ValidationResult<List<Long>, Defect> validateIds(List<Long> objectIds,
                                                            ClientId clientId,
                                                            Class<? extends ModelWithId> objectClass) {
        if (objectIds.isEmpty()) {
            return ValidationResult.success(Collections.emptyList());
        }
        ListValidationBuilder<Long, Defect> lvb = ListValidationBuilder.of(objectIds);
        Set<Long> valid = clientAccessCheckerTypeSupportFacade.sendToCheck(Map.of(objectClass, objectIds),
                clientId).get(objectClass);
        lvb.checkEach(inSet(valid));
        return lvb.getResult();
    }
}
