package ru.yandex.direct.core.entity.client.service.validation;

import javax.annotation.Nonnull;

import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.currency.service.CurrencyService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.LoginOrUid;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Service
@Lazy
public class AddClientValidationService {
    private final CurrencyService currencyService;

    public AddClientValidationService(CurrencyService currencyService) {
        this.currencyService = currencyService;
    }

    @Nonnull
    public ValidationResult<Object, Defect> validateAddClientRequest(LoginOrUid loginOrUid, String name,
                                                                     String surname, long country, CurrencyCode currencyCode) {
        ItemValidationBuilder<Object, Defect> ivb = ModelItemValidationBuilder.of(new Object());
        ivb.item(name, "name")
                .check(notNull())
                .check(notBlank());
        ivb.item(surname, "surname")
                .check(notNull())
                .check(notBlank());

        ivb.checkBy(o -> checkCommonFields(o, country, currencyCode, loginOrUid));

        return ivb.getResult();
    }

    @Nonnull
    public ValidationResult<Object, Defect> validateAddClientRequest(LoginOrUid loginOrUid, String fio,
                                                                     long country, CurrencyCode currencyCode) {
        ItemValidationBuilder<Object, Defect> ivb = ModelItemValidationBuilder.of(new Object());
        ivb.item(fio, "fio")
                .check(notNull())
                .check(notBlank());

        ivb.checkBy(o -> checkCommonFields(o, country, currencyCode, loginOrUid));

        return ivb.getResult();
    }

    ValidationResult<Object, Defect> checkCommonFields(Object o, long country, CurrencyCode currencyCode,
                                                       LoginOrUid loginOrUid) {
        ItemValidationBuilder<Object, Defect> ivb = ModelItemValidationBuilder.of(o);

        if (loginOrUid.isLogin()) {
            ivb.item(loginOrUid.getLogin(), "login")
                    .check(notBlank());
        } else {
            ivb.item(loginOrUid.getUid(), "uid")
                    .check(validId());
        }

        ivb.item(country, "country").check(validId());
        ivb.item(currencyCode, "currency").check(notNull());
        //валидация на балансовые валюты внутри validateExistingClientCountryCurrency
        ivb.item(currencyCode, "currency").check(
                Constraint.fromPredicate(t -> isValidCountryCurrencyInterconnection(null, t, country),
                        invalidValue()), When.isValid());

        return ivb.getResult();
    }

    @Nonnull
    public ValidationResult<Object, Defect> validateCountryCurrencyInterconnections(ClientId clientId,
                                                                                    CurrencyCode currencyCode,
                                                                                    long country) {
        ItemValidationBuilder<Object, Defect> ivb = ModelItemValidationBuilder.of(new Object());

        ivb.item(currencyCode, "currency").check(
                Constraint.fromPredicate(t -> isValidCountryCurrencyInterconnection(clientId, t, country),
                        invalidValue()), When.isValid());

        return ivb.getResult();
    }

    private boolean isValidCountryCurrencyInterconnection(ClientId clientId, CurrencyCode currencyCode, long country) {
        return currencyService
                .getAllowedCountryCurrenciesForClient(clientId, null)
                .getOrDefault(country, emptySet())
                .contains(currencyCode);
    }
}
