package ru.yandex.direct.core.entity.client.service.validation;

import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.client.exception.NoAvailableCurrenciesException;
import ru.yandex.direct.core.entity.currency.service.CurrencyService;
import ru.yandex.direct.core.validation.defects.MoneyDefects;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;

import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;

/**
 * Валидатор валюты клиента агентства
 */
@ParametersAreNonnullByDefault
public final class AgencyClientCurrencyValidator implements DefaultValidator<CurrencyCode> {
    private final CurrencyService currencyService;
    private final ClientId agencyClientId;

    public AgencyClientCurrencyValidator(CurrencyService currencyService, ClientId agencyClientId) {
        Objects.requireNonNull(currencyService, "currencyService");
        Objects.requireNonNull(agencyClientId, "agencyClientId");

        this.currencyService = currencyService;
        this.agencyClientId = agencyClientId;
    }

    @Override
    public ValidationResult<CurrencyCode, Defect> apply(CurrencyCode currency) {
        Set<CurrencyCode> validCurrencies = currencyService.getAllowedCurrenciesForAgencyClient(
                null,
                agencyClientId);

        if (validCurrencies.isEmpty()) {
            throw new NoAvailableCurrenciesException();
        }

        Defect unavailableCurrency = MoneyDefects.unavailableCurrency(currency, validCurrencies);

        return ItemValidationBuilder.<CurrencyCode, Defect>of(currency)
                .check(inSet(validCurrencies), unavailableCurrency)
                .getResult();
    }
}
