package ru.yandex.direct.core.entity.client.service.validation;

import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.client.service.validation.ClientDefects.mustNotBeFreelancer;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userAssociatedWithAnotherClient;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userIsBlocked;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userIsNotRepresentative;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userNotFound;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;

@Service
public class ClientChiefValidationService {

    private final FreelancerService freelancerService;
    private final BalanceService balanceService;

    public ClientChiefValidationService(FreelancerService freelancerService, BalanceService balanceService) {
        this.freelancerService = freelancerService;
        this.balanceService = balanceService;
    }

    public ValidationResult<User, Defect> validateChangeChief(@Nullable User user, ClientId clientId,
                                                              List<Long> allRepresentativeUids) {
        if (user == null) {
            return ValidationResult.failed(null, userNotFound());
        }

        Set<Long> allRepresentatives = new HashSet<>(allRepresentativeUids);
        ModelItemValidationBuilder<User> vb = ModelItemValidationBuilder.of(user);
        vb.item(User.UID)
                .check(inSet(allRepresentatives), userIsNotRepresentative())
                .check(userAssociatedWithClient(clientId));
        vb.item(User.STATUS_BLOCKED)
                .check(userNotBlocked());
        vb.item(User.CLIENT_ID)
                .check(clientMustNotBeFreelancer());
        return vb.getResult();
    }

    private static Constraint<Boolean, Defect> userNotBlocked() {
        return fromPredicate(statusBlocked -> !statusBlocked, userIsBlocked());
    }

    private Constraint<ClientId, Defect> clientMustNotBeFreelancer() {
        return fromPredicate(clientId -> !freelancerService.isFreelancer(clientId), mustNotBeFreelancer());
    }

    private Constraint<Long, Defect> userAssociatedWithClient(ClientId expectedClientId) {
        return fromPredicate(
                uid -> Objects.equals(balanceService.getClientIdByUid(uid), expectedClientId),
                userAssociatedWithAnotherClient());
    }

}
