package ru.yandex.direct.core.entity.client.service.validation;

import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientPrimaryManager;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.client.service.validation.ClientDefects.notAppropriateRole;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userMustBeManager;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userMustBeManagerInBalance;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userNotFound;
import static ru.yandex.direct.core.validation.constraints.Constraints.validClientId;
import static ru.yandex.direct.utils.CommonUtils.notEquals;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.result.ValidationResult.failed;
import static ru.yandex.direct.validation.result.ValidationResult.success;

@Service
@ParametersAreNonnullByDefault
public class PrimaryManagerValidationService {

    private static final Set<RbacRole> CLIENT_REQUIRED_ROLES = Set.of(RbacRole.AGENCY, RbacRole.CLIENT);

    private final ClientService clientService;
    private final UserService userService;
    private final BalanceService balanceService;

    @Autowired
    public PrimaryManagerValidationService(ClientService clientService, UserService userService,
                                           BalanceService balanceService) {
        this.clientService = clientService;
        this.userService = userService;
        this.balanceService = balanceService;
    }

    public ValidationResult<ClientPrimaryManager, Defect> updateValidation(ClientPrimaryManager manager) {
        ModelItemValidationBuilder<ClientPrimaryManager> vb = ModelItemValidationBuilder.of(manager);

        vb.item(ClientPrimaryManager.IS_IDM_PRIMARY_MANAGER)
                .check(notNull());

        vb.item(ClientPrimaryManager.SUBJECT_CLIENT_ID)
                .check(notNull())
                .check(validClientId())
                .checkBy(getClientValidator(), When.isValid());

        vb.item(ClientPrimaryManager.PRIMARY_MANAGER_UID)
                .check(validId())
                .checkBy(getManagerValidator(), When.isValid())
                .checkBy(getBalanceManagerValidator(), When.isValid());

        return vb.getResult();
    }

    private Validator<ClientId, Defect> getClientValidator() {
        return clientId -> {
            Client client = Optional.of(clientId)
                    .map(clientService::getClient)
                    .orElse(null);
            if (client == null) {
                return failed(clientId, CommonDefects.objectNotFound());
            }
            if (client.getRole() == null || !CLIENT_REQUIRED_ROLES.contains(client.getRole())) {
                return failed(clientId, notAppropriateRole());
            }
            return success(clientId);
        };
    }

    private Validator<Long, Defect> getManagerValidator() {
        return managerUid -> {
            if (managerUid == null) {
                return success(managerUid);
            }
            User user = userService.getUser(managerUid);
            if (user == null) {
                return failed(managerUid, userNotFound());
            }
            if (notEquals(user.getRole(), RbacRole.MANAGER)) {
                return failed(managerUid, userMustBeManager());
            }
            return success(managerUid);
        };
    }

    private Validator<Long, Defect> getBalanceManagerValidator() {
        return managerUid -> Optional.ofNullable(managerUid)
                .map(balanceService::checkManagerExist)
                .filter(isExist -> !isExist)
                .map(l -> failed(managerUid, userMustBeManagerInBalance()))
                .orElse(success(managerUid));
    }

}
