package ru.yandex.direct.core.entity.clientphone;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция обновляет клиентские номера client_phones, которые используются для коллтрекинга на сайте.
 * Это такие номера, у которых phoneType = TELEPHONY, permalinkId = null.
 */
public class CalltrackingOnSitePhoneUpdateOperation extends SimpleAbstractUpdateOperation<ClientPhone, Long> {

    private final ClientPhoneRepository clientPhoneRepository;
    private final ModelChangesValidationTool updateValidationTool;

    private final int shard;
    private final ClientId clientId;

    CalltrackingOnSitePhoneUpdateOperation(
            int shard,
            ClientId clientId,
            List<ModelChanges<ClientPhone>> modelChanges,
            ClientPhoneRepository clientPhoneRepository) {
        super(Applicability.PARTIAL, modelChanges, id -> new ClientPhone().withId(id));
        this.shard = shard;
        this.clientId = clientId;
        this.clientPhoneRepository = clientPhoneRepository;
        updateValidationTool = ModelChangesValidationTool.builder().build();
    }

    @Override
    protected ValidationResult<List<ModelChanges<ClientPhone>>, Defect> validateModelChanges(
            List<ModelChanges<ClientPhone>> modelChanges
    ) {
        List<Long> phoneIds = mapList(modelChanges, ModelChanges::getId);
        List<ClientPhone> byPhoneIds = clientPhoneRepository.getByPhoneIds(clientId, phoneIds);
        Set<Long> existedIds = listToSet(byPhoneIds, ClientPhone::getId);
        return updateValidationTool.validateModelChangesList(modelChanges, existedIds);
    }

    @Override
    protected Collection<ClientPhone> getModels(Collection<Long> phoneIds) {
        return clientPhoneRepository.getByPhoneIds(clientId, phoneIds);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<ClientPhone>> appliedChanges) {
        clientPhoneRepository.update(shard, appliedChanges);
        return mapList(appliedChanges, c -> c.getModel().getId());
    }
}
