package ru.yandex.direct.core.entity.clientphone;

import java.util.List;
import java.util.Map;
import java.util.function.Function;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.clientphone.validation.ClientPhoneValidationService;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.operationwithid.AbstractOperationWithId;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

public class ClientPhoneDeleteOperation extends AbstractOperationWithId {

    private final ClientPhoneRepository clientPhoneRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final ClientId clientId;
    private final int shard;

    public ClientPhoneDeleteOperation(ClientId clientId,
                                      int shard,
                                      List<Long> phoneIds,
                                      ClientPhoneRepository clientPhoneRepository,
                                      BannerCommonRepository bannerCommonRepository) {
        super(Applicability.PARTIAL, phoneIds);
        this.clientPhoneRepository = clientPhoneRepository;
        this.clientId = clientId;
        this.shard = shard;
        this.bannerCommonRepository = bannerCommonRepository;
    }

    @Override
    protected ValidationResult<List<Long>, Defect> validate(List<Long> clientPhoneIds) {
        List<ClientPhone> existingClientPhones = clientPhoneRepository.getByPhoneIds(clientId, clientPhoneIds);
        return ClientPhoneValidationService.validateForDelete(clientPhoneIds, existingClientPhones);
    }

    @Override
    protected void execute(List<Long> clientPhoneIds) {
        clientPhoneRepository.delete(clientId, clientPhoneIds);
    }

    @Override
    protected void afterExecution(List<Long> clientPhoneIds) {
        unlinkBannerPhoneIds(shard, clientPhoneIds);
        unlinkCampaignPhoneIds(shard, clientPhoneIds);
    }

    private void unlinkBannerPhoneIds(int shard, List<Long> clientPhoneIds) {
        Map<Long, List<Long>> bannerIdsByPhoneId =
                clientPhoneRepository.getBannerIdsByPhoneId(shard, clientPhoneIds);
        List<Long> bannerIds = StreamEx.of(bannerIdsByPhoneId.values())
                .flatCollection(Function.identity())
                .toList();
        clientPhoneRepository.unlinkBannerPhonesByBannerId(shard, bannerIds);
        bannerCommonRepository.resetStatusBsSyncedByIds(shard, bannerIds);
    }

    /**
     * При отвязке телефонов от кампании сброс statusBsSynced не нужен, т.к. в БК отправляется только
     * телефон с баннера
     */
    private void unlinkCampaignPhoneIds(int shard, List<Long> clientPhoneIds) {
        Map<Long, List<Long>> campaignIdsByPhoneId =
                clientPhoneRepository.getCampaignIdsByPhoneId(shard, clientPhoneIds);
        List<Long> campaignIds = StreamEx.of(campaignIdsByPhoneId.values())
                .flatCollection(Function.identity())
                .toList();
        clientPhoneRepository.unlinkCampaignPhonesByCampaignId(shard, campaignIds);
    }

}
