package ru.yandex.direct.core.entity.clientphone;

import java.util.Set;

import javax.annotation.Nullable;

import com.google.i18n.phonenumbers.NumberParseException;
import com.google.i18n.phonenumbers.PhoneNumberUtil;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.vcard.model.Phone;
import ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.vcard.repository.VcardMappings.normalizePhoneNumber;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

public class ClientPhoneFormatter {

    private static final int DEFAULT_COUNTRY_CODE_LENGTH = 2;
    private static final int DEFAULT_CITY_CODE_LENGTH = 3;
    private static final int BY_CITY_CODE_LENGTH = 2;
    private static final int UA_CITY_CODE_LENGTH = 2;
    private static final int UZ_CITY_CODE_LENGTH = 2;

    /**
     * В России города, телефоны которых начинаются с {@link ClientPhoneFormatter#RU_3_CITY_CODE_PREFIXES},
     * имеют 3-х буквенные коды городов. Среди оставшихся большинство городов имеет 4-х буквенные,
     * совсем мелкие города и поселки 5-и буквенные. Пока считаем, что города за исключением
     * {@link ClientPhoneFormatter#RU_3_CITY_CODE_PREFIXES}, имеют 4-х буквенные коды.
     */
    private static final int DEFAULT_RU_CITY_CODE_LENGTH = 4;
    private static final int RU_3_CITY_CODE_LENGTH = 3;
    private static final int RU_MOBILE_CODE_LENGTH = 3;
    /**
     * Префиксы кодов городов России, которые означают, что телефон нужно отформатировть с 3-х буквенным кодом
     *
     * <a href="https://ru.wikipedia.org/wiki/Телефонный_план_нумерации_России">Телефонный_план_нумерации_России</a>
     * Раздел "Семизначные номера"
     */
    private static final Set<String> RU_3_CITY_CODE_PREFIXES = Set.of(
            "800", //Бесплатная линия
            "495", "499", //Москва
            "812", //Санкт-Петербург
            "496", //Сергиев Посад
            "343", //Екатеринбург
            "861", //Краснодар
            "863", //Ростов-на-Дону
            "351", //Челябинск
            "383", //Новосибирск
            "846", //Самара
            "342", //Пермь
            "843", //Казань
            "347", //Уфа
            "831", //Нижний Новгород
            "391", //Красноярск
            "473", //Воронеж
            "423", //Владивосток
            "862", //Сочи
            "48439", //Обнинск
            "48458" //Обнинск
    );

    /**
     * Большинство всех телефонов -- российские.
     * Этот регион используется для библиотеки libphonenumber, которая при передаче неверного региона
     * все равно верно определит регион. Так что используется как заглушка.
     */
    private static final String DEFAULT_REGION = "RU";
    private static final Set<String> KZ_PHONE_PREFIXES = Set.of("+76", "+77", "+86", "+87");
    private static final Set<String> RU_PHONE_PREFIXES = Set.of("+7", "+8");
    private static final String BY_COUNTRY_CODE = "+375";
    private static final String UA_COUNTRY_CODE = "+380";
    private static final String UZ_COUNTRY_CODE = "+998";

    public static Phone format(String fullPhone, @Nullable Long extension) {
        checkState(
                fullPhone.length() >= PhoneValidator.ENTIRE_PHONE_MIN_LENGTH,
                "The phone number must be at least 8 symbols"
        );
        if (!fullPhone.startsWith("+")) {
            fullPhone = "+" + fullPhone;
        }
        String countryCode = "+" + extractCountryCode(fullPhone);
        int cityCodeBeginIndex = countryCode.length();
        String cityCode = extractCityCode(fullPhone, countryCode, cityCodeBeginIndex);
        String phone = normalizePhoneNumber(fullPhone.substring(cityCodeBeginIndex + cityCode.length()));
        return new Phone()
                .withCityCode(cityCode)
                .withCountryCode(countryCode)
                .withPhoneNumber(phone)
                .withExtension(ifNotNull(extension, String::valueOf));
    }

    private static String extractCountryCode(String phoneNumber) {
        PhoneNumberUtil instance = PhoneNumberUtil.getInstance();
        try {
            var parsedPhone = instance.parse(phoneNumber, DEFAULT_REGION);
            return String.valueOf(parsedPhone.getCountryCode());
        } catch (NumberParseException e) {
            // Начинаем с 1, т.к. первый символ -- знак "+"
            return phoneNumber.substring(1, DEFAULT_COUNTRY_CODE_LENGTH);
        }
    }

    private static String extractCityCode(String phone, String countryCode, int cityCodeBeginIndex) {
        int cityCodeLength = getCityCodeLength(phone, countryCode);
        return phone.substring(cityCodeBeginIndex, cityCodeBeginIndex + cityCodeLength);
    }

    private static int getCityCodeLength(String phone, String countryCode) {
        if (RU_PHONE_PREFIXES.contains(countryCode)) {
            if (StreamEx.of(KZ_PHONE_PREFIXES).anyMatch(phone::startsWith)) {
                return DEFAULT_CITY_CODE_LENGTH;
            }
            if (StreamEx.of(RU_PHONE_PREFIXES).anyMatch(phone::startsWith)) {
                return findRuCityCodeLength(phone);
            }
        } else if (BY_COUNTRY_CODE.equals(countryCode)) {
            return BY_CITY_CODE_LENGTH;
        } else if (UA_COUNTRY_CODE.equals(countryCode)) {
            return UA_CITY_CODE_LENGTH;
        } else if (UZ_COUNTRY_CODE.equals(countryCode)) {
            return UZ_CITY_CODE_LENGTH;
        }
        return DEFAULT_CITY_CODE_LENGTH;
    }

    private static int findRuCityCodeLength(String phone) {
        // С 9-ки начинаются мобильные номера
        for (String prefix : RU_PHONE_PREFIXES) {
            if (phone.startsWith("9", prefix.length())) {
                return RU_MOBILE_CODE_LENGTH;
            }
            for (String code : RU_3_CITY_CODE_PREFIXES) {
                if (phone.startsWith(code, prefix.length())) {
                    return RU_3_CITY_CODE_LENGTH;
                }
            }
        }
        return DEFAULT_RU_CITY_CODE_LENGTH;
    }
}
