package ru.yandex.direct.core.entity.clientphone;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.banner.service.DatabaseMode;
import ru.yandex.direct.core.entity.campaign.model.TextCampaign;
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService;
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions;
import ru.yandex.direct.core.entity.campaign.service.RestrictedCampaignsUpdateOperation;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;

@Service
public class ClientPhoneReplaceService {

    private static final Logger logger = LoggerFactory.getLogger(ClientPhoneReplaceService.class);
    private static final int REPLACE_CHUNK_SIZE = 1000;

    private final ShardHelper shardHelper;
    private final ClientPhoneRepository clientPhoneRepository;
    private final OrganizationRepository organizationRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final OrganizationService organizationService;
    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final CampaignOperationService campaignOperationService;

    public ClientPhoneReplaceService(
            ShardHelper shardHelper,
            ClientPhoneRepository clientPhoneRepository,
            OrganizationRepository organizationRepository,
            BannerCommonRepository bannerCommonRepository,
            OrganizationService organizationService,
            BannersUpdateOperationFactory bannersUpdateOperationFactory,
            CampaignOperationService campaignOperationService
    ) {
        this.shardHelper = shardHelper;
        this.clientPhoneRepository = clientPhoneRepository;
        this.organizationRepository = organizationRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.organizationService = organizationService;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.campaignOperationService = campaignOperationService;
    }

    /**
     * Заменить телефоны {@code phoneIds} на {@code replacePhoneId}. Если какой-то из телефонов привязан
     * к организации, для которой нет прав, то для баннеров и кампаний с этим телефоном телефон будет просто отвязан.
     */
    public MassResult<Long> replaceTrackingPhone(
            UidAndClientId uidAndClientId,
            Long operatorUid,
            Collection<Long> phoneIds,
            Long replacePhoneId
    ) {
        ClientId clientId = uidAndClientId.getClientId();
        int shard = shardHelper.getShardByClientId(clientId);
        Map<Long, Long> permalinkIdsByBannerIds = getPermalinkIdsByBannerIds(shard, clientId, phoneIds);
        Map<Long, Long> permalinkIdsByCampaignIds = getPermalinkIdsByCampaignIds(shard, phoneIds);
        if (permalinkIdsByBannerIds.isEmpty() && permalinkIdsByCampaignIds.isEmpty()) {
            return MassResult.successfulMassAction(emptyList(), ValidationResult.success(emptyList()));
        }

        List<Long> permalinksToUpdate = new ArrayList<>(permalinkIdsByBannerIds.values());
        permalinksToUpdate.addAll(permalinkIdsByCampaignIds.values());

        var permalinkToHasAccess = organizationService.hasAccess(clientId, permalinksToUpdate);

        MassResult<Long> bannerResult = updateBannerTrackingPhones(clientId, operatorUid, permalinkIdsByBannerIds,
                permalinkToHasAccess,
                replacePhoneId);
        if (bannerResult.getValidationResult().hasAnyErrors()) {
            var defects = bannerResult.getValidationResult().flattenErrors();
            logger.error("Failed to replace phones {} to phone {} for banners, defects: {}",
                    phoneIds, replacePhoneId, defects);
        }

        MassResult<Long> campaignResult = updateCampaignTrackingPhones(uidAndClientId, operatorUid,
                permalinkIdsByCampaignIds, permalinkToHasAccess,
                replacePhoneId);
        if (campaignResult.getValidationResult().hasAnyErrors()) {
            var defects = campaignResult.getValidationResult().flattenErrors();
            logger.error("Failed to replace phones {} to phone {} for campaigns, defects: {}",
                    phoneIds, replacePhoneId, defects);
        }
        // Осознанно не возвращаем {@code campaignResult},
        // т.к. почти никогда не ожидаем ошибок валидации в этом методе
        return bannerResult;
    }

    /**
     * Заменить телефоны {@code phoneIds} на {@code replacePhoneId}. Если какой-то из телефонов привязан
     * к организации, для которой нет прав, то для баннеров и кампаний с этим телефоном телефон будет просто отвязан.
     * <p>
     * (!) В отличие от {@link #replaceTrackingPhone(UidAndClientId, Long, Collection, Long)} замена происходит
     * НЕ через операцию, соответственно, здесь отсутствует какая-либо валидация
     */
    public void replaceTrackingPhoneWithoutValidation(
            ClientId clientId,
            Collection<Long> phoneIds,
            Long replacePhoneId
    ) {
        int shard = shardHelper.getShardByClientId(clientId);
        Map<Long, Long> permalinkIdsByBannerIds = getPermalinkIdsByBannerIds(shard, clientId, phoneIds);
        Map<Long, Long> permalinkIdsByCampaignIds = getPermalinkIdsByCampaignIds(shard, phoneIds);
        if (permalinkIdsByBannerIds.isEmpty() && permalinkIdsByCampaignIds.isEmpty()) {
            return;
        }

        List<Long> permalinksToUpdate = new ArrayList<>(permalinkIdsByBannerIds.values());
        permalinksToUpdate.addAll(permalinkIdsByCampaignIds.values());

        var permalinkToHasAccess = organizationService.hasAccess(clientId, permalinksToUpdate);

        Map<Long, Long> bannerIdToNewPhoneId = EntryStream.of(permalinkIdsByBannerIds)
                .mapValues(permalink -> getReplacePhoneId(permalinkToHasAccess, permalink, replacePhoneId))
                .toMap();

        Map<Long, Long> campaignIdToNewPhoneId = EntryStream.of(permalinkIdsByCampaignIds)
                .mapValues(permalink -> getReplacePhoneId(permalinkToHasAccess, permalink, replacePhoneId))
                .toMap();
        replaceForBanners(shard, bannerIdToNewPhoneId);
        replaceForCampaigns(shard, campaignIdToNewPhoneId);
    }

    private Map<Long, Long> getPermalinkIdsByBannerIds(int shard, ClientId clientId, Collection<Long> phoneIds) {
        Map<Long, List<Long>> bannerIdsByPhoneId = clientPhoneRepository.getBannerIdsByPhoneId(shard, phoneIds);
        List<Long> bannerIdsToUpdate = StreamEx.ofValues(bannerIdsByPhoneId).flatMap(StreamEx::of).toList();
        if (bannerIdsToUpdate.isEmpty()) {
            return emptyMap();
        }
        return organizationService.getPermalinkIdsByBannerIds(clientId, bannerIdsToUpdate);
    }

    private Map<Long, Long> getPermalinkIdsByCampaignIds(int shard, Collection<Long> phoneIds) {
        Map<Long, List<Long>> campaignIdsByPhoneId =
                clientPhoneRepository.getCampaignIdsByPhoneId(shard, phoneIds);
        if (campaignIdsByPhoneId.isEmpty()) {
            return emptyMap();
        }
        List<Long> campaignIdsToUpdate = StreamEx.ofValues(campaignIdsByPhoneId).flatMap(StreamEx::of).toList();
        return organizationRepository.getDefaultPermalinkIdsByCampaignId(shard, campaignIdsToUpdate);
    }

    private MassResult<Long> updateBannerTrackingPhones(ClientId clientId, Long operatorUid,
                                                        Map<Long, Long> permalinkIdsByBannerIds,
                                                        Map<Long, Boolean> permalinkToHasAccess,
                                                        Long replacePhoneId) {
        List<ModelChanges<BannerWithSystemFields>> modelChanges = EntryStream.of(permalinkIdsByBannerIds)
                .mapKeyValue((bid, permalink) -> {
                    var phoneId = getReplacePhoneId(permalinkToHasAccess, permalink, replacePhoneId);
                    return ModelChanges.build(bid, TextBanner.class, TextBanner.PHONE_ID, phoneId)
                            .castModelUp(BannerWithSystemFields.class);
                })
                .toList();
        return bannersUpdateOperationFactory
                .createFullUpdateOperation(modelChanges, operatorUid, clientId, DatabaseMode.ONLY_MYSQL)
                .prepareAndApply();
    }

    private MassResult<Long> updateCampaignTrackingPhones(UidAndClientId uidAndClientId,
                                                          Long operatorUid,
                                                          Map<Long, Long> permalinkIdsByCampaignIds,
                                                          Map<Long, Boolean> permalinkToHasAccess,
                                                          Long replacePhoneId) {
        List<ModelChanges<TextCampaign>> modelChanges = EntryStream.of(permalinkIdsByCampaignIds)
                .mapKeyValue((cid, permalink) -> {
                    var phoneId = getReplacePhoneId(permalinkToHasAccess, permalink, replacePhoneId);
                    return ModelChanges.build(cid, TextCampaign.class,
                            TextCampaign.DEFAULT_TRACKING_PHONE_ID, phoneId);
                })
                .toList();
        var options = new CampaignOptions();
        RestrictedCampaignsUpdateOperation updateOperation =
                campaignOperationService.createRestrictedCampaignUpdateOperation(
                        modelChanges,
                        operatorUid,
                        uidAndClientId,
                        options
                );
        return updateOperation.apply();
    }

    /**
     * Получить номер телефона для замены
     * Если к организации нет прав, то {@code null}
     */
    private Long getReplacePhoneId(Map<Long, Boolean> permalinkToHasAccess, Long permalink, Long replacePhoneId) {
        return permalinkToHasAccess.getOrDefault(permalink, false) ? replacePhoneId : null;
    }

    private void replaceForBanners(int shard, Map<Long, Long> bannerIdToNewPhoneId) {
        Iterables.partition(bannerIdToNewPhoneId.keySet(), REPLACE_CHUNK_SIZE)
                .forEach(bannerIds -> {
                    Map<Long, Long> chunk = new HashMap<>();
                    List<Long> deletePhoneBannerIds = new ArrayList<>();
                    for (Long bannerId : bannerIds) {
                        Long newPhoneId = bannerIdToNewPhoneId.get(bannerId);
                        if (newPhoneId == null) {
                            deletePhoneBannerIds.add(newPhoneId);
                        } else {
                            chunk.put(bannerId, newPhoneId);
                        }
                    }
                    clientPhoneRepository.linkBannerPhones(shard, chunk);
                    clientPhoneRepository.unlinkBannerPhonesByBannerId(shard, deletePhoneBannerIds);
                    bannerCommonRepository.resetStatusBsSyncedByIds(shard, bannerIds);
                });
    }

    private void replaceForCampaigns(int shard, Map<Long, Long> campaignIdToNewPhoneId) {
        Iterables.partition(campaignIdToNewPhoneId.keySet(), REPLACE_CHUNK_SIZE)
                .forEach(campaignIds -> {
                    Map<Long, Long> chunk = new HashMap<>();
                    List<Long> deletePhoneCampaignIds = new ArrayList<>();
                    for (Long campaignId : campaignIds) {
                        Long newPhoneId = campaignIdToNewPhoneId.get(campaignId);
                        if (newPhoneId == null) {
                            deletePhoneCampaignIds.add(newPhoneId);
                        } else {
                            chunk.put(campaignId, newPhoneId);
                        }
                    }
                    clientPhoneRepository.linkCampaignPhones(shard, chunk);
                    clientPhoneRepository.unlinkCampaignPhonesByCampaignId(shard, deletePhoneCampaignIds);
                });
    }
}
