package ru.yandex.direct.core.entity.clientphone;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import ru.yandex.altay.model.language.LanguageOuterClass;
import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings;
import ru.yandex.direct.core.entity.calltracking.model.CampCalltrackingPhones;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneMapping;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.container.CounterIdWithDomain;
import ru.yandex.direct.core.entity.organization.model.Organization;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhoneType;
import ru.yandex.direct.core.entity.trackingphone.model.PhoneNumber;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.organizations.swagger.OrganizationApiInfo;
import ru.yandex.direct.organizations.swagger.OrganizationInfo;
import ru.yandex.direct.organizations.swagger.OrganizationInfoConverters;
import ru.yandex.direct.organizations.swagger.model.CompanyPhone;
import ru.yandex.direct.redislock.DistributedLock;
import ru.yandex.direct.redislock.DistributedLockException;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.telephony.client.TelephonyClient;
import ru.yandex.direct.telephony.client.model.TelephonyPhoneRequest;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.clientphone.validation.ClientPhoneValidationService.CLIENT_PHONE_SPRAV_VALIDATOR;
import static ru.yandex.direct.organizations.swagger.OrganizationsClient.getLanguageByName;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class ClientPhoneService {
    private static final Logger logger = LoggerFactory.getLogger(ClientPhoneService.class);

    private final ShardHelper shardHelper;
    private final ClientPhoneRepository clientPhoneRepository;
    private final TelephonyClient telephonyClient;
    private final BannerCommonRepository bannerCommonRepository;
    private final OrganizationService organizationService;
    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final UserService userService;
    private final ClientService clientService;
    private final TranslationService translationService;
    private final FeatureService featureService;
    private final ClientPhoneLocker locker;
    private final TelephonyPhoneService telephonyPhoneService;
    private final ClientPhoneReplaceService clientPhoneReplaceService;

    public ClientPhoneService(ShardHelper shardHelper,
                              ClientPhoneRepository clientPhoneRepository,
                              TelephonyClient telephonyClient,
                              BannerCommonRepository bannerCommonRepository,
                              OrganizationService organizationService,
                              BannersUpdateOperationFactory bannersUpdateOperationFactory,
                              UserService userService,
                              ClientService clientService,
                              TranslationService translationService,
                              FeatureService featureService,
                              ClientPhoneLocker locker,
                              TelephonyPhoneService telephonyPhoneService,
                              ClientPhoneReplaceService clientPhoneReplaceService) {
        this.shardHelper = shardHelper;
        this.clientPhoneRepository = clientPhoneRepository;
        this.telephonyClient = telephonyClient;
        this.bannerCommonRepository = bannerCommonRepository;
        this.organizationService = organizationService;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.userService = userService;
        this.clientService = clientService;
        this.translationService = translationService;
        this.featureService = featureService;
        this.locker = locker;
        this.telephonyPhoneService = telephonyPhoneService;
        this.clientPhoneReplaceService = clientPhoneReplaceService;
    }

    /**
     * Возвращает список номеров, которые принадлежат указанным организациями
     * или имеют тип manual (для личных телефонов клиента {@code permalinkId} не указан)
     */
    public List<ClientPhone> getAllClientPhones(ClientId clientId, List<Long> permalinkIds) {
        return clientPhoneRepository.getAllClientPhones(clientId, permalinkIds);
    }

    public List<ClientPhone> getTelephonyPhonesBySettings(ClientId clientId, List<CalltrackingSettings> settings) {
        return clientPhoneRepository.getTelephonyPhonesBySettings(clientId, settings);
    }

    public List<ClientPhone> getAllClientOrganizationPhones(ClientId clientId, Collection<Long> permalinkIds) {
        return clientPhoneRepository.getAllClientOrganizationPhones(clientId, permalinkIds);
    }

    public List<ClientPhone> getByPhoneIds(ClientId clientId, Collection<Long> phoneIds) {
        return clientPhoneRepository.getByPhoneIds(clientId, phoneIds);
    }

    public Map<Long, List<ClientPhone>> getTelephonyPhonesBySettingIds(
            ClientId clientId,
            List<Long> settingIds
    ) {
        return clientPhoneRepository.getTelephonyPhonesBySettingIds(clientId, settingIds);
    }

    public void resetLastShowTime(int shard, List<Long> clientPhoneIds, LocalDateTime now) {
        clientPhoneRepository.resetLastShowTime(shard, clientPhoneIds, now);
    }

    /**
     * Запросить и сохранить телефоны одной организации
     *
     * @param clientId         — id клиента
     * @param organizationInfo — информация об организации, полученная из Справочника
     * @param metrikaCounterId — id счётчика метрики организации
     * @return коллекция телефонов для этой организации в формате ClientPhone
     */
    @NotNull
    public Collection<ClientPhone> getAndSaveOrganizationPhones(
            ClientId clientId,
            OrganizationInfo organizationInfo,
            long metrikaCounterId
    ) {
        var permalinkId = organizationInfo.getPermalinkId();
        return saveOrganizationPhones(
                clientId,
                List.of(permalinkId),
                List.of(organizationInfo),
                Map.of(permalinkId, metrikaCounterId)
        ).get(permalinkId);
    }

    /**
     * Получение номеров организации.
     * Делается два запроса в API Справочника: за телефонами и за счетчиками метрики.
     * После этого номера вместе со счетчиками сохраняются в таблицу {@code client_phones}
     *
     * @return отображение: идентификатор организации -> список телефонов Справочника/пустой список, если телефонов нет.
     * Порядок в списке важен, соблюдается такой же порядок как в Справочнике.
     */
    public Map<Long, List<ClientPhone>> getAndSaveOrganizationPhones(
            ClientId clientId,
            List<Long> permalinkIds
    ) {
        if (permalinkIds.isEmpty()) {
            return emptyMap();
        }
        LanguageOuterClass.Language language = getLanguageByName(LocaleContextHolder.getLocale().getLanguage())
                .orElse(LanguageOuterClass.Language.EN);
        List<OrganizationApiInfo> organizationsByIds =
                organizationService.getClientOrganizationsByIds(permalinkIds, language);
        List<OrganizationApiInfo> organizationsWithPhones = getOrganizationsWithPhones(organizationsByIds);
        Map<Long, Long> metrikaCountersByPermalinkIds = getMetrikaCountersByPermalinks(organizationsByIds);
        return saveOrganizationPhones(clientId, permalinkIds, organizationsWithPhones,
                metrikaCountersByPermalinkIds);
    }

    @NotNull
    private Map<Long, List<ClientPhone>> saveOrganizationPhones(
            ClientId clientId,
            List<Long> permalinkIds,
            List<? extends OrganizationInfo> organizationsWithPhones,
            Map<Long, Long> metrikaCountersByPermalinkIds
    ) {
        List<ClientPhone> dbPhones = clientPhoneRepository.getAllClientPhones(clientId, permalinkIds);
        List<ClientPhone> dbOrgPhones = filterList(dbPhones, p -> p.getPhoneType() == ClientPhoneType.SPRAV);
        List<ClientPhone> dbManualPhones = filterList(dbPhones, p -> p.getPhoneType() == ClientPhoneType.MANUAL);
        var dbOrgPhonesByPermalinkIds = StreamEx.of(dbOrgPhones).groupingBy(ClientPhone::getPermalinkId);

        Map<Long, Set<Long>> linkedPhoneIdsByPermalink = getLinkedPhoneIdsByPermalink(clientId, permalinkIds);

        Map<Long, List<ClientPhone>> validatedOrgPhonesByPermalink =
                getValidatedOrgPhones(clientId, organizationsWithPhones, metrikaCountersByPermalinkIds);

        Map<Long, List<ClientPhone>> phonesByPermalink = handleOrganizationPhones(
                clientId,
                dbOrgPhonesByPermalinkIds,
                dbManualPhones,
                linkedPhoneIdsByPermalink,
                validatedOrgPhonesByPermalink
        );
        permalinkIds.forEach(permalink -> phonesByPermalink.putIfAbsent(permalink, emptyList()));
        return phonesByPermalink;
    }

    /**
     * Провалидировать телефоны организаций {@code orgsWithPhones}.
     *
     * @return отображение: идентификатор организации -> валидные номера телефонов этой организации
     */
    private Map<Long, List<ClientPhone>> getValidatedOrgPhones(
            ClientId clientId,
            List<? extends OrganizationInfo> orgsWithPhones,
            Map<Long, Long> metrikaCountersByPermalinkIds
    ) {
        Map<Long, List<ClientPhone>> validatedOrgPhonesByPermalink = new HashMap<>();
        for (var org : orgsWithPhones) {
            Long permalink = org.getPermalinkId();
            Long counter = metrikaCountersByPermalinkIds.get(permalink);
            List<ClientPhone> validatedOrgPhones = getValidatedOrgPhones(org.getPhones(), clientId, permalink, counter);
            validatedOrgPhonesByPermalink.put(permalink, validatedOrgPhones);
        }
        return validatedOrgPhonesByPermalink;
    }

    /**
     * Провалидировать телефоны организации {@code orgPhones} и вернуть только валидные.
     */
    private List<ClientPhone> getValidatedOrgPhones(
            List<CompanyPhone> orgPhones,
            ClientId clientId,
            Long permalink,
            @Nullable Long counter
    ) {
        return StreamEx.of(orgPhones)
                .remove(Objects::isNull)
                .map(phone -> ClientPhoneUtils.companyPhoneToClientPhone(clientId, permalink, phone, counter))
                .remove(phone -> CLIENT_PHONE_SPRAV_VALIDATOR.apply(phone).hasAnyErrors())
                .toList();
    }

    /**
     * Обработка телефонов организации.
     * <ol>
     * <li>Запрашиваем все номера из Справочника</li>
     * <li>Телефонам из Справочника, которые уже есть в БД, проставляем id и обновляем их</li>
     * <li>Телефоны из Справочника, которых еще нет в БД, добавляем</li>
     * <li>Телефоны из БД, которых уже нет в Справочнике:</li>
     * <ul>
     *     <li>если они не привязаны ни к одному баннеру, то удаляем, чтобы клиент не мог их выбрать</li>
     *     <li>если привязан, то будет переведен в ручной (тип {@link ClientPhoneType#MANUAL}</li>
     * </ul>
     * </ol>
     *
     * @param clientId                  идентификатор клиента
     * @param dbOrgPhonesByPermalinkIds отображение: идентификатор организации -> телефоны организации в БД
     * @param dbManualPhones            список ручных номеров данного клиента {@code clientId}
     * @param linkedPhoneIdsByPermalink отображение: идентификатор организации -> идентификаторы телефонов,
     *                                  привязанные к баннерам с этой организацией
     * @param orgPhonesByPermalink      отображение: идентификатор организации -> валидные телефоны организации в
     *                                  Справочнике
     * @return отображение: идентификатор организации -> номера телефонов этой организации
     */
    private Map<Long, List<ClientPhone>> handleOrganizationPhones(
            ClientId clientId,
            Map<Long, List<ClientPhone>> dbOrgPhonesByPermalinkIds,
            List<ClientPhone> dbManualPhones,
            Map<Long, Set<Long>> linkedPhoneIdsByPermalink,
            Map<Long, List<ClientPhone>> orgPhonesByPermalink
    ) {
        String oldOrgPhoneComment = getOldOrgPhoneLocaleComment(clientId);

        List<ClientPhone> mainPhoneAddOrUpdateContainer = new ArrayList<>();
        List<ClientPhone> phoneAddOrUpdateContainer = new ArrayList<>();
        List<Long> phoneDeleteContainer = new ArrayList<>();
        Map<Long, Set<Long>> phoneIdsToPhoneIdReplaceContainer = new HashMap<>();

        var dbManualPhonesByNumber = listToMap(dbManualPhones, ClientPhone::getPhoneNumber, Function.identity());

        orgPhonesByPermalink.forEach((permalink, orgPhones) -> {
            var dbPhonesByNumber = getDbPhonesByNumber(permalink, dbOrgPhonesByPermalinkIds);

            Map<String, ClientPhone> orgPhonesByNumber;
            if (orgPhones.isEmpty()) {
                orgPhonesByNumber = emptyMap();
            } else {
                // Обрабатываем "главный" телефон из Справочника, то есть тот, который приходит первым в списке
                var mainOrgPhone = orgPhones.get(0);
                handleMainOrgPhone(mainPhoneAddOrUpdateContainer, mainOrgPhone, dbPhonesByNumber);

                // Обрабатываем оставшиеся телефоны из Справочника
                // Телефонам из Справочника, которые уже есть в БД, проставляем id и обновляем
                // Телефоны из Справочника, которых еще нет в БД, добавляем
                var orgNotMainPhonesByNumber = getOrgNotMainPhonesByNumber(orgPhones);
                handleOrgPhones(phoneAddOrUpdateContainer, orgNotMainPhonesByNumber, dbPhonesByNumber);

                orgPhonesByNumber = getOrgPhonesByNumber(mainOrgPhone, orgNotMainPhonesByNumber);
            }

            // Обрабатываем телефоны в БД, которых уже нет в Справочнике:
            // Если телефон не привязан ни к одному баннеру или кампании -- удаляем
            // Если привязан, то переводим номер в ручной. Если такой ручной уже есть,
            // то заменяем использование старого телефона организации на этот ручной во всех его баннерах и кампаниях
            Set<Long> linkedPhoneIds = linkedPhoneIdsByPermalink.getOrDefault(permalink, emptySet());
            handleDbNotOrgPhones(
                    phoneAddOrUpdateContainer,
                    phoneDeleteContainer,
                    linkedPhoneIds,
                    dbPhonesByNumber,
                    dbManualPhonesByNumber,
                    orgPhonesByNumber,
                    oldOrgPhoneComment,
                    phoneIdsToPhoneIdReplaceContainer
            );
        });

        phoneIdsToPhoneIdReplaceContainer.forEach((replaceId, phoneIds) ->
                clientPhoneReplaceService.replaceTrackingPhoneWithoutValidation(clientId, phoneIds, replaceId));
        clientPhoneRepository.delete(clientId, phoneDeleteContainer);
        List<Long> mainPhoneIds = clientPhoneRepository.addOrUpdate(clientId, mainPhoneAddOrUpdateContainer);
        List<Long> notMainPhoneIds = clientPhoneRepository.addOrUpdate(clientId, phoneAddOrUpdateContainer);

        List<Long> phoneIds = new ArrayList<>(mainPhoneIds);
        phoneIds.addAll(notMainPhoneIds);

        List<ClientPhone> phones = clientPhoneRepository.getByPhoneIds(clientId, phoneIds);
        Map<Long, ClientPhone> phonesByPhoneId = listToMap(phones, ClientPhone::getId, Function.identity());
        Map<Long, List<ClientPhone>> phonesByPermalink = new LinkedHashMap<>();
        phoneIds.forEach(id -> {
            ClientPhone phone = phonesByPhoneId.get(id);
            if (phone == null) {
                return;
            }
            Long permalinkId = phone.getPermalinkId();
            if (permalinkId != null) {
                phonesByPermalink.computeIfAbsent(permalinkId, k -> new ArrayList<>()).add(phone);
                var orgPhones = orgPhonesByPermalink.getOrDefault(permalinkId, emptyList());
                Boolean isHidden =
                        orgPhones.stream()
                                .filter(cp -> cp.getPhoneNumber().equals(phone.getPhoneNumber()))
                                .findAny() // ищем любой совпадающий номер
                                .map(ClientPhone::getIsHidden)
                                .orElse(false);
                phone.setIsHidden(isHidden);
            }
        });
        return phonesByPermalink;
    }

    private void handleMainOrgPhone(
            List<ClientPhone> mainPhoneAddOrUpdateContainer,
            ClientPhone mainOrgPhone,
            Map<String, ClientPhone> dbPhonesByNumber
    ) {
        String orgPhoneNumber = ClientPhoneMapping.phoneNumberToDb(mainOrgPhone.getPhoneNumber());
        if (dbPhonesByNumber.containsKey(orgPhoneNumber)) {
            ClientPhone dbPhone = dbPhonesByNumber.get(orgPhoneNumber);
            mainOrgPhone.setId(dbPhone.getId());
        }
        mainPhoneAddOrUpdateContainer.add(mainOrgPhone);
    }

    private void handleOrgPhones(
            List<ClientPhone> phoneAddOrUpdateContainer,
            Map<String, ClientPhone> orgPhonesByNumber,
            Map<String, ClientPhone> dbPhonesByNumber
    ) {
        for (Map.Entry<String, ClientPhone> entry : orgPhonesByNumber.entrySet()) {
            String number = entry.getKey();
            ClientPhone phone = entry.getValue();
            if (dbPhonesByNumber.containsKey(number)) {
                ClientPhone dbPhone = dbPhonesByNumber.get(number);
                phone.setId(dbPhone.getId());
            }
        }
        phoneAddOrUpdateContainer.addAll(orgPhonesByNumber.values());
    }

    private void handleDbNotOrgPhones(
            List<ClientPhone> phoneAddOrUpdateContainer,
            List<Long> phoneDeleteContainer,
            Set<Long> linkedPhoneIds,
            Map<String, ClientPhone> dbPhonesByNumber,
            Map<PhoneNumber, ClientPhone> dbManualPhonesByNumber,
            Map<String, ClientPhone> orgPhonesByNumber,
            String oldOrgPhoneComment,
            Map<Long, Set<Long>> phoneIdsToPhoneIdReplaceContainer
    ) {
        dbPhonesByNumber.forEach((number, phone) -> {
            if (orgPhonesByNumber.containsKey(number)) {
                return;
            }
            // Обрабатываем телефоны организации из БД, которых уже нет в Справочнике
            Long orgPhoneId = phone.getId();
            if (linkedPhoneIds.contains(orgPhoneId)) {
                var sameManualOrNull = dbManualPhonesByNumber.get(phone.getPhoneNumber());
                if (sameManualOrNull == null) {
                    // Такого же номера нет среди ручных номеров БД
                    convertToManual(phone, oldOrgPhoneComment);
                    phoneAddOrUpdateContainer.add(phone);
                } else {
                    // Телефон организации нужно заменить на такой же ручной из БД, а телефон организации удалить
                    Long manualId = sameManualOrNull.getId();
                    phoneIdsToPhoneIdReplaceContainer.computeIfAbsent(manualId, v -> new HashSet<>()).add(orgPhoneId);
                    phoneDeleteContainer.add(orgPhoneId);
                    updateComment(sameManualOrNull, oldOrgPhoneComment);
                    phoneAddOrUpdateContainer.add(sameManualOrNull);
                }
            } else {
                phoneDeleteContainer.add(orgPhoneId);
            }
        });
    }

    private void convertToManual(ClientPhone phone, String comment) {
        phone.setPhoneType(ClientPhoneType.MANUAL);
        phone.setComment(comment);
        phone.setPermalinkId(null);
        phone.setCounterId(null);
    }

    private void updateComment(ClientPhone phone, String comment) {
        // Если у ручного номера уже есть непустой комментарий, оставляем его. Иначе comment
        if (phone.getComment().isBlank()) {
            phone.setComment(comment);
        }
    }

    private List<OrganizationApiInfo> getOrganizationsWithPhones(List<OrganizationApiInfo> organizations) {
        Map<Boolean, List<OrganizationApiInfo>> organizationsByPhones = StreamEx.of(organizations)
                .partitioningBy(org -> !org.getPhones().isEmpty());
        List<OrganizationApiInfo> organizationsWithoutPhones = organizationsByPhones.get(false);
        List<OrganizationApiInfo> organizationsWithPhones = organizationsByPhones.get(true);
        if (!organizationsWithoutPhones.isEmpty()) {
            logger.info("Organizations without phones will be ignored: [{}]", StreamEx.of(organizationsWithoutPhones)
                    .map(Organization::getPermalinkId)
                    .joining(", "));
        }
        return organizationsWithPhones;
    }

    /**
     * Проверяет, включена ли у клиента фича Телефонии. Если да, отправляет запросы в Телефонию.
     */
    public List<ClientPhone> getAndSaveTelephonyPhones(ClientId clientId, List<Long> permalinkIds) {
        List<ClientPhone> telephonyPhones;
        boolean isTelephonyAllowed = featureService.isEnabledForClientId(clientId, FeatureName.TELEPHONY_ALLOWED);
        if (isTelephonyAllowed) {
            List<ClientPhone> clientPhones = getAllClientPhones(clientId, permalinkIds);
            List<ClientPhone> existedTelephonyPhones = filterList(clientPhones,
                    p -> p.getPhoneType() == ClientPhoneType.TELEPHONY
            );
            List<ClientPhone> newTelephonyPhones = handleTelephonyPhones(clientId, clientPhones);
            telephonyPhones = ListUtils.union(existedTelephonyPhones, newTelephonyPhones);

            if (!telephonyPhones.isEmpty()) {
                changeTelephonyRedirectIfNeeded(clientId, telephonyPhones);
            }
        } else {
            telephonyPhones = emptyList();
        }
        return telephonyPhones;
    }

    /**
     * Добавляет номера телефонии с перенаправлением на основной номер организации.
     * Валидируется, что номера телефонии для организации и клиента нет, так как он может быть только один.
     *
     * @return добавленные номера
     */
    public List<ClientPhone> handleTelephonyPhones(ClientId clientId, List<ClientPhone> clientPhones) {
        Set<Long> permalinkIds =
                clientPhones.stream()
                        .map(ClientPhone::getPermalinkId)
                        .filter(Objects::nonNull)
                        .collect(Collectors.toSet());
        logger.info("Try to add Telephony phones to permalinks {}", permalinkIds);
        var permalinksToAdd = getPermalinksToAddTelephony(permalinkIds, clientPhones);

        if (isEmpty(permalinksToAdd)) {
            return emptyList();
        }
        logger.info("Will add Telephony phone for next permalinks: {}", permalinksToAdd);

        List<ClientPhone> models = mapList(permalinksToAdd, permalink ->
                new ClientPhone()
                        .withClientId(clientId)
                        .withPermalinkId(permalink)
                        .withPhoneType(ClientPhoneType.TELEPHONY)
                        .withIsDeleted(false)
        );

        TelephonyPhoneAddOperation operation = new TelephonyPhoneAddOperation(
                clientId,
                models,
                clientPhoneRepository,
                organizationService,
                telephonyPhoneService
        );

        List<ClientPhone> resultPhones = new ArrayList<>();
        // Считаем, что работа в большинстве случаев ведется с одним пермалинком,
        // поэтому ставим лок на клиента. Подробнее см. тикет DIRECT-129094
        Optional<List<ClientPhone>> clientPhonesOptional = tryLockAndAllocate(clientId, operation);
        if (clientPhonesOptional.isPresent()) {
            resultPhones = clientPhonesOptional.get();
        } else {
            if (waitTillUnlocked(clientId)) {
                resultPhones = filterList(clientPhoneRepository.getByPermalinkIds(clientId, permalinkIds),
                        p -> p.getPhoneType().equals(ClientPhoneType.TELEPHONY));
            }
        }
        return resultPhones;
    }

    /**
     * Добавляет номера c подменниками Телефонии для коллтрекинга на сайте
     *
     * @return информация о привязке новых номеров к кампаниям, в которых используется коллтрекинг
     */
    public List<CampCalltrackingPhones> addNewCalltrackingOnSitePhones(
            ClientId clientId, List<Pair<ClientPhone, Long>> clientPhonesWithCampaignId) {
        List<ClientPhone> clientPhones = mapList(clientPhonesWithCampaignId, Pair::getKey);
        List<String> phones = mapList(clientPhones, clientPhone -> clientPhone.getPhoneNumber().getPhone());
        logger.info("Adding client phones {} for clientId {}", phones, clientId.asLong());

        CalltrackingOnSitePhoneAddOperation operation = new CalltrackingOnSitePhoneAddOperation(
                clientId,
                clientPhones,
                clientPhoneRepository,
                telephonyPhoneService
        );

        MassResult<Long> result = operation.prepareAndApply();

        List<CampCalltrackingPhones> campCalltrackingPhones = new ArrayList<>();
        List<Result<Long>> innerResults = result.getResult();
        for (int i = 0; i < innerResults.size(); i++) {
            var innerResult = innerResults.get(i);
            var campaignId = clientPhonesWithCampaignId.get(i).getValue();
            if (!innerResult.isSuccessful()) {
                logger.info("Can't allocate telephony phone for campaign {}", campaignId);
                continue;
            }
            Long clientPhoneId = innerResult.getResult();
            campCalltrackingPhones.add(new CampCalltrackingPhones()
                    .withClientPhoneId(clientPhoneId)
                    .withCid(clientPhonesWithCampaignId.get(i).getValue()));
        }

        return campCalltrackingPhones;
    }

    /**
     * Выполняет добавление номера Телефонии под локом.
     * При успехе возвращает добавленные номера.
     * Если лок взять не удалось, добавление не происходит и возвращается пустой результат
     */
    private Optional<List<ClientPhone>> tryLockAndAllocate(ClientId clientId, TelephonyPhoneAddOperation operation) {
        try {
            return tryLockAndAllocateUnsafe(clientId, operation);
        } catch (DistributedLockException e) {
            logger.warn("Lock error", e);
            return Optional.empty();
        }
    }

    private Optional<List<ClientPhone>> tryLockAndAllocateUnsafe(
            ClientId clientId,
            TelephonyPhoneAddOperation operation
    ) {
        DistributedLock lock = locker.create(clientId.toString());
        if (!lock.tryLock()) {
            return Optional.empty();
        }
        MassResult<Long> result;
        try {
            result = operation.prepareAndApply();
            if (!result.getValidationResult().flattenErrors().isEmpty()) {
                logger.warn("Can't add Telephony phones. Errors: {}", result.getValidationResult().flattenErrors());
            }
        } finally {
            lock.unlock();
        }

        List<Long> phoneIds = result.getResult()
                .stream()
                .filter(Result::isSuccessful)
                .map(Result::getResult)
                .collect(Collectors.toList());
        List<ClientPhone> resultPhones = clientPhoneRepository.getByPhoneIds(clientId, phoneIds);
        return Optional.of(resultPhones);
    }

    private boolean waitTillUnlocked(ClientId clientId) {
        DistributedLock lock = locker.create(clientId.toString());
        try {
            if (!lock.lock()) {
                // не удалось дождаться освобождения лока
                return false;
            }
            lock.unlock();
            logger.info("Success unlock after waiting");
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        } catch (DistributedLockException e) {
            logger.warn("Wait lock error", e);
        }
        return true;
    }

    /**
     * Заменить редиректы на ручные номера на главные номера организаций, если прав на организацию нет.
     */
    private void changeTelephonyRedirectIfNeeded(ClientId clientId, List<ClientPhone> telephonyPhones) {
        List<Long> permalinks = mapList(telephonyPhones, ClientPhone::getPermalinkId);
        Map<Long, Boolean> accessByPermalinks = organizationService.hasAccess(clientId, permalinks);
        Set<Long> notAccessPermalinks = StreamEx.of(permalinks)
                .remove(p -> accessByPermalinks.getOrDefault(p, false))
                .toSet();
        if (notAccessPermalinks.isEmpty()) {
            return;
        }

        LanguageOuterClass.Language language = getLanguageByName(LocaleContextHolder.getLocale().getLanguage())
                .orElse(LanguageOuterClass.Language.EN);
        List<OrganizationApiInfo> organizationsByIds =
                organizationService.getClientOrganizationsByIds(notAccessPermalinks, language);
        Map<Long, List<String>> orgPhonesByPermalink = StreamEx.of(organizationsByIds)
                .mapToEntry(Organization::getPermalinkId, OrganizationInfo::getPhones)
                .mapValues(phones -> mapList(phones, p -> ClientPhoneUtils.toPhoneNumber(p).getPhone()))
                .toMap();

        Function<ClientPhone, ClientPhone> chooseNewRedirectPhoneOrNull = p -> {
            Long permalink = p.getPermalinkId();
            List<String> orgPhones = orgPhonesByPermalink.get(permalink);
            if (isEmpty(orgPhones)) {
                logger.info("Cannot update redirect phone for phone {}. " +
                                "Organization phones for permalink {} are not found",
                        p.getId(), permalink);
                return null;
            }
            String oldRedirectPhone = p.getPhoneNumber().getPhone();
            if (orgPhones.contains(oldRedirectPhone)) {
                return null;
            }
            String mainOrgPhone = orgPhones.get(0);
            logger.info("New redirect phone for phone {}. " +
                            "Old redirect phone: {}, organization phones: {}, new redirect phone: {}",
                    p.getId(), oldRedirectPhone, orgPhones, mainOrgPhone);
            p.setPhoneNumber(new PhoneNumber().withPhone(mainOrgPhone));
            return p;
        };
        List<ClientPhone> redirectUpdatePhones = StreamEx.of(telephonyPhones)
                .filter(p -> notAccessPermalinks.contains(p.getPermalinkId()))
                .map(chooseNewRedirectPhoneOrNull)
                .filter(Objects::nonNull)
                .toList();

        updateTelephonyRedirectPhones(clientId, redirectUpdatePhones);
    }

    /**
     * Отбираем из {@param permalinkIds} номера организаций, которым можем прикрепить номер Телефонии,
     * то есть номера Телефонии еще нет
     *
     * @param clientPhones все номера клиента
     */
    private Set<Long> getPermalinksToAddTelephony(
            Set<Long> permalinkIds,
            List<ClientPhone> clientPhones
    ) {
        Set<Long> permalinksWithTelephony = filterAndMapToSet(
                clientPhones,
                phone -> phone.getPhoneType() == ClientPhoneType.TELEPHONY,
                ClientPhone::getPermalinkId
        );
        if (!permalinksWithTelephony.isEmpty()) {
            logger.info("Next permalinks already has Telephony phone: {}", permalinksWithTelephony);
        }

        if (permalinksWithTelephony.size() == permalinkIds.size()) {
            logger.info("No permalinks to add Telephony phone");
            return emptySet();
        }

        return Sets.difference(permalinkIds, permalinksWithTelephony);
    }

    public Result<Long> addManualClientPhone(ClientPhone clientPhone) {
        clientPhone.setPhoneType(ClientPhoneType.MANUAL);
        ClientPhoneAddOperation operation = new ClientPhoneAddOperation(
                clientPhone.getClientId(),
                List.of(clientPhone),
                clientPhoneRepository);
        MassResult<Long> result = operation.prepareAndApply();
        if (!result.isSuccessful()) {
            return Result.broken(ValidationResult.failed(clientPhone.getId(), CommonDefects.invalidValue()));
        }
        return result.get(0);
    }

    public MassResult<Long> delete(ClientId clientId, List<Long> clientPhoneIds) {
        ClientPhoneDeleteOperation operation = new ClientPhoneDeleteOperation(
                clientId,
                shardHelper.getShardByClientId(clientId),
                clientPhoneIds,
                clientPhoneRepository,
                bannerCommonRepository);
        return operation.prepareAndApply();
    }

    public Result<Long> updateClientPhone(ClientPhone updatedPhone) {
        int shard = shardHelper.getShardByClientId(updatedPhone.getClientId());
        List<ModelChanges<ClientPhone>> modelChanges = List.of(getManualModelChanges(updatedPhone));
        ClientPhoneUpdateOperation operation = new ClientPhoneUpdateOperation(
                shard,
                updatedPhone.getClientId(),
                modelChanges,
                clientPhoneRepository,
                bannerCommonRepository);
        MassResult<Long> result = operation.prepareAndApply();
        if (!result.isSuccessful()) {
            return Result.broken(ValidationResult.failed(updatedPhone.getId(), CommonDefects.invalidValue()));
        }
        return result.get(0);
    }

    public MassResult<Long> updateBannersPhone(ClientId clientId, Long operatorUid, Long phoneId, List<Long> bannerIds,
                                               List<Long> permalinkIds) {
        Set<Long> uniqPermalinkIds = listToSet(permalinkIds, Function.identity());
        Map<Long, Long> existedBannerPermalinks = organizationService.getPermalinkIdsByBannerIds(clientId, bannerIds);
        List<Long> bannerIdsToUpdate = EntryStream.of(existedBannerPermalinks)
                .filterValues(uniqPermalinkIds::contains)
                .keys()
                .toList();

        if (bannerIdsToUpdate.isEmpty()) {
            return MassResult.successfulMassAction(emptyList(), ValidationResult.success(emptyList()));
        }

        List<ModelChanges<BannerWithSystemFields>> modelChanges = mapList(bannerIdsToUpdate,
                id -> ModelChanges.build(id, TextBanner.class, TextBanner.PHONE_ID, phoneId)
                        .castModelUp(BannerWithSystemFields.class));
        return bannersUpdateOperationFactory
                .createPartialUpdateOperation(modelChanges, operatorUid, clientId)
                .prepareAndApply();
    }

    private ModelChanges<ClientPhone> getManualModelChanges(ClientPhone clientPhone) {
        return new ModelChanges<>(clientPhone.getId(), ClientPhone.class)
                .process(clientPhone.getPhoneNumber(), ClientPhone.PHONE_NUMBER)
                .process(clientPhone.getComment(), ClientPhone.COMMENT);
    }

    /**
     * Возвращает подменники для колтрекинга на сайте ({@code permalink_id = 0})
     */
    public Map<ClientId, Map<Long, ClientPhone>> getSiteTelephonyByIdAndClientId(int shard) {
        Map<ClientId, List<ClientPhone>> clientPhonesByClientId =
                clientPhoneRepository.getSiteTelephonyByClientId(shard, false);
        return clientPhonesByClientId.entrySet()
                .stream()
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        e -> e.getValue()
                                .stream()
                                .collect(Collectors.toMap(
                                        clientPhone -> clientPhone.getId(),
                                        Function.identity()
                                ))
                ));
    }

    public Result<Long> updateTelephonyRedirectPhone(ClientPhone phone) {
        return updateTelephonyRedirectPhones(phone.getClientId(), List.of(phone)).get(0);
    }

    public MassResult<Long> updateTelephonyRedirectPhones(ClientId clientId, List<ClientPhone> phones) {
        int shard = shardHelper.getShardByClientId(clientId);

        List<ModelChanges<ClientPhone>> modelChanges = mapList(phones, phone ->
                new ModelChanges<>(phone.getId(), ClientPhone.class)
                        .processNotNull(phone.getPhoneNumber(), ClientPhone.PHONE_NUMBER)
        );

        TelephonyPhoneUpdateOperation operation = new TelephonyPhoneUpdateOperation(
                shard,
                clientId,
                modelChanges,
                clientPhoneRepository,
                bannerCommonRepository
        );

        MassResult<Long> massResult = operation.prepareAndApply();
        List<Long> successUpdatedPhoneIds = StreamEx.of(massResult.getResult())
                .filter(Result::isSuccessful)
                .map(Result::getResult)
                .toList();

        List<ClientPhone> successUpdatedPhones = clientPhoneRepository.getByPhoneIds(clientId, successUpdatedPhoneIds);
        successUpdatedPhones.forEach(this::updatePhoneInTelephony);
        return massResult;
    }

    private void updatePhoneInTelephony(ClientPhone phone) {
        String telephonyServiceId = phone.getTelephonyServiceId();
        if (telephonyServiceId == null) {
            // номер Телефонии отвязан, обновлять нечего
            return;
        }
        String redirectPhone = phone.getPhoneNumber().getPhone();
        telephonyClient.linkServiceNumber(
                phone.getClientId().asLong(),
                new TelephonyPhoneRequest()
                        .withRedirectPhone(redirectPhone)
                        .withTelephonyServiceId(telephonyServiceId)
                        .withCounterId(phone.getCounterId())
                        .withPermalinkId(phone.getPermalinkId())
        );
    }

    private Map<String, ClientPhone> getDbPhonesByNumber(
            Long permalink,
            Map<Long, List<ClientPhone>> dbPhonesByPermalinkIds
    ) {
        List<ClientPhone> dbPhones = dbPhonesByPermalinkIds.getOrDefault(permalink, emptyList());
        return listToMap(
                dbPhones,
                p -> ClientPhoneMapping.phoneNumberToDb(p.getPhoneNumber()),
                Function.identity()
        );
    }

    private Map<String, ClientPhone> getOrgNotMainPhonesByNumber(List<ClientPhone> validatedOrgPhones) {
        String mainOrgPhoneNumber = ClientPhoneMapping.phoneNumberToDb(validatedOrgPhones.get(0).getPhoneNumber());
        Map<String, ClientPhone> orgPhonesByNumber = new LinkedHashMap<>();
        // Первые "главные" телефоны организации уже обработаны
        StreamEx.of(validatedOrgPhones).skip(1).forEach(orgPhone -> {
            String orgPhoneNumber = ClientPhoneMapping.phoneNumberToDb(orgPhone.getPhoneNumber());
            // Если телефон совпадает с "главным", то не нужно обрабатывать его повторно
            if (!mainOrgPhoneNumber.equals(orgPhoneNumber)) {
                orgPhonesByNumber.put(orgPhoneNumber, orgPhone);
            }
        });
        return orgPhonesByNumber;
    }

    private Map<String, ClientPhone> getOrgPhonesByNumber(
            ClientPhone mainOrgPhone,
            Map<String, ClientPhone> orgNotMainPhonesByNumber
    ) {
        Map<String, ClientPhone> orgPhonesByNumber = new HashMap<>(orgNotMainPhonesByNumber);
        String mainOrgPhoneNumber = ClientPhoneMapping.phoneNumberToDb(mainOrgPhone.getPhoneNumber());
        orgPhonesByNumber.put(mainOrgPhoneNumber, mainOrgPhone);
        return orgPhonesByNumber;
    }

    private Map<Long, Set<Long>> getLinkedPhoneIdsByPermalink(ClientId clientId, List<Long> permalinkIds) {
        var bannerPhoneIdsByPermalink = organizationService.getBannerPhoneIdsByPermalink(clientId, permalinkIds);
        var campaignPhoneIdsByPermalink = organizationService.getCampaignPhoneIdsByPermalink(clientId, permalinkIds);
        Map<Long, Set<Long>> linkedPhoneIdsByPermalink = new HashMap<>();
        bannerPhoneIdsByPermalink.forEach((permalink, phoneIds) ->
                linkedPhoneIdsByPermalink.computeIfAbsent(permalink, k -> new HashSet<>()).addAll(phoneIds));
        campaignPhoneIdsByPermalink.forEach((permalink, phoneIds) ->
                linkedPhoneIdsByPermalink.computeIfAbsent(permalink, k -> new HashSet<>()).addAll(phoneIds));
        return linkedPhoneIdsByPermalink;
    }

    /**
     * Получить комментарий для старого телефона организации на языке главного представителя данного клиента.
     */
    private String getOldOrgPhoneLocaleComment(ClientId clientId) {
        Long chiefUid = clientService.massGetChiefUidsByClientIds(List.of(clientId)).get(clientId);
        User chiefUser = userService.getUser(chiefUid);
        Language chiefLang = chiefUser == null ? Language.RU : chiefUser.getLang();
        Locale locale = Locale.forLanguageTag(chiefLang.getLangString());
        Translatable comment = ClientPhoneTranslations.INSTANCE.oldOrganizationPhoneComment();
        return translationService.translate(comment, locale);
    }

    private Map<Long, Long> getMetrikaCountersByPermalinks(List<OrganizationApiInfo> organizations) {
        Map<Long, Long> metrikaCountersByPermalinkIds = new HashMap<>();
        Set<Long> permalinkWithoutCounters = new TreeSet<>();

        for (OrganizationApiInfo organization : organizations) {
            Long counter = OrganizationInfoConverters.getMetrikaCounterId(organization.getMetrikaData());
            if (counter == null) {
                permalinkWithoutCounters.add(organization.getPermalinkId());
            } else {
                metrikaCountersByPermalinkIds.put(organization.getPermalinkId(), counter);
            }
        }
        if (!permalinkWithoutCounters.isEmpty()) {
            logger.warn("Permalinks without counters: {}", permalinkWithoutCounters);
        }
        return metrikaCountersByPermalinkIds;
    }

    public Map<Long, Long> getPhoneIdsByBannerIds(ClientId clientId, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }
        int shard = shardHelper.getShardByClientId(clientId);
        return clientPhoneRepository.getPhoneIdsByBannerIds(shard, bannerIds);
    }

    /**
     * Атрибутировать клики по подменникам оригинальному номеру
     */
    public Map<CounterIdWithDomain, Map<String, Integer>> attributeClicksOnPhones(
            int shard,
            ClientId clientId,
            Map<CounterIdWithDomain, Map<String, Integer>> clicksOnPhonesByCounter
    ) {
        var metrikaPhones = StreamEx.ofValues(clicksOnPhonesByCounter)
                .map(Map::keySet)
                .flatMap(StreamEx::of)
                .toSet();
        var telephonesByOrigin = getTelephonesByOrigin(shard, clientId, metrikaPhones);

        Map<CounterIdWithDomain, Map<String, Integer>> attributedClicksOnPhones = new HashMap<>();
        clicksOnPhonesByCounter.forEach((key, clicksOnPhones) -> {
            var attributed = attributeClicksOnPhones(clicksOnPhones, telephonesByOrigin);
            attributedClicksOnPhones.put(key, attributed);
        });

        return attributedClicksOnPhones;
    }

    /**
     * Получить отображение: номер телефона клиента, который подменяли -> номера Телефонии
     * Номеров Телефонии может быть несколько -- например, если один и тот же подменяемый номер используется
     * в нескольких настройках для коллтрекинга на сайте
     * <p>
     * (!) Все телефоны и на входе и на выходе в формате E164
     */
    private Map<String, List<String>> getTelephonesByOrigin(
            int shard,
            ClientId clientId,
            Collection<String> telephonyPhones
    ) {
        var phones = mapList(telephonyPhones, p -> ClientPhoneMapping.phoneNumberToDb(new PhoneNumber().withPhone(p)));
        var telephonesByOrigin = clientPhoneRepository.getTelephohesByOrigin(shard, clientId, phones);
        return EntryStream.of(telephonesByOrigin)
                .mapKeys(ClientPhoneUtils::fromDbToE164)
                .mapValues(p -> mapList(p, ClientPhoneUtils::fromDbToE164))
                .toMap();
    }

    private Map<String, Integer> attributeClicksOnPhones(
            Map<String, Integer> clicksOnPhones,
            Map<String, List<String>> telephonesByOrigin
    ) {
        var attributedClicksByPhone = new HashMap<>(clicksOnPhones);
        telephonesByOrigin.forEach((origin, telephones) ->
                telephones.forEach(t -> {
                    var telephonyClicks = clicksOnPhones.get(t);
                    if (telephonyClicks != null) {
                        attributedClicksByPhone.merge(origin, telephonyClicks, Integer::sum);
                        attributedClicksByPhone.remove(t);
                    }
                }));
        return attributedClicksByPhone;
    }

}
