package ru.yandex.direct.core.entity.clientphone;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.clientphone.validation.ClientPhoneValidationService;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class ClientPhoneUpdateOperation extends SimpleAbstractUpdateOperation<ClientPhone, Long> {

    private final ClientPhoneRepository clientPhoneRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final ModelChangesValidationTool updateValidationTool;

    private final int shard;
    private final ClientId clientId;

    ClientPhoneUpdateOperation(
            int shard,
            ClientId clientId,
            List<ModelChanges<ClientPhone>> modelChanges,
            ClientPhoneRepository clientPhoneRepository,
            BannerCommonRepository bannerCommonRepository) {
        super(Applicability.PARTIAL, modelChanges, id -> new ClientPhone().withId(id));
        this.shard = shard;
        this.clientId = clientId;
        this.clientPhoneRepository = clientPhoneRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        updateValidationTool = ModelChangesValidationTool.builder().build();
    }

    @Override
    protected ValidationResult<List<ModelChanges<ClientPhone>>, Defect> validateModelChanges(
            List<ModelChanges<ClientPhone>> modelChanges
    ) {
        List<Long> phoneIds = mapList(modelChanges, ModelChanges::getId);
        List<ClientPhone> byPhoneIds = clientPhoneRepository.getByPhoneIds(clientId, phoneIds);
        Set<Long> existedIds = listToSet(byPhoneIds, ClientPhone::getId);
        return updateValidationTool.validateModelChangesList(modelChanges, existedIds);
    }

    @Override
    protected ValidationResult<List<ClientPhone>, Defect> validateAppliedChanges(
            ValidationResult<List<ClientPhone>, Defect> validationResult) {
        Map<Long, String> existingManualNumbers = clientPhoneRepository.getManualPhoneNumbers(clientId);
        return ClientPhoneValidationService.validate(validationResult.getValue(), existingManualNumbers);
    }

    @Override
    protected Collection<ClientPhone> getModels(Collection<Long> phoneIds) {
        return clientPhoneRepository.getByPhoneIds(clientId, phoneIds);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<ClientPhone>> appliedChanges) {
        clientPhoneRepository.update(shard, appliedChanges);
        List<Long> clientPhoneIds = mapList(appliedChanges, c -> c.getModel().getId());
        Map<Long, List<Long>> bannerIdsByPhoneId =
                clientPhoneRepository.getBannerIdsByPhoneId(shard, clientPhoneIds);
        List<Long> bannerIds = StreamEx.of(bannerIdsByPhoneId.values())
                .flatCollection(Function.identity())
                .toList();
        bannerCommonRepository.resetStatusBsSyncedByIds(shard, bannerIds);
        return clientPhoneIds;
    }

}
