package ru.yandex.direct.core.entity.clientphone;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneMapping;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhoneType;
import ru.yandex.direct.core.entity.trackingphone.model.PhoneNumber;
import ru.yandex.direct.core.entity.vcard.model.Phone;
import ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.organizations.swagger.model.CompanyPhone;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class ClientPhoneUtils {

    private static final Logger logger = LoggerFactory.getLogger(ClientPhoneUtils.class);

    public static ClientPhone companyPhoneToClientPhone(
            ClientId clientId,
            Long permalink,
            @Nonnull CompanyPhone orgPhone,
            @Nullable Long counter
    ) {
        return new ClientPhone()
                .withClientId(clientId)
                .withPhoneType(ClientPhoneType.SPRAV)
                .withPermalinkId(permalink)
                .withCounterId(counter)
                .withPhoneNumber(toPhoneNumber(orgPhone))
                .withIsDeleted(false)
                .withIsHidden(orgPhone.isHide());
    }

    public static PhoneNumber toPhoneNumber(CompanyPhone companyPhone) {
        if (companyPhone == null) {
            return new PhoneNumber();
        }
        String countryCode = companyPhone.getCountryCode();
        if (countryCode != null && !countryCode.startsWith("+")) {
            // если countryCode есть и не начинается с "+", добавим его
            countryCode = "+" + countryCode;
        }
        String phoneStr = nvl(countryCode, "") +
                nvl(companyPhone.getRegionCode(), "") +
                nvl(companyPhone.getNumber(), "");
        Long extension = parseExtension(companyPhone.getExt());
        return new PhoneNumber()
                .withPhone(phoneStr)
                .withExtension(extension);
    }

    public static Phone toPhone(@Nonnull PhoneNumber phoneNumber) {
        String fullPhone = phoneNumber.getPhone();
        Long extension = phoneNumber.getExtension();
        return toPhone(fullPhone, extension);
    }

    public static Phone toPhone(String fullPhone, @Nullable Long extension) {
        return ClientPhoneFormatter.format(fullPhone, extension);
    }

    /**
     * Получить код города телефона строкового формата.
     */
    @Nullable
    public static String getRegionCode(@Nullable String phoneStr) {
        if (phoneStr == null) {
            return null;
        }
        var phoneNumber = new PhoneNumber().withPhone(phoneStr);
        String dbPhoneNumber = ClientPhoneMapping.phoneNumberToDb(phoneNumber);
        String[] parts = dbPhoneNumber.split("#");
        return parts[1];
    }

    @Nullable
    static Long parseExtension(String companyExt) {
        if (isBlank(companyExt)) {
            return null;
        }
        if (companyExt.length() > PhoneValidator.EXTENSION_MAX_LENGTH) {
            logger.info("Cannot parse phone extension '{}': too long", companyExt);
            return null;
        }
        try {
            long parsedExt = Long.parseLong(companyExt);
            if (parsedExt < 1L) {
                logger.info("Cannot parse phone extension '{}': not positive number", companyExt);
                return null;
            }
            return parsedExt;
        } catch (NumberFormatException e) {
            logger.info("Cannot parse phone extension '{}' to number", companyExt);
            return null;
        }
    }

    public static ClientPhone clone(ClientPhone clientPhone) {
        return new ClientPhone()
                .withId(clientPhone.getId())
                .withClientId(clientPhone.getClientId())
                .withPermalinkId(clientPhone.getPermalinkId())
                .withCounterId(clientPhone.getCounterId())
                .withComment(clientPhone.getComment())
                .withPhoneType(clientPhone.getPhoneType())
                .withPhoneNumber(clientPhone.getPhoneNumber() != null ?
                        new PhoneNumber()
                                .withPhone(clientPhone.getPhoneNumber().getPhone())
                                .withExtension(clientPhone.getPhoneNumber().getExtension())
                        : null)
                .withTelephonyServiceId(clientPhone.getTelephonyServiceId())
                .withTelephonyPhone(clientPhone.getTelephonyPhone() != null ?
                        new PhoneNumber()
                                .withPhone(clientPhone.getTelephonyPhone().getPhone())
                                .withExtension(clientPhone.getTelephonyPhone().getExtension())
                        : null)
                .withIsDeleted(false)
                .withIsHidden(clientPhone.getIsHidden());
    }

    public static List<ClientPhone> clone(Collection<ClientPhone> clientPhones) {
        return mapList(clientPhones, ClientPhoneUtils::clone);
    }

    /**
     * Конвертировать телефон из записи в формате "[код страны]#[код города]#[номер]#[добавочный]" в E164
     * (!) Добавочный номер игнорируется
     */
    @Nonnull
    public static String fromDbToE164(@Nonnull String phone) {
        return ClientPhoneMapping.phoneNumberFromDb(phone).getPhone();

    }

}
