package ru.yandex.direct.core.entity.clientphone;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.i18n.LocaleContextHolder;

import ru.yandex.altay.model.language.LanguageOuterClass;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.organization.model.Organization;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.trackingphone.model.PhoneNumber;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.ModelsPreValidatedStep;
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation;
import ru.yandex.direct.organizations.swagger.OrganizationApiInfo;
import ru.yandex.direct.organizations.swagger.OrganizationInfoConverters;
import ru.yandex.direct.organizations.swagger.model.CompanyPhone;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.clientphone.ClientPhoneUtils.toPhoneNumber;
import static ru.yandex.direct.core.entity.clientphone.validation.ClientPhoneValidationService.preValidateTelephonyPhones;
import static ru.yandex.direct.core.entity.clientphone.validation.ClientPhoneValidationService.validateTelephonyPhones;
import static ru.yandex.direct.organizations.swagger.OrganizationsClient.getLanguageByName;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

public class TelephonyPhoneAddOperation extends SimpleAbstractAddOperation<ClientPhone, Long> {
    private static final Logger logger = LoggerFactory.getLogger(TelephonyPhoneAddOperation.class);

    private final ClientId clientId;

    private final ClientPhoneRepository clientPhoneRepository;
    private final OrganizationService organizationService;
    private final TelephonyPhoneService telephonyPhoneService;

    TelephonyPhoneAddOperation(
            ClientId clientId,
            List<ClientPhone> clientPhones,
            ClientPhoneRepository clientPhoneRepository,
            OrganizationService organizationService,
            TelephonyPhoneService telephonyPhoneService
    ) {
        super(Applicability.PARTIAL, clientPhones);

        this.clientId = clientId;

        this.clientPhoneRepository = clientPhoneRepository;
        this.organizationService = organizationService;
        this.telephonyPhoneService = telephonyPhoneService;
    }

    @Override
    public Optional<MassResult<Long>> prepare() {

        Set<Long> permalinkIds = listToSet(getModels(), ClientPhone::getPermalinkId);
        LanguageOuterClass.Language language = getLanguageByName(LocaleContextHolder.getLocale().getLanguage())
                .orElse(LanguageOuterClass.Language.EN);

        List<OrganizationApiInfo> organizationInfos =
                organizationService.getClientOrganizationsByIds(permalinkIds, language);

        Map<Long, OrganizationApiInfo> permalinkToInfo = listToMap(organizationInfos, Organization::getPermalinkId);

        for (ClientPhone phone : getModels()) {
            Long permalinkId = phone.getPermalinkId();
            OrganizationApiInfo info = permalinkToInfo.get(permalinkId);

            if (info == null) {
                continue;
            }

            Long counterId = OrganizationInfoConverters.getMetrikaCounterId(info.getMetrikaData());
            phone.setCounterId(counterId);

            CompanyPhone orgPhone = info.getPhone();
            PhoneNumber redirectPhone = toPhoneNumber(orgPhone);
            phone.setPhoneNumber(redirectPhone);
        }

        return super.prepare();
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected ValidationResult<List<ClientPhone>, Defect> preValidate(List<ClientPhone> models) {
        return preValidateTelephonyPhones(models);
    }

    @Override
    protected void onPreValidated(ModelsPreValidatedStep<ClientPhone> modelsPreValidatedStep) {
        modelsPreValidatedStep.getPreValidModelsMap().values().forEach(
                phone -> {
                    String redirectPhone = phone.getPhoneNumber().getPhone();
                    fillModel(phone, redirectPhone);
                }
        );
    }

    private void fillModel(ClientPhone phone, String redirectPhone) {
        var telephonyPhoneValues = telephonyPhoneService.attachTelephony(clientId, phone, redirectPhone);
        if (telephonyPhoneValues != null) {
            phone.setTelephonyServiceId(telephonyPhoneValues.getTelephonyServiceId());
            phone.setTelephonyPhone(telephonyPhoneValues.getTelephonyPhone());
        }
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected void validate(ValidationResult<List<ClientPhone>, Defect> preValidationResult) {
        ValidationResult<List<ClientPhone>, Defect> validationResult = validateTelephonyPhones(getModels());
        preValidationResult.merge(validationResult);
    }

    @Override
    protected List<Long> execute(List<ClientPhone> validModelsToApply) {
        return clientPhoneRepository.add(clientId, validModelsToApply);
    }

}
