package ru.yandex.direct.core.entity.clientphone;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.trackingphone.model.PhoneNumber;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.model.response.TurnOnCallTrackingResponse;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.telephony.client.TelephonyClient;
import ru.yandex.direct.telephony.client.TelephonyClientException;
import ru.yandex.direct.telephony.client.model.TelephonyPhoneRequest;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.ServiceNumber;

import static ru.yandex.direct.feature.FeatureName.ABC_NUMBER_TELEPHONY_ALLOWED;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
public class TelephonyPhoneService {

    private static final Logger logger = LoggerFactory.getLogger(TelephonyPhoneService.class);

    static final String MSK_495_REGION_CODE = "495";
    static final String MSK_499_REGION_CODE = "499";
    private static final Set<String> MSK_REGION_CODES = Set.of(MSK_495_REGION_CODE, MSK_499_REGION_CODE);
    /**
     * Максимальное количество попыток зарезервировать номер в Телефонии
     */
    private static final int LINK_RETRIES = 3;

    private final TelephonyClient telephonyClient;
    private final ClientPhoneRepository clientPhoneRepository;
    private final FeatureService featureService;
    private final ShardHelper shardHelper;
    private final BannerCommonRepository newBannerCommonRepository;
    private final MetrikaClient metrikaClient;
    private final CampaignRepository campaignRepository;
    private final BannerRelationsRepository bannerRelationsRepository;

    public TelephonyPhoneService(
            TelephonyClient telephonyClient,
            ClientPhoneRepository clientPhoneRepository,
            FeatureService featureService,
            ShardHelper shardHelper,
            BannerCommonRepository newBannerCommonRepository,
            MetrikaClient metrikaClient,
            CampaignRepository campaignRepository,
            BannerRelationsRepository bannerRelationsRepository
    ) {
        this.telephonyClient = telephonyClient;
        this.clientPhoneRepository = clientPhoneRepository;
        this.featureService = featureService;
        this.shardHelper = shardHelper;
        this.newBannerCommonRepository = newBannerCommonRepository;
        this.metrikaClient = metrikaClient;
        this.campaignRepository = campaignRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
    }

    public static Set<Long> getPhoneIdsIfTelephonyPhoneChanged(List<AppliedChanges<ClientPhone>> appliedChanges) {
        return StreamEx.of(appliedChanges)
                .filter(ac -> ac.changed(ClientPhone.TELEPHONY_PHONE))
                .map(ac -> ac.getModel().getId())
                .toSet();
    }

    /**
     * Запрашиваем номер из телефонии и привязываем с перенаправлением на {@code redirectPhone}.
     * Создаем цель "Звонок" для счетчика, установленного на телефоне через запрос к Метрике
     * <p>
     * Если номер удалось привязать, возвращает заполненный данными от Телефонии объект типа
     * {@link TelephonyPhoneValue}, иначе {@code null}
     */
    @Nullable
    public TelephonyPhoneValue attachTelephony(ClientId clientId, ClientPhone phone, String redirectPhone) {
        return attachTelephony(clientId, phone, redirectPhone, true);
    }

    /**
     * Запрашиваем номер из телефонии и привязываем с перенаправлением на {@link ClientPhone#getPhoneNumber()}
     * <p>
     * Если номер удалось привязать, возвращает заполненный данными от Телефонии объект типа
     * {@link TelephonyPhoneValue}, иначе {@code null}
     *
     * @param needEnableCalltrackingForCounter {@code true}, если необходимо создать цель "Звонок" для счетчика,
     *                                         установленного на телефоне, через запрос к Метрике
     */
    @Nullable
    public TelephonyPhoneValue attachTelephony(ClientPhone phone, boolean needEnableCalltrackingForCounter) {
        return attachTelephony(
                phone.getClientId(),
                phone,
                phone.getPhoneNumber().getPhone(),
                needEnableCalltrackingForCounter
        );
    }

    @Nullable
    private TelephonyPhoneValue attachTelephony(
            ClientId clientId,
            ClientPhone phone,
            String redirectPhone,
            boolean needEnableCalltrackingForCounter
    ) {
        for (int currentRetry = 0; currentRetry < LINK_RETRIES; currentRetry++) {
            try {
                ServiceNumber serviceNumber = getServiceNumber(clientId, redirectPhone);

                if (serviceNumber == null) {
                    logger.warn("No serviceNumber available in Telephony. Skip reserving");
                    return null;
                }

                String serviceId = serviceNumber.getServiceNumberID();
                String number = serviceNumber.getNum();

                TelephonyPhoneRequest request = new TelephonyPhoneRequest()
                        .withCounterId(phone.getCounterId())
                        .withPermalinkId(phone.getPermalinkId())
                        .withRedirectPhone(redirectPhone)
                        .withTelephonyServiceId(serviceId);

                telephonyClient.linkServiceNumber(clientId.asLong(), request, serviceNumber.getVersion());

                if (needEnableCalltrackingForCounter) {
                    tryEnableCalltrackingForCounter(phone.getCounterId());
                }
                return new TelephonyPhoneValue(serviceId, number);
            } catch (TelephonyClientException e) {
                logger.warn("Can't reserve Telephony number", e);
            } catch (StatusRuntimeException e) {
                if (e.getStatus().getCode() == Status.Code.INVALID_ARGUMENT) {
                    // если версия объекта не совпадает с той, которую обновляем, телефония возвращает ошибку вида:
                    // io.grpc.StatusRuntimeException:
                    //     INVALID_ARGUMENT: service number with id xxx and version yyy not found
                    logger.warn("Telephony phone already reserved", e);
                } else {
                    throw e;
                }
            }
        }
        if (phone.getTelephonyServiceId() == null) {
            logger.warn("Can't reserve Telephony number after 3 retries");
        }
        return null;
    }

    /**
     * Отвязать номер Телефонии по заданному {@code serviceNumberId} и отправить номер в карантин
     */
    public void detachTelephony(String serviceNumberId) {
        detachTelephony(serviceNumberId, true);
    }

    /**
     * Отвязать номер Телефонии по заданному {@code serviceNumberId}
     *
     * @param shouldSendToQuarantine {@code true}, если необходимо отправить телефон в карантин
     */
    public void detachTelephony(String serviceNumberId, boolean shouldSendToQuarantine) {
        telephonyClient.unlinkServiceNumber(serviceNumberId, shouldSendToQuarantine);
    }

    /**
     * Заменить номера Телефонии у данных телефонов, сбросив статус синхронизации с БК.
     * Возвращает старые телефоны до обновления, которые удалось успешно обновить.
     */
    public Set<ClientPhone> reassignTelephonyPhones(Collection<ClientPhone> clientPhones) {
        if (clientPhones.isEmpty()) {
            return Collections.emptySet();
        }
        List<ClientPhone> phonesToUpdate = new ArrayList<>(clientPhones);

        Map<Long, ClientPhone> existedPhonesById = StreamEx.of(ClientPhoneUtils.clone(clientPhones))
                .mapToEntry(ClientPhone::getId, Function.identity())
                .toMap();

        for (ClientPhone phone : clientPhones) {
            try {
                String redirectPhone = phone.getPhoneNumber().getPhone();
                ClientId clientId = phone.getClientId();
                ServiceNumber serviceNumber = getServiceNumber(clientId, redirectPhone);

                if (serviceNumber == null) {
                    logger.warn("No serviceNumber available in Telephony. Skip reserving");
                    phonesToUpdate.remove(phone);
                    continue;
                }

                String serviceId = serviceNumber.getServiceNumberID();
                String number = serviceNumber.getNum();

                TelephonyPhoneRequest request = new TelephonyPhoneRequest()
                        .withCounterId(phone.getCounterId())
                        .withPermalinkId(phone.getPermalinkId())
                        .withRedirectPhone(redirectPhone)
                        .withTelephonyServiceId(serviceId);

                telephonyClient.linkServiceNumber(clientId.asLong(), request, serviceNumber.getVersion());

                phone.setTelephonyServiceId(serviceId);
                phone.setTelephonyPhone(new PhoneNumber().withPhone(number));
            } catch (TelephonyClientException e) {
                String telephonyPhone =
                        Optional.ofNullable(phone.getTelephonyPhone())
                                .map(PhoneNumber::getPhone)
                                .orElse(null);
                logger.warn("Cannot update telephony phone {}: {}", telephonyPhone, e);
                phonesToUpdate.remove(phone);
            } catch (StatusRuntimeException e) {
                if (e.getStatus().getCode() == Status.Code.INVALID_ARGUMENT) {
                    // если версия объекта не совпадает с той, которую обновляем, телефония возвращает ошибку вида:
                    // io.grpc.StatusRuntimeException:
                    //     INVALID_ARGUMENT: service number with id xxx and version yyy not found
                    logger.warn("Telephony phone already reserved", e);
                    phonesToUpdate.remove(phone);
                } else {
                    throw e;
                }
            }
        }

        logger.info("Phone ids to update: {}", mapList(phonesToUpdate, ClientPhone::getId));
        Set<ClientPhone> updatedPhones = reassignTelephonyPhones(phonesToUpdate, existedPhonesById);
        updateStatusBsSynced(phonesToUpdate);
        return updatedPhones;
    }

    public Set<String> reassignTelephonyPhonesByServiceIds(Set<String> serviceNumberIds) {
        Set<ClientPhone> phonesForReassign = clientPhoneRepository.getByServiceNumberIds(serviceNumberIds);
        Set<ClientPhone> obsoletePhones = reassignTelephonyPhones(phonesForReassign);
        var obsoleteServiceNumberIds = mapSet(obsoletePhones, ClientPhone::getTelephonyServiceId);
        if (obsoletePhones.size() != phonesForReassign.size()) {
            // Если обновить номера удалось не у всех телефонов, то кидаем ошибку, чтобы попробовать выполнить еще раз
            var serviceNumberIdsForReassign = mapSet(phonesForReassign, ClientPhone::getTelephonyServiceId);
            var failedPhones = Sets.difference(serviceNumberIdsForReassign, obsoleteServiceNumberIds);
            throw new RuntimeException("Cannot reassign telephony phones for " + failedPhones);
        }
        return obsoleteServiceNumberIds;
    }

    private Set<ClientPhone> reassignTelephonyPhones(
            List<ClientPhone> phonesToUpdate,
            Map<Long, ClientPhone> existedPhonesById
    ) {
        Map<ClientId, List<ClientPhone>> phonesToUpdateByClientId = StreamEx.of(phonesToUpdate)
                .mapToEntry(ClientPhone::getClientId, Function.identity())
                .grouping();
        Map<ClientId, List<ModelChanges<ClientPhone>>> modelChanges = EntryStream.of(phonesToUpdateByClientId)
                .mapValues(this::getTelephonyModelChanges)
                .toMap();

        Set<ClientPhone> updatedPhones = new HashSet<>();
        for (ClientId clientId : phonesToUpdateByClientId.keySet()) {
            int shard = shardHelper.getShardByClientId(clientId);
            TelephonyPhoneUpdateOperation operation = new TelephonyPhoneUpdateOperation(
                    shard,
                    clientId,
                    modelChanges.get(clientId),
                    clientPhoneRepository,
                    newBannerCommonRepository
            );
            MassResult<Long> result = operation.prepareAndApply();
            var validationResult = result.getValidationResult();
            if (!result.getResult().isEmpty()) {
                result.getResult().forEach(r -> {
                    if (r.isSuccessful()) {
                        ClientPhone oldPhone = existedPhonesById.get(r.getResult());
                        detachTelephony(oldPhone.getTelephonyServiceId());
                        updatedPhones.add(oldPhone);
                    }
                });
            }
            if (validationResult.hasAnyErrors()) {
                String errorsStr = validationResult.flattenErrors().stream()
                        .map(DefectInfo::toString)
                        .collect(Collectors.joining(", "));
                logger.warn("Errors on update phones: {}", errorsStr);
            }
        }
        return updatedPhones;
    }

    public void setCalltrackingOnSitePhonesDeleted(int shard, ClientId clientId,
                                                   Collection<ClientPhone> updatedPhones) {
        LocalDateTime now = LocalDateTime.now();
        List<ModelChanges<ClientPhone>> modelChanges = mapList(updatedPhones, clientPhone ->
                new ModelChanges<>(clientPhone.getId(), ClientPhone.class)
                        .process(true, ClientPhone.IS_DELETED)
                        .process(now, ClientPhone.LAST_SHOW_TIME)
        );
        CalltrackingOnSitePhoneUpdateOperation operation = new CalltrackingOnSitePhoneUpdateOperation(
                shard,
                clientId,
                modelChanges,
                clientPhoneRepository);
        operation.prepareAndApply();
    }

    public void updateCounterIds(int shard,
                                 ClientId clientId,
                                 List<Pair<ClientPhone, Long>> clientPhonesToNewCounterId) {
        var modelChanges = mapList(clientPhonesToNewCounterId, clientPhoneAndNewCounterId ->
                new ModelChanges<>(clientPhoneAndNewCounterId.getKey().getId(), ClientPhone.class)
                        .process(clientPhoneAndNewCounterId.getValue(), ClientPhone.COUNTER_ID)
        );
        CalltrackingOnSitePhoneUpdateOperation operation = new CalltrackingOnSitePhoneUpdateOperation(
                shard,
                clientId,
                modelChanges,
                clientPhoneRepository);
        operation.prepareAndApply();

        for (var clientPhoneAndNewCounterId : clientPhonesToNewCounterId) {
            ClientPhone clientPhone = clientPhoneAndNewCounterId.getKey();
            long newCounterId = clientPhoneAndNewCounterId.getValue();
            if (clientPhone.getTelephonyServiceId() == null) {
                continue;
            }

            var request = new TelephonyPhoneRequest()
                    .withCounterId(newCounterId)
                    .withRedirectPhone(clientPhone.getPhoneNumber().getPhone())
                    .withTelephonyServiceId(clientPhone.getTelephonyServiceId())
                    .withPermalinkId(0L);
            telephonyClient.linkServiceNumber(clientId.asLong(), request);
        }
    }

    /**
     * Получить все привязанные телефоны Телефонии
     */
    public Map<TelephonyPhoneType, List<ClientPhone>> getAttachedTelephonyPhones(int shard) {
        return getTelephonyPhones(shard, TelephonyPhoneState.ATTACHED);
    }

    /**
     * Получить все отвязанные телефоны Телефонии
     */
    public Map<TelephonyPhoneType, List<ClientPhone>> getDetachedTelephonyPhones(int shard) {
        return getTelephonyPhones(shard, TelephonyPhoneState.DETACHED);
    }

    /**
     * Получить все телефоны Телефонии
     */
    public Map<TelephonyPhoneType, List<ClientPhone>> getTelephonyPhones(int shard) {
        List<ClientPhone> allPhones = clientPhoneRepository.getAllTelephonyPhones(shard, null);
        logger.info("Found {} phones", allPhones.size());
        return StreamEx.of(allPhones)
                .groupingBy(p -> p.getPermalinkId() == null ? TelephonyPhoneType.SITE : TelephonyPhoneType.ADV);
    }

    /**
     * Получить все телефоны Телефонии для коллтрекинга на сайте c флагом is_deleted = 1
     */
    public Map<TelephonyPhoneState, List<ClientPhone>> getDeletedTelephonyPhones(
            int shard,
            LocalDateTime latestLastShowTime
    ) {
        List<ClientPhone> allPhones = clientPhoneRepository.getDeletedTelephonyPhones(shard, latestLastShowTime);
        logger.info("Found {} phones", allPhones.size());
        return StreamEx.of(allPhones)
                .groupingBy(p ->
                        p.getTelephonyPhone() == null ? TelephonyPhoneState.DETACHED : TelephonyPhoneState.ATTACHED);
    }

    /**
     * Получить все телефоны Телефонии, в статусе {@code state}
     */
    public Map<TelephonyPhoneType, List<ClientPhone>> getTelephonyPhones(int shard, TelephonyPhoneState state) {
        List<ClientPhone> phones = clientPhoneRepository.getAllTelephonyPhones(shard, state);
        logger.info("Found {} phones", phones.size());
        return StreamEx.of(phones)
                .groupingBy(p -> p.getPermalinkId() == null ? TelephonyPhoneType.SITE : TelephonyPhoneType.ADV);
    }

    public Map<ClientPhone, LocalDateTime> getLastShowTimesByPhone(int shard, List<ClientPhone> phones) {
        var phoneByIds = listToMap(phones, ClientPhone::getId, Function.identity());
        var campaignIdsByPhoneId = getRelatedCampaignIdsByPhoneId(shard, phoneByIds);
        var campaignIds = StreamEx.ofValues(campaignIdsByPhoneId).flatMap(StreamEx::of).toSet();
        var lastShowTimeByCampaignId = campaignRepository.getLastShowTimeByCampaignId(shard, campaignIds);

        Map<ClientPhone, LocalDateTime> lastShowTimeByPhone = new IdentityHashMap<>();
        phoneByIds.forEach((id, phone) -> {
            Set<Long> phoneCampaignIds = campaignIdsByPhoneId.get(phone.getId());
            if (!isEmpty(phoneCampaignIds)) {
                LocalDateTime latestTime = computeLatestLastShowTime(phoneCampaignIds, lastShowTimeByCampaignId);
                lastShowTimeByPhone.put(phone, latestTime);
            }
        });
        return lastShowTimeByPhone;
    }

    private Map<Long, Set<Long>> getRelatedCampaignIdsByPhoneId(int shard, Map<Long, ClientPhone> phoneByIds) {
        var campaignIdsByPhoneId = clientPhoneRepository.getCampaignIdsByPhoneId(shard, phoneByIds.keySet());
        var bannerIdsByPhoneId = clientPhoneRepository.getBannerIdsByPhoneId(shard, phoneByIds.keySet());
        var bannerIds = StreamEx.ofValues(bannerIdsByPhoneId).flatMap(StreamEx::of).toSet();
        var campaignIdsByBannerId = bannerRelationsRepository.getCampaignIdsByBannerIds(shard, bannerIds);

        Map<Long, Set<Long>> relatedCampaignIdsByPhoneId = new HashMap<>();
        campaignIdsByPhoneId.forEach((phoneId, phoneCampaignIds) ->
                relatedCampaignIdsByPhoneId.computeIfAbsent(phoneId, v -> new HashSet<>()).addAll(phoneCampaignIds)
        );
        // Достаем кампании со всех баннеров, к которым привязаны телефоны
        bannerIdsByPhoneId.forEach((phoneId, phoneBannerIds) ->
                phoneBannerIds.forEach(bannerId -> {
                    Long phoneCampaignId = campaignIdsByBannerId.get(bannerId);
                    relatedCampaignIdsByPhoneId.computeIfAbsent(phoneId, v -> new HashSet<>()).add(phoneCampaignId);
                })
        );
        return relatedCampaignIdsByPhoneId;
    }

    private LocalDateTime computeLatestLastShowTime(
            Set<Long> campaignIds,
            Map<Long, LocalDateTime> lastShowTimeByCampaignId
    ) {
        LocalDateTime maxTime = null;
        for (Long campaignId : campaignIds) {
            LocalDateTime time = lastShowTimeByCampaignId.get(campaignId);
            if (maxTime == null || (time != null && time.isAfter(maxTime))) {
                maxTime = time;
            }
        }
        return maxTime;
    }

    private void updateStatusBsSynced(List<ClientPhone> phonesToUpdate) {
        Map<ClientId, List<ClientPhone>> phonesToUpdateByClientId = StreamEx.of(phonesToUpdate)
                .mapToEntry(ClientPhone::getClientId, Function.identity())
                .grouping();

        Map<Integer, List<Long>> clientPhoneIdsByShard = EntryStream.of(phonesToUpdateByClientId)
                .mapKeys(shardHelper::getShardByClientId)
                .mapValues(phones -> mapList(phones, ClientPhone::getId))
                .flatMapValues(Collection::stream)
                .grouping();

        clientPhoneIdsByShard.forEach(this::resetStatusBsSynced);
    }

    /**
     * Копия из {@code ClientPhoneUpdateOperation}
     */
    public void resetStatusBsSynced(int shard, Collection<Long> clientPhoneIds) {
        Map<Long, List<Long>> bannerIdsByPhoneId =
                clientPhoneRepository.getBannerIdsByPhoneId(shard, clientPhoneIds);
        List<Long> bannerIds = StreamEx.of(bannerIdsByPhoneId.values())
                .flatCollection(Function.identity())
                .toList();
        logger.info("Reset statusBsSynced for {} banners. BannerIds: {}", bannerIds.size(), bannerIds);
        newBannerCommonRepository.resetStatusBsSyncedByIds(shard, bannerIds);
    }

    private List<ModelChanges<ClientPhone>> getTelephonyModelChanges(List<ClientPhone> phones) {
        return StreamEx.of(phones)
                .map(phone -> new ModelChanges<>(phone.getId(), ClientPhone.class)
                        .process(phone.getTelephonyPhone(), ClientPhone.TELEPHONY_PHONE)
                        .process(phone.getTelephonyServiceId(), ClientPhone.TELEPHONY_SERVICE_ID))
                .toList();
    }

    /**
     * Получить номер Телефонии
     * <p>
     * Если {@link ru.yandex.direct.feature.FeatureName#ABC_NUMBER_TELEPHONY_ALLOWED} выключена,
     * то запрашиваем у Телефонии DEF телефон.
     * <p>
     * Если включена, то для организаций, главный номер которых является московским, запрашивается ABC-номер
     * с таким же кодом города; в остальных случаях, DEF-номер.
     * Если запросить ABC-номер с таким же кодом города не удалось (пул исчерпан), то запрашиваем DEF-номер.
     */
    @Nullable
    ServiceNumber getServiceNumber(ClientId clientId, String orgPhone) {
        boolean abcNumberAllowed = featureService.isEnabledForClientId(clientId, ABC_NUMBER_TELEPHONY_ALLOWED);
        if (!abcNumberAllowed) {
            logger.info("Feature {} is not enabled for client {}", ABC_NUMBER_TELEPHONY_ALLOWED.getName(), clientId);
            return telephonyClient.getServiceNumber();
        }
        String regionCode = ClientPhoneUtils.getRegionCode(orgPhone);
        if (!MSK_REGION_CODES.contains(regionCode)) {
            logger.info("Not a Moscow region code for organization phone {}. DEF phone number will be used", orgPhone);
            return telephonyClient.getServiceNumber();
        }
        return MSK_499_REGION_CODE.equals(regionCode)
                ? telephonyClient.tryToGetServiceNumber(MSK_499_REGION_CODE)
                : telephonyClient.tryToGetServiceNumber(MSK_495_REGION_CODE);
    }

    public void tryEnableCalltrackingForCounters(Set<Long> counterIds) {
        for (var counterId : counterIds) {
            tryEnableCalltrackingForCounter(counterId);
        }
    }

    private void tryEnableCalltrackingForCounter(Long counterId) {
        try {
            TurnOnCallTrackingResponse metrikaResponse = metrikaClient.turnOnCallTracking(counterId);
            logger.info("Response from direct/turn_on_call_tracking: {}", metrikaResponse);
        } catch (RuntimeException e) {
            logger.warn("Can't enable calltracking for counter {}", counterId, e);
        }
    }
}
