package ru.yandex.direct.core.entity.clientphone;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.clientphone.validation.ClientPhoneValidationService;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class TelephonyPhoneUpdateOperation extends SimpleAbstractUpdateOperation<ClientPhone, Long> {

    private final ClientPhoneRepository clientPhoneRepository;
    private final ModelChangesValidationTool updateValidationTool;
    private final BannerCommonRepository bannerCommonRepository;

    private final int shard;
    private final ClientId clientId;

    public TelephonyPhoneUpdateOperation(
            int shard,
            ClientId clientId,
            List<ModelChanges<ClientPhone>> modelChanges,
            ClientPhoneRepository clientPhoneRepository,
            BannerCommonRepository bannerCommonRepository
    ) {
        super(Applicability.PARTIAL, modelChanges, id -> new ClientPhone().withId(id));
        this.shard = shard;
        this.clientId = clientId;
        this.clientPhoneRepository = clientPhoneRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        updateValidationTool = ModelChangesValidationTool.builder().build();
    }

    @Override
    protected ValidationResult<List<ModelChanges<ClientPhone>>, Defect> validateModelChanges(
            List<ModelChanges<ClientPhone>> modelChanges
    ) {
        List<Long> phoneIds = mapList(modelChanges, ModelChanges::getId);
        List<ClientPhone> byPhoneIds = clientPhoneRepository.getByPhoneIds(clientId, phoneIds);
        Set<Long> existedIds = listToSet(byPhoneIds, ClientPhone::getId);
        return updateValidationTool.validateModelChangesList(modelChanges, existedIds);
    }

    @Override
    protected ValidationResult<List<ClientPhone>, Defect> validateAppliedChanges(
            ValidationResult<List<ClientPhone>, Defect> validationResult) {
        return ClientPhoneValidationService.validateTelephony(validationResult);
    }

    @Override
    protected Collection<ClientPhone> getModels(Collection<Long> phoneIds) {
        return clientPhoneRepository.getByPhoneIds(clientId, phoneIds);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<ClientPhone>> appliedChanges) {
        clientPhoneRepository.update(shard, appliedChanges);
        return mapList(appliedChanges, c -> c.getModel().getId());
    }

    @Override
    protected void afterExecution(ExecutionStep<ClientPhone> executionStep) {
        // Если номер редиректа обновился, то нужно сбросить статус синхронизации с БК для всех баннеров,
        // где установлен такой телефон. Это важно, потому что мы можем ображать номер редиректа
        // на выдаче, пока номер Телефонии оторван
        var updatedRedirectPhoneIds = StreamEx.of(executionStep.getValidModelChanges())
                .filter(mc -> mc.isPropChanged(ClientPhone.PHONE_NUMBER))
                .map(ModelChanges::getId)
                .toSet();
        var bannerIdsByPhoneId = clientPhoneRepository.getBannerIdsByPhoneId(shard, updatedRedirectPhoneIds);
        var bannerIds = StreamEx.ofValues(bannerIdsByPhoneId).flatMap(StreamEx::of).toSet();
        bannerCommonRepository.resetStatusBsSyncedByIds(shard, bannerIds);
    }

}
