package ru.yandex.direct.core.entity.clientphone.repository;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings;
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneState;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhoneType;
import ru.yandex.direct.dbschema.ppc.enums.ClientPhonesPhoneType;
import ru.yandex.direct.dbschema.ppc.tables.records.ClientPhonesRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.UpdateHelper;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.clientIdProperty;
import static ru.yandex.direct.core.entity.trackingphone.model.ClientPhoneType.TELEPHONY;
import static ru.yandex.direct.dbschema.ppc.tables.BannerPhones.BANNER_PHONES;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.CalltrackingSettings.CALLTRACKING_SETTINGS;
import static ru.yandex.direct.dbschema.ppc.tables.CampCalltrackingPhones.CAMP_CALLTRACKING_PHONES;
import static ru.yandex.direct.dbschema.ppc.tables.CampCalltrackingSettings.CAMP_CALLTRACKING_SETTINGS;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignPhones.CAMPAIGN_PHONES;
import static ru.yandex.direct.dbschema.ppc.tables.ClientPhones.CLIENT_PHONES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapperhelper.InsertHelper.saveModelObjectsToDbTable;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class ClientPhoneRepository {

    public static final int BANNER_IDS_CHUNK_SIZE = 2500;

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final JooqMapperWithSupplier<ClientPhone> mapper = createMapper();

    public ClientPhoneRepository(DslContextProvider dslContextProvider,
                                 ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    private static JooqMapperWithSupplier<ClientPhone> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(ClientPhone::new)
                .map(property(ClientPhone.ID, CLIENT_PHONES.CLIENT_PHONE_ID))
                .map(clientIdProperty(ClientPhone.CLIENT_ID, CLIENT_PHONES.CLIENT_ID))
                .map(convertibleProperty(ClientPhone.PHONE_NUMBER, CLIENT_PHONES.PHONE,
                        ClientPhoneMapping::phoneNumberFromDb, ClientPhoneMapping::phoneNumberToDb))
                .map(convertibleProperty(ClientPhone.COMMENT, CLIENT_PHONES.COMMENT,
                        c -> c, RepositoryUtils::nullableStringToNotNullDbField))
                .map(convertibleProperty(ClientPhone.PHONE_TYPE, CLIENT_PHONES.PHONE_TYPE,
                        ClientPhoneType::fromSource, ClientPhoneType::toSource))
                .map(convertibleProperty(ClientPhone.COUNTER_ID, CLIENT_PHONES.COUNTER_ID,
                        RepositoryUtils::zeroToNull, RepositoryUtils::nullToZero))
                .map(convertibleProperty(ClientPhone.PERMALINK_ID, CLIENT_PHONES.PERMALINK_ID,
                        RepositoryUtils::zeroToNull, RepositoryUtils::nullToZero))
                .map(convertibleProperty(ClientPhone.TELEPHONY_SERVICE_ID, CLIENT_PHONES.TELEPHONY_SERVICE_ID,
                        RepositoryUtils::nullableStringFromNotNullDbField,
                        RepositoryUtils::nullableStringToNotNullDbField))
                .map(convertibleProperty(ClientPhone.TELEPHONY_PHONE, CLIENT_PHONES.TELEPHONY_PHONE,
                        ClientPhoneMapping::phoneNumberFromDb, ClientPhoneMapping::phoneNumberToDb))
                .map(convertibleProperty(ClientPhone.LAST_SHOW_TIME, CLIENT_PHONES.LAST_SHOW_TIME,
                        t -> t, t -> nvl(t, LocalDateTime.now())))
                .map(convertibleProperty(ClientPhone.IS_DELETED, CLIENT_PHONES.IS_DELETED,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::nullSafeBooleanToLong))
                .build();
    }

    public List<ClientPhone> getByClientId(ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(CLIENT_PHONES.CLIENT_ID.eq(clientId.asLong()))
                .fetch(mapper::fromDb);
    }

    public List<ClientPhone> getTelephonyPhonesBySettings(ClientId clientId, List<CalltrackingSettings> settings) {
        int shard = shardHelper.getShardByClientId(clientId);

        List<Long> settingIds = mapList(settings, CalltrackingSettings::getId);
        var isTelephonyPhone = CLIENT_PHONES.PHONE_TYPE.eq(ClientPhoneType.toSource(TELEPHONY));
        var isAttached = getTelephonyPhoneStateCondition(TelephonyPhoneState.ATTACHED);

        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .join(CAMP_CALLTRACKING_PHONES)
                .on(CLIENT_PHONES.CLIENT_PHONE_ID.eq(CAMP_CALLTRACKING_PHONES.CLIENT_PHONE_ID))
                .join(CAMP_CALLTRACKING_SETTINGS)
                .on(CAMP_CALLTRACKING_PHONES.CID.eq(CAMP_CALLTRACKING_SETTINGS.CID))
                .join(CALLTRACKING_SETTINGS)
                .on(CAMP_CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID
                        .eq(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID))
                .where(CLIENT_PHONES.CLIENT_ID.eq(clientId.asLong()))
                .and(isTelephonyPhone)
                .and(isAttached)
                .and(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID.in(settingIds))
                .fetch(mapper::fromDb);
    }

    public List<ClientPhone> getByIds(ClientId clientId, Collection<Long> clientPhoneIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        Condition idCondition = CLIENT_PHONES.CLIENT_PHONE_ID.in(clientPhoneIds);
        Condition clientIdCondition = CLIENT_PHONES.CLIENT_ID.eq(clientId.asLong());
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(idCondition.and(clientIdCondition))
                .fetch(mapper::fromDb);
    }

    /**
     * Возвращает список номеров, которые принадлежат указанным организациями
     * или имеют тип manual (для личных телефонов клиента permalinkId не указан)
     */
    public List<ClientPhone> getAllClientPhones(ClientId clientId, Collection<Long> permalinkIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(CLIENT_PHONES.CLIENT_ID.eq(clientId.asLong())
                        .and(CLIENT_PHONES.PERMALINK_ID.in(permalinkIds)
                                .or(CLIENT_PHONES.PHONE_TYPE.eq(ClientPhoneType.toSource(ClientPhoneType.MANUAL)))))
                .fetch(mapper::fromDb);
    }

    public Map<Long, List<ClientPhone>> getTelephonyPhonesBySettingIds(ClientId clientId, List<Long> settingIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        var isTelephonyPhone = CLIENT_PHONES.PHONE_TYPE.eq(ClientPhoneType.toSource(TELEPHONY));
        Set<Field<?>> fields = mapper.getFieldsToRead();
        fields.add(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID);
        return dslContextProvider.ppc(shard)
                .select(fields)
                .from(CLIENT_PHONES)
                .join(CAMP_CALLTRACKING_PHONES)
                .on(CLIENT_PHONES.CLIENT_PHONE_ID.eq(CAMP_CALLTRACKING_PHONES.CLIENT_PHONE_ID))
                .join(CAMP_CALLTRACKING_SETTINGS)
                .on(CAMP_CALLTRACKING_PHONES.CID.eq(CAMP_CALLTRACKING_SETTINGS.CID))
                .join(CALLTRACKING_SETTINGS)
                .on(CAMP_CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID
                        .eq(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID))
                .where(CLIENT_PHONES.CLIENT_ID.eq(clientId.asLong()))
                .and(CLIENT_PHONES.IS_DELETED.eq(0L))
                .and(isTelephonyPhone)
                .and(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID.in(settingIds))
                .fetchGroups(
                        r -> r.getValue(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID),
                        mapper::fromDb
                );
    }

    /**
     * Возвращает список ручных телефонов клиента в формате,
     * в котором они сохранены в БД: [код страны]#[код города]#[номер]#[добавочный]
     */
    public Map<Long, String> getManualPhoneNumbers(ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        var isManual = CLIENT_PHONES.PHONE_TYPE.eq(ClientPhoneType.toSource(ClientPhoneType.MANUAL));
        return dslContextProvider.ppc(shard)
                .select(CLIENT_PHONES.CLIENT_PHONE_ID, CLIENT_PHONES.PHONE)
                .from(CLIENT_PHONES)
                .where(CLIENT_PHONES.CLIENT_ID.eq(clientId.asLong()).and(isManual))
                .fetchMap(CLIENT_PHONES.CLIENT_PHONE_ID, CLIENT_PHONES.PHONE);
    }

    public List<ClientPhone> getAllClientOrganizationPhones(ClientId clientId, Collection<Long> permalinkIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return getAllClientOrganizationPhones(dslContextProvider.ppc(shard), clientId, permalinkIds);
    }

    public List<ClientPhone> getAllClientOrganizationPhones(
            DSLContext dslContext,
            ClientId clientId,
            Collection<Long> permalinkIds) {
        return dslContext
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(CLIENT_PHONES.CLIENT_ID.eq(clientId.asLong())
                        .and(CLIENT_PHONES.PERMALINK_ID.in(permalinkIds)))
                .fetch(mapper::fromDb);
    }

    public List<ClientPhone> getByPhoneIds(ClientId clientId, Collection<Long> phoneIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(CLIENT_PHONES.CLIENT_ID.eq(clientId.asLong()))
                .and(CLIENT_PHONES.CLIENT_PHONE_ID.in(phoneIds))
                .fetch(mapper::fromDb);
    }

    public Set<ClientPhone> getByServiceNumberIds(Collection<String> serviceNumberIds) {
        Set<ClientPhone> phones = new HashSet<>();
        shardHelper.forEachShard(shard -> phones.addAll(dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(CLIENT_PHONES.TELEPHONY_SERVICE_ID.in(serviceNumberIds))
                .fetchSet(mapper::fromDb)));
        return phones;
    }

    @QueryWithoutIndex("Таблица небольшая, запрос используется только во внутренних отчётах")
    public List<ClientPhone> getByTelephonyNumbers(List<String> numbers) {
        List<ClientPhone> phones = new ArrayList<>();
        shardHelper.forEachShard(shard -> phones.addAll(dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(CLIENT_PHONES.TELEPHONY_PHONE.in(numbers))
                .fetch(mapper::fromDb)));
        return phones;
    }

    /**
     * Получить отображение: номер телефона клиента, который подменяли -> номера Телефонии
     * <p>
     * (!) Все телефоны и на входе и на выходе в формате "[код страны]#[код города]#[номер]#[добавочный]"
     */
    public Map<String, List<String>> getTelephohesByOrigin(
            int shard,
            ClientId clientId,
            Collection<String> telephonyPhones
    ) {
        Condition clientIdCondition = CLIENT_PHONES.CLIENT_ID.eq(clientId.asLong());
        Condition isTelephony = CLIENT_PHONES.PHONE_TYPE.eq(ClientPhonesPhoneType.telephony);
        Condition phoneCondition = CLIENT_PHONES.TELEPHONY_PHONE.in(telephonyPhones);
        return dslContextProvider.ppc(shard)
                .select(CLIENT_PHONES.PHONE, CLIENT_PHONES.TELEPHONY_PHONE)
                .from(CLIENT_PHONES)
                .where(clientIdCondition.and(isTelephony).and(phoneCondition))
                .fetchGroups(CLIENT_PHONES.PHONE, CLIENT_PHONES.TELEPHONY_PHONE);
    }

    public List<ClientPhone> getByPermalinkIds(ClientId clientId, Collection<Long> permalinkIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(CLIENT_PHONES.CLIENT_ID.eq(clientId.asLong()))
                .and(CLIENT_PHONES.PERMALINK_ID.in(permalinkIds))
                .fetch(mapper::fromDb);
    }

    public List<ClientPhone> getTelephony(int shard, Collection<Long> clientPhoneIds) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(CLIENT_PHONES.CLIENT_PHONE_ID.in(clientPhoneIds)
                        .and(CLIENT_PHONES.PHONE_TYPE.eq(ClientPhoneType.toSource(TELEPHONY)))
                )
                .fetch(mapper::fromDb);
    }

    public List<ClientPhone> getTelephony(Collection<Long> phoneIds) {
        List<ClientPhone> result = new ArrayList<>();
        shardHelper.forEachShard(shard -> result.addAll(getTelephony(shard, phoneIds)));
        return result;
    }

    public List<ClientPhone> getTelephony(ClientId clientId, Collection<Long> phoneIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return getTelephony(shard, phoneIds);
    }

    /**
     * Используется только для внутреннего отчета на ТС и devtest
     * Возвращаются только привязанные телефоны
     */
    public List<ClientPhone> getAllTelephonyPhones(int shard) {
        return getAllTelephonyPhones(shard, TelephonyPhoneState.ATTACHED);
    }

    /**
     * Вернуть все телефоны Телефонии для указанного шарда
     * <p>
     * Если {@code phoneState == TelephonyPhoneState#DETACHED},
     * то вернуть только оторванные номера Телефонии
     * <p>
     * Если {@code phoneState == TelephonyPhoneState#ATTACHED},
     * то вернуть только привязанные номера Телефонии
     * <p>
     * Если {@code phoneState == null}, вернуть все номера, которые имеют тип {@link ClientPhoneType#TELEPHONY}
     */
    public List<ClientPhone> getAllTelephonyPhones(int shard, @Nullable TelephonyPhoneState phoneState) {
        var phoneStateCondition = getTelephonyPhoneStateCondition(phoneState);
        var isTelephonyPhone = CLIENT_PHONES.PHONE_TYPE.eq(ClientPhoneType.toSource(TELEPHONY));
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(isTelephonyPhone.and(phoneStateCondition))
                .fetch(mapper::fromDb);
    }

    /**
     * Вернуть все телефоны Телефонии для коллтрекинга на сайте, c флагом is_deleted = 1.
     * Телефоны Телефонии для коллтрекинга на сайте имеют значение PERMALINK_ID = 0
     */
    public List<ClientPhone> getDeletedTelephonyPhones(int shard, LocalDateTime latestLastShowTime) {
        var isTelephonyPhone = CLIENT_PHONES.PHONE_TYPE.eq(ClientPhoneType.toSource(TELEPHONY));
        var isCalltrackingOnSite = CLIENT_PHONES.PERMALINK_ID.eq(0L);
        var isDeleted = CLIENT_PHONES.IS_DELETED.eq(RepositoryUtils.TRUE);
        var lastShowTimeLessOrEqThenNow = CLIENT_PHONES.LAST_SHOW_TIME.lessOrEqual(latestLastShowTime);
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(isTelephonyPhone
                        .and(isCalltrackingOnSite)
                        .and(isDeleted)
                        .and(lastShowTimeLessOrEqThenNow))
                .fetch(mapper::fromDb);
    }

    public List<ClientPhone> getPhonesByServiceId(int shard, String serviceId) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(CLIENT_PHONES.TELEPHONY_SERVICE_ID.eq(serviceId))
                .fetch(mapper::fromDb);
    }

    public List<Long> add(ClientId clientId, List<ClientPhone> clientPhones) {
        int shard = shardHelper.getShardByClientId(clientId);
        List<Long> ids = shardHelper.generateClientPhoneIds(clientPhones.size());
        StreamEx.of(clientPhones).zipWith(ids.stream())
                .forKeyValue(ClientPhone::setId);
        saveModelObjectsToDbTable(dslContextProvider.ppc(shard), CLIENT_PHONES, mapper, clientPhones);
        return ids;
    }

    public List<Long> addOrUpdate(ClientId clientId, List<ClientPhone> clientPhones) {
        if (clientPhones.isEmpty()) {
            return emptyList();
        }
        int shard = shardHelper.getShardByClientId(clientId);
        generateAndSetIds(clientPhones);

        InsertHelper<ClientPhonesRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), CLIENT_PHONES)
                        .addAll(mapper, clientPhones);

        if (insertHelper.hasAddedRecords()) {
            insertHelper
                    .onDuplicateKeyUpdate()
                    .set(CLIENT_PHONES.PHONE, MySQLDSL.values(CLIENT_PHONES.PHONE))
                    .set(CLIENT_PHONES.COMMENT, MySQLDSL.values(CLIENT_PHONES.COMMENT))
                    .set(CLIENT_PHONES.COUNTER_ID, MySQLDSL.values(CLIENT_PHONES.COUNTER_ID))
                    .set(CLIENT_PHONES.PERMALINK_ID, MySQLDSL.values(CLIENT_PHONES.PERMALINK_ID))
                    .set(CLIENT_PHONES.PHONE_TYPE, MySQLDSL.values(CLIENT_PHONES.PHONE_TYPE))
                    .set(CLIENT_PHONES.TELEPHONY_SERVICE_ID, MySQLDSL.values(CLIENT_PHONES.TELEPHONY_SERVICE_ID));
        }

        insertHelper.executeIfRecordsAdded();
        return mapList(clientPhones, ClientPhone::getId);
    }

    private void generateAndSetIds(Collection<ClientPhone> clientPhones) {
        List<ClientPhone> newPhones = filterList(clientPhones, p -> p.getId() == null);
        Iterator<Long> ids = shardHelper.generateClientPhoneIds(newPhones.size()).iterator();
        newPhones.forEach(phone -> phone.setId(ids.next()));
    }

    public void update(int shard, Collection<AppliedChanges<ClientPhone>> appliedChanges) {
        DSLContext context = dslContextProvider.ppc(shard);

        new UpdateHelper<>(context, CLIENT_PHONES.CLIENT_PHONE_ID)
                .processUpdateAll(mapper, appliedChanges)
                .execute();
    }

    public void delete(ClientId clientId, Collection<Long> clientPhoneIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        dslContextProvider.ppc(shard)
                .deleteFrom(CLIENT_PHONES)
                .where(CLIENT_PHONES.CLIENT_ID.eq(clientId.asLong())
                        .and(CLIENT_PHONES.CLIENT_PHONE_ID.in(clientPhoneIds)))
                .execute();
    }

    public void delete(int shard, Collection<Long> clientIds, Collection<Long> clientPhoneIds) {
        dslContextProvider.ppc(shard)
                .deleteFrom(CLIENT_PHONES)
                .where(CLIENT_PHONES.CLIENT_ID.in(clientIds)
                        .and(CLIENT_PHONES.CLIENT_PHONE_ID.in(clientPhoneIds)))
                .execute();
    }

    @Nonnull
    public Map<Long, Long> getPhoneIdsByBannerIds(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }
        Map<Long, Long> result = new HashMap<>();
        for (var chunkedBannerIds : Iterables.partition(bannerIds, BANNER_IDS_CHUNK_SIZE)) {
            result.putAll(dslContextProvider.ppc(shard)
                    .select(BANNER_PHONES.BID, BANNER_PHONES.CLIENT_PHONE_ID)
                    .from(BANNER_PHONES)
                    .where(BANNER_PHONES.BID.in(chunkedBannerIds))
                    .fetchMap(BANNER_PHONES.BID, BANNER_PHONES.CLIENT_PHONE_ID));
        }
        return result;
    }

    @Nonnull
    public Map<Long, List<Long>> getBannerIdsByPhoneId(int shard, Collection<Long> phoneIds) {
        if (phoneIds.isEmpty()) {
            return emptyMap();
        }
        Map<Long, List<Long>> result = new HashMap<>();
        for (var chunkedPhoneIds : Iterables.partition(phoneIds, BANNER_IDS_CHUNK_SIZE)) {
            result.putAll(dslContextProvider.ppc(shard)
                    .select(BANNER_PHONES.CLIENT_PHONE_ID, BANNER_PHONES.BID)
                    .from(BANNER_PHONES)
                    .where(BANNER_PHONES.CLIENT_PHONE_ID.in(chunkedPhoneIds))
                    .fetchGroups(BANNER_PHONES.CLIENT_PHONE_ID, BANNER_PHONES.BID));
        }
        return result;
    }

    @Nonnull
    public Map<Long, List<Long>> getBannerIdsByPhoneId(ClientId clientId, Collection<Long> phoneIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return getBannerIdsByPhoneId(shard, phoneIds);
    }

    @Nonnull
    public Map<Long, List<Long>> getCampaignIdsByPhoneId(int shard, Collection<Long> phoneIds) {
        if (phoneIds.isEmpty()) {
            return emptyMap();
        }
        Map<Long, List<Long>> result = new HashMap<>();
        for (var chunkedPhoneIds : Iterables.partition(phoneIds, BANNER_IDS_CHUNK_SIZE)) {
            result.putAll(dslContextProvider.ppc(shard)
                    .select(CAMPAIGN_PHONES.CLIENT_PHONE_ID, CAMPAIGN_PHONES.CID)
                    .from(CAMPAIGN_PHONES)
                    .where(CAMPAIGN_PHONES.CLIENT_PHONE_ID.in(chunkedPhoneIds))
                    .fetchGroups(CAMPAIGN_PHONES.CLIENT_PHONE_ID, CAMPAIGN_PHONES.CID));
        }
        return result;
    }

    @Nonnull
    public Map<Long, List<Long>> getCampaignIdsByPhoneId(ClientId clientId, Collection<Long> phoneIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return getCampaignIdsByPhoneId(shard, phoneIds);
    }

    /**
     * Создает связки ID баннера с ID подменного номера телефона при добавлении баннера
     * <p>
     * Телефон не может быть равен {@code null}
     */
    public void linkBannerPhones(int shard, Map<Long, Long> phoneIdsByBannerId) {
        linkBannerPhones(dslContextProvider.ppc(shard), phoneIdsByBannerId);
    }

    public void linkBannerPhones(DSLContext context, Map<Long, Long> phoneIdsByBannerId) {
        if (phoneIdsByBannerId.isEmpty()) {
            return;
        }

        boolean hasNullPhone = phoneIdsByBannerId.values().stream().anyMatch(Objects::isNull);
        checkArgument(!hasNullPhone, "client_phone_id == null");

        var insertHelper = new InsertHelper<>(context, BANNER_PHONES);
        phoneIdsByBannerId.forEach((bid, phoneId) -> insertHelper
                .set(BANNER_PHONES.BID, bid)
                .set(BANNER_PHONES.CLIENT_PHONE_ID, phoneId)
                .newRecord());
        insertHelper
                .onDuplicateKeyUpdate()
                .set(BANNER_PHONES.CLIENT_PHONE_ID, MySQLDSL.values(BANNER_PHONES.CLIENT_PHONE_ID))
                .executeIfRecordsAdded();

        var now = LocalDateTime.now();
        context
                .update(BANNERS)
                .set(BANNERS.LAST_CHANGE, now)
                .where(BANNERS.BID.in(phoneIdsByBannerId.keySet()))
                .execute();
    }

    public void linkCampaignPhones(int shard, Map<Long, Long> phoneIdByCampaignId) {
        if (phoneIdByCampaignId.isEmpty()) {
            return;
        }

        boolean hasNullPhone = phoneIdByCampaignId.values().stream().anyMatch(Objects::isNull);
        checkArgument(!hasNullPhone, "client_phone_id == null");

        var insertHelper = new InsertHelper<>(dslContextProvider.ppc(shard), CAMPAIGN_PHONES);
        phoneIdByCampaignId.forEach((cid, phoneId) -> insertHelper
                .set(CAMPAIGN_PHONES.CID, cid)
                .set(CAMPAIGN_PHONES.CLIENT_PHONE_ID, phoneId)
                .newRecord());
        insertHelper
                .onDuplicateKeyUpdate()
                .set(CAMPAIGN_PHONES.CLIENT_PHONE_ID, MySQLDSL.values(CAMPAIGN_PHONES.CLIENT_PHONE_ID))
                .executeIfRecordsAdded();
    }

    public void unlinkBannerPhonesByBannerId(int shard, Collection<Long> bannerIds) {
        unlinkBannerPhonesByBannerId(dslContextProvider.ppc(shard), bannerIds);
    }

    public void unlinkBannerPhonesByBannerId(DSLContext context, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }

        context
                .delete(BANNER_PHONES)
                .where(BANNER_PHONES.BID.in(bannerIds))
                .execute();

        context
                .update(BANNERS)
                .set(BANNERS.LAST_CHANGE, LocalDateTime.now())
                .where(BANNERS.BID.in(bannerIds))
                .execute();
    }

    public void unlinkCampaignPhonesByCampaignId(int shard, List<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return;
        }

        dslContextProvider.ppc(shard)
                .delete(CAMPAIGN_PHONES)
                .where(CAMPAIGN_PHONES.CID.in(campaignIds))
                .execute();
    }

    /**
     * Используется только для внутреннего отчета.
     */
    public void unlinkBannerPhonesByPhoneId(int shard, Collection<Long> phoneIds) {
        dslContextProvider.ppc(shard)
                .delete(BANNER_PHONES)
                .where(BANNER_PHONES.CLIENT_PHONE_ID.in(phoneIds))
                .execute();
    }

    /**
     * Используется только для внутреннего отчета.
     */
    public void unlinkCampaignPhonesByPhoneId(int shard, Collection<Long> phoneIds) {
        dslContextProvider.ppc(shard)
                .delete(CAMPAIGN_PHONES)
                .where(CAMPAIGN_PHONES.CLIENT_PHONE_ID.in(phoneIds))
                .execute();
    }

    private Condition getTelephonyPhoneStateCondition(@Nullable TelephonyPhoneState phoneState) {
        if (phoneState == null) {
            return DSL.trueCondition();
        }
        if (phoneState == TelephonyPhoneState.ATTACHED) {
            return CLIENT_PHONES.TELEPHONY_PHONE.isNotNull();
        }
        if (phoneState == TelephonyPhoneState.DETACHED) {
            return CLIENT_PHONES.TELEPHONY_PHONE.isNull();
        }
        throw new IllegalArgumentException("Unknown phoneState " + phoneState.name());
    }

    public Map<ClientId, List<ClientPhone>> getSiteTelephonyByClientId(int shard, boolean withDeleted) {
        var condition = CLIENT_PHONES.PHONE_TYPE.eq(ClientPhonesPhoneType.telephony)
                .and(CLIENT_PHONES.PERMALINK_ID.eq(0L));
        if (!withDeleted) {
            condition = condition.and(CLIENT_PHONES.IS_DELETED.isFalse());
        }
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_PHONES)
                .where(condition)
                .orderBy(CLIENT_PHONES.CLIENT_PHONE_ID)
                .fetchGroups(r -> ClientId.fromLong(r.getValue(CLIENT_PHONES.CLIENT_ID)), mapper::fromDb);
    }

    public void resetLastShowTime(int shard, List<Long> clientPhoneIds, LocalDateTime now) {
        dslContextProvider.ppc(shard)
                .update(CLIENT_PHONES)
                .set(CLIENT_PHONES.LAST_SHOW_TIME, now)
                .where(CLIENT_PHONES.CLIENT_PHONE_ID.in(clientPhoneIds))
                .execute();
    }
}
