package ru.yandex.direct.core.entity.clientphone.validation;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Pattern;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneMapping;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhoneType;
import ru.yandex.direct.core.entity.trackingphone.model.PhoneNumber;
import ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.defect.params.StringDefectParams;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ValidationUtils;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.clientphone.validation.ClientPhoneDefects.phoneCountryNotAllowed;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.ENTIRE_PHONE_WITH_EXTENSION_MAX_LENGTH;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isEqual;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.minStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@SuppressWarnings("rawtypes")
public class ClientPhoneValidationService {

    //Длина добавочного номера по аналогии с визитками, не более 6 символов
    private static final long EXTENSION_MAX_VALUE = 999999L;
    private static final int COMMENT_LENGTH = 255;
    private static final int ENTIRE_PHONE_WITH_EXTENSION_MAX_LENGTH_BEFORE_DB = 19;
    private static final int ENTIRE_PHONE_MAX_LENGTH = 14;
    private static final int ENTIRE_PHONE_MIN_LENGTH = 8;
    private static final Pattern E164_REGEXP = Pattern.compile("^\\+[1-9]\\d{10,14}$");
    // Правило не совсем честное, но простое: первая цифра после +7 не должна быть 6 или 7. Подробнее в DIRECT-120863
    private static final Pattern RUS_NUMBER_REGEXP = Pattern.compile("^\\+7[0-589]\\d{9}$");

    private static final Validator<PhoneNumber, Defect> PHONE_NUMBER_VALIDATOR = req -> {
        ModelItemValidationBuilder<PhoneNumber> vb = ModelItemValidationBuilder.of(req);
        vb.item(PhoneNumber.PHONE)
                .check(notNull())
                .check(notBlank())
                .check(minStringLength(ENTIRE_PHONE_MIN_LENGTH))
                .check(maxStringLength(ENTIRE_PHONE_MAX_LENGTH));
        vb.item(PhoneNumber.EXTENSION)
                .check(notLessThan(1L), When.notNull())
                .check(notGreaterThan(EXTENSION_MAX_VALUE), When.notNull());
        return vb.getResult();
    };

    private static ValidationResult<ClientPhone, Defect> validateManualPhone(ClientPhone phone) {
        ModelItemValidationBuilder<ClientPhone> vb = ModelItemValidationBuilder.of(phone);
        vb.item(ClientPhone.PHONE_TYPE)
                .check(fromPredicate(p -> p.equals(ClientPhoneType.MANUAL), invalidValue()));
        vb.item(ClientPhone.COMMENT)
                .check(maxStringLength(COMMENT_LENGTH), When.notNull());
        vb.item(ClientPhone.PHONE_NUMBER)
                .checkBy(PHONE_NUMBER_VALIDATOR)
                .check(maxEntirePhoneWithExtensionLength(), When.isValid());
        return vb.getResult();
    }

    private static ValidationResult<ClientPhone, Defect> validateUnique(
            ClientPhone phone,
            Map<Long, String> existingManualNumbers,
            Set<String> duplicatedNewNumbers
    ) {
        ModelItemValidationBuilder<ClientPhone> vb = ModelItemValidationBuilder.of(phone);
        vb.item(ClientPhone.PHONE_NUMBER)
                .check(validateUniquePhone(phone.getId(), existingManualNumbers, duplicatedNewNumbers));
        return vb.getResult();
    }

    public static final Validator<ClientPhone, Defect> CLIENT_PHONE_SPRAV_VALIDATOR = req -> {
        ModelItemValidationBuilder<ClientPhone> vb = ModelItemValidationBuilder.of(req);
        vb.item(ClientPhone.PHONE_NUMBER)
                .checkBy(PHONE_NUMBER_VALIDATOR)
                .check(maxEntirePhoneWithExtensionLength(), When.isValid());
        return vb.getResult();
    };

    private static Constraint<PhoneNumber, Defect> validateUniquePhone(
            Long phoneId,
            Map<Long, String> existingNumbers,
            Set<String> duplicatedNewNumbers
    ) {
        Predicate<PhoneNumber> isNumberUnique = number -> {
            var dbNumber = ClientPhoneMapping.phoneNumberToDb(number);
            // Не нужно сравнивать телефон с самим собой, поэтому убираем его из списка по id
            Set<String> numbers = getExistingNumbersWithoutItself(phoneId, existingNumbers);
            return !numbers.contains(dbNumber) && !duplicatedNewNumbers.contains(dbNumber);
        };
        return fromPredicate(isNumberUnique, duplicatedObject());
    }

    private static Set<String> getExistingNumbersWithoutItself(Long phoneId, Map<Long, String> existingNumbers) {
        Map<Long, String> numbers = new HashMap<>(existingNumbers);
        if (phoneId != null) {
            numbers.remove(phoneId);
        }
        return new HashSet<>(numbers.values());
    }

    /**
     * Валидация подменника, установленного вручную
     */
    public static ValidationResult<List<ClientPhone>, Defect> validate(
            List<ClientPhone> clientPhones,
            Map<Long, String> existingManualNumbers
    ) {
        ListValidationBuilder<ClientPhone, Defect> lvb = ListValidationBuilder.of(clientPhones);
        lvb.check(eachNotNull()).checkEachBy(ClientPhoneValidationService::validateManualPhone);
        if (lvb.getResult().hasAnyErrors()) {
            return lvb.getResult();
        }
        Set<String> duplicatedNewNumbers = getDuplicatedNumbers(clientPhones);
        lvb.checkEachBy(phone -> validateUnique(phone, existingManualNumbers, duplicatedNewNumbers));
        return lvb.getResult();
    }

    /**
     * Валидация телефонов для удаления
     *
     * @param clientPhoneIds       идентификаторы телефонов, которые нужно провалидировать
     * @param existingClientPhones существующие телефоны клиента
     * @return результат валидации
     */
    public static ValidationResult<List<Long>, Defect> validateForDelete(
            List<Long> clientPhoneIds,
            List<ClientPhone> existingClientPhones
    ) {
        var existingClientPhonesById = StreamEx.of(existingClientPhones).toMap(ClientPhone::getId, Function.identity());
        ListValidationBuilder<Long, Defect> lvb = ListValidationBuilder.of(clientPhoneIds);
        lvb
                .checkEach(validId())
                .checkEach(inSet(existingClientPhonesById.keySet()), objectNotFound(), When.isValid())
                .checkEach(isPhoneManual(existingClientPhonesById), When.isValid());
        return lvb.getResult();
    }

    /**
     * Валидация перед резервированием номера в Телефонии.
     * Проверяем корректность и наличие необходимых полей
     */
    public static ValidationResult<List<ClientPhone>, Defect> preValidateTelephonyPhones(
            List<ClientPhone> clientPhones
    ) {
        ListValidationBuilder<ClientPhone, Defect> lvb = ListValidationBuilder.of(clientPhones);
        lvb.checkEachBy(ClientPhoneValidationService::preValidateTelephonyPhone);
        return lvb.getResult();
    }

    private static Set<String> getDuplicatedNumbers(List<ClientPhone> clientPhones) {
        return StreamEx.of(clientPhones)
                .map(ClientPhone::getPhoneNumber)
                .map(ClientPhoneMapping::phoneNumberToDb)
                .remove(Objects::isNull)
                .distinct(2)
                .toSet();
    }

    private static ValidationResult<ClientPhone, Defect> preValidateTelephonyPhone(ClientPhone phone) {
        ModelItemValidationBuilder<ClientPhone> vb = ModelItemValidationBuilder.of(phone);
        vb.item(ClientPhone.PHONE_NUMBER)
                .check(notNull())
                .checkBy(redirectPhoneValidator());
        vb.item(ClientPhone.COUNTER_ID)
                .check(notNull());
        vb.item(ClientPhone.PERMALINK_ID)
                .check(notNull());
        vb.item(ClientPhone.PHONE_TYPE)
                .check(isEqual(ClientPhoneType.TELEPHONY, CommonDefects.inconsistentState()));
        return vb.getResult();
    }

    /**
     * Проверяем, что системные поля заполнены после резервирования номера в Телефонии
     */
    public static ValidationResult<List<ClientPhone>, Defect> validateTelephonyPhones(List<ClientPhone> clientPhones) {
        ListValidationBuilder<ClientPhone, Defect> lvb = ListValidationBuilder.of(clientPhones);
        lvb.checkEachBy(ClientPhoneValidationService::validateTelephonyPhone);
        return lvb.getResult();
    }

    private static ValidationResult<ClientPhone, Defect> validateTelephonyPhone(ClientPhone clientPhone) {
        ModelItemValidationBuilder<ClientPhone> vb = ModelItemValidationBuilder.of(clientPhone);
        vb.item(ClientPhone.TELEPHONY_SERVICE_ID)
                .check(notNull());
        vb.item(ClientPhone.TELEPHONY_PHONE)
                .check(notNull());
        return vb.getResult();
    }

    /**
     * Валидация, вызываемая при обновлении параметров номера Телефонии.
     * Менять можно только номер-редирект. Проверяем его и на всякий случай тип
     */
    public static ValidationResult<List<ClientPhone>, Defect> validateTelephony(
            ValidationResult<List<ClientPhone>, Defect> validationResult) {
        ListValidationBuilder<ClientPhone, Defect> lvb = new ListValidationBuilder<>(validationResult);
        lvb.checkEachBy(ClientPhoneValidationService::validateTelephony);
        return lvb.getResult();
    }

    private static ValidationResult<ClientPhone, Defect> validateTelephony(ClientPhone phone) {
        ModelItemValidationBuilder<ClientPhone> vb = ModelItemValidationBuilder.of(phone);
        vb.item(ClientPhone.PHONE_TYPE)
                .check(Constraint.fromPredicate(p -> p.equals(ClientPhoneType.TELEPHONY), invalidValue()));
        vb.item(ClientPhone.PHONE_NUMBER)
                .check(notNull())
                .checkBy(redirectPhoneValidator());
        return vb.getResult();
    }

    private static Constraint<PhoneNumber, Defect> maxEntirePhoneWithExtensionLength() {
        Predicate<PhoneNumber> predicate = phone -> {
            // поле `phone` varchar(25),
            // проверим что длина строки (с разделителями) не превышает 25 (как в vcards.phone)
            // и что после преобразования для записи в БД она не превысит это значение
            // Этот Constraint используется только для телефонов типа manual и org, поэтому phone != null
            //noinspection ConstantConditions
            int dbPhoneLength = ClientPhoneMapping.phoneNumberToDb(phone).length();

            return dbPhoneLength <= ENTIRE_PHONE_WITH_EXTENSION_MAX_LENGTH;
        };
        return fromPredicate(predicate, tooLongEntirePhoneWithExtension());
    }

    public static Defect<StringDefectParams> tooLongEntirePhoneWithExtension() {
        return new Defect<>(PhoneValidator.StringLengthDefectIds.ENTIRE_PHONE_WITH_EXTENSION_IS_TOO_LONG,
                new StringDefectParams().withMaxLength(ENTIRE_PHONE_WITH_EXTENSION_MAX_LENGTH_BEFORE_DB));
    }

    public static Constraint<Long, Defect> isPhoneManual(Map<Long, ClientPhone> existedPhonesById) {
        return Constraint.fromPredicate(
                phoneId -> {
                    ClientPhone phone = existedPhonesById.get(phoneId);
                    return phone.getPhoneType() == ClientPhoneType.MANUAL;
                },
                CommonDefects.inconsistentState()
        );
    }

    public static Constraint<String, Defect> rusNumberConstraint(Defect<Void> defect) {
        return Constraint.fromPredicate(p -> RUS_NUMBER_REGEXP.matcher(p).matches(), defect);
    }

    private static Validator<PhoneNumber, Defect> redirectPhoneValidator() {
        return phoneNumber -> {
            var vb = ModelItemValidationBuilder.of(phoneNumber);
            if (phoneNumber == null) {
                return vb.getResult();
            }
            vb.item(PhoneNumber.PHONE)
                    .check(notNull())
                    .check(matchPhonePattern(E164_REGEXP, invalidValue()))
                    .check(matchPhonePattern(RUS_NUMBER_REGEXP, phoneCountryNotAllowed()), When.isValid());

            // Для обратной совместимости переносим дефекты на верхний уровень
            ModelItemValidationBuilder<PhoneNumber> vb2 = ModelItemValidationBuilder.of(phoneNumber);
            ValidationUtils.transferIssuesFromValidationToTopLevel(vb.getResult(), vb2.getResult());
            return vb2.getResult();
        };
    }

    private static Constraint<String, Defect> matchPhonePattern(Pattern pattern, Defect<Void> defect) {
        return Constraint.fromPredicate(p -> pattern.matcher(p).matches(), defect);
    }
}
